<?php
/**
 * Search request handler
 *
 * Processes search requests (GET/POST), executes searches, applies filters,
 * and returns structured data for the template to render.
 */

if (!defined('ABSPATH')) {
    exit;
}

// Only require search-functions if not already loaded (prevents circular dependency)
if (!function_exists('psfa_amazon_search_products')) {
    require_once PSFA_PLUGIN_DIR . 'core/search/functions.php';
}
// Only require search-view-helpers if not already loaded
if (!function_exists('psfa_is_premium')) {
    require_once PSFA_PLUGIN_DIR . 'core/search/view-helpers.php';
}

/**
 * Handle POST form submission and redirect if needed
 *
 * @return void|array Returns void if redirecting, or array with search params if not redirecting
 */
function psfa_handle_post_request() {
    // Check for both search form submission and filter form submission
    if (!isset($_SERVER['REQUEST_METHOD']) || $_SERVER['REQUEST_METHOD'] !== 'POST') {
        return null;
    }
    
    // Check if this is our form - must have at least one of our fields
    $is_our_form = isset($_POST['amazon_search_query']) || isset($_POST['amazon_search_brand']) || 
        isset($_POST['amazon_search_price_min']) || isset($_POST['amazon_search_price_max']) ||
        isset($_POST['amazon_search_condition']) || isset($_POST['amazon_search_sort']) ||
        isset($_POST['amazon_search_availability']) || isset($_POST['psfa_nonce']);
    
    if (!$is_our_form) {
        return null;
    }
    
    // Verify nonce for form submission
    $psfa_raw_nonce = isset($_POST['psfa_nonce']) ? sanitize_text_field(wp_unslash($_POST['psfa_nonce'])) : '';
    if (empty($psfa_raw_nonce) || !wp_verify_nonce($psfa_raw_nonce, 'psfa_search_nonce')) {
        wp_die('Security check failed');
    }
    
    $psfa_q = isset($_POST['amazon_search_query']) ? sanitize_text_field(wp_unslash($_POST['amazon_search_query'])) : '';
    $psfa_cat = isset($_POST['amazon_search_category']) ? sanitize_text_field(wp_unslash($_POST['amazon_search_category'])) : '';
    
    // Handle price values - empty strings should be treated as 0
    $psfa_min = 0;
    if (isset($_POST['amazon_search_price_min']) && $_POST['amazon_search_price_min'] !== '') {
        $psfa_min = floatval($_POST['amazon_search_price_min']);
    }
    $psfa_max = 0;
    if (isset($_POST['amazon_search_price_max']) && $_POST['amazon_search_price_max'] !== '') {
        $psfa_max = floatval($_POST['amazon_search_price_max']);
    }
    $psfa_filter_brand = isset($_POST['amazon_search_brand']) ? sanitize_text_field(wp_unslash($_POST['amazon_search_brand'])) : '';
    $psfa_sort = isset($_POST['amazon_search_sort']) ? sanitize_text_field(wp_unslash($_POST['amazon_search_sort'])) : '';
    $psfa_condition = isset($_POST['amazon_search_condition']) ? sanitize_text_field(wp_unslash($_POST['amazon_search_condition'])) : '';
    $psfa_merchant = isset($_POST['amazon_search_merchant']) ? sanitize_text_field(wp_unslash($_POST['amazon_search_merchant'])) : '';
    $psfa_availability = isset($_POST['amazon_search_availability']) ? sanitize_text_field(wp_unslash($_POST['amazon_search_availability'])) : '';
    
    // Premium filters (field names match URL params directly)
    $psfa_has_deal = isset($_POST['has_deal']) ? sanitize_text_field(wp_unslash($_POST['has_deal'])) : '';
    $psfa_min_rating = isset($_POST['min_rating']) && $_POST['min_rating'] !== '' ? absint($_POST['min_rating']) : 0;
    $psfa_min_saving = isset($_POST['min_saving_percent']) && $_POST['min_saving_percent'] !== '' ? absint($_POST['min_saving_percent']) : 0;
    
    // If this is a search form submission (not just filter), redirect to URL with query parameter
    // OR if filters are applied, redirect to include filter parameters in URL
    $psfa_has_filters = !empty($psfa_filter_brand) || $psfa_min > 0 || $psfa_max > 0 || 
                        !empty($psfa_sort) || !empty($psfa_condition) || !empty($psfa_merchant) ||
                        (!empty($psfa_availability) && $psfa_availability !== 'Available') ||
                        !empty($psfa_has_deal) || $psfa_min_rating > 0 || $psfa_min_saving > 0;
    $psfa_is_search_only = !empty($psfa_q) && !$psfa_has_filters;
    
    // ALWAYS redirect when form is submitted (has query OR category) to ensure clean URLs
    $psfa_should_redirect = !empty($psfa_q) || !empty($psfa_cat) || $psfa_has_filters;
    
    if ($psfa_should_redirect) {
        // Build redirect URL
        $psfa_redirect_url = get_permalink();
        $psfa_params = [];
        
        if (!empty($psfa_q)) {
            $psfa_params['q'] = $psfa_q;
        }
        if (!empty($psfa_cat)) {
            $psfa_params['category'] = $psfa_cat;
        }
        if ($psfa_min > 0) {
            $psfa_params['min_price'] = $psfa_min;
        }
        if ($psfa_max > 0) {
            $psfa_params['max_price'] = $psfa_max;
        }
        if (!empty($psfa_filter_brand)) {
            $psfa_params['brand'] = $psfa_filter_brand;
        }
        if (!empty($psfa_sort)) {
            $psfa_params['sort'] = $psfa_sort;
        }
        if (!empty($psfa_condition)) {
            $psfa_params['condition'] = $psfa_condition;
        }
        if (!empty($psfa_merchant)) {
            $psfa_params['merchant'] = $psfa_merchant;
        }
        // Add availability to URL (skip default 'Available' to keep URLs clean)
        if (!empty($psfa_availability) && $psfa_availability !== 'Available') {
            $psfa_params['availability'] = $psfa_availability;
        }
        // Premium filters
        if (!empty($psfa_has_deal)) {
            $psfa_params['has_deal'] = $psfa_has_deal;
        }
        if ($psfa_min_rating > 0) {
            $psfa_params['min_rating'] = $psfa_min_rating;
        }
        if ($psfa_min_saving > 0) {
            $psfa_params['min_saving_percent'] = $psfa_min_saving;
        }
        if (!empty($psfa_params)) {
            $psfa_join_char = (false === strpos($psfa_redirect_url, '?')) ? '?' : '&';
            $psfa_redirect_url .= $psfa_join_char . http_build_query($psfa_params);
        }
        
        // Fallback to JS redirect if headers already sent (shouldn't happen with early hook)
        if (headers_sent()) {
            echo '<script>window.location.href = "' . esc_js($psfa_redirect_url) . '";</script>';
            exit;
        }
        
        wp_safe_redirect($psfa_redirect_url);
        exit;
    }
    
    return array(
        'query' => $psfa_q,
        'category' => $psfa_cat,
        'price_min' => $psfa_min,
        'price_max' => $psfa_max,
        'brand' => $psfa_filter_brand,
        'sort' => $psfa_sort,
        'condition' => $psfa_condition,
        'merchant' => $psfa_merchant,
        'availability' => $psfa_availability,
        'has_deal' => $psfa_has_deal,
        'min_rating' => $psfa_min_rating,
        'min_saving_percent' => $psfa_min_saving,
    );
}


// Helper functions moved to search-helpers.php to avoid circular dependencies

/**
 * Process query-based search request.
 *
 * Handles search by keyword/query, fetches initial results, applies filters,
 * and merges page 2 for better filter options and initial display.
 *
 * @param string $query Search query/keyword.
 * @param string $category Category filter (optional).
 * @param string $sort Sort option (e.g., 'Price:LowToHigh').
 * @param string $brand Brand filter (optional).
 * @param float $min_price Minimum price filter (0 to disable).
 * @param float $max_price Maximum price filter (0 to disable).
 * @param string $condition Condition filter (New/Used/Refurbished/Collectible).
 * @param string $merchant Merchant filter (Amazon/All).
 * @return array {
 *   @type string error_message
 *   @type array|null results Raw API results
 *   @type array|null filtered_results Results after client-side filtering
 *   @type array|null filter_options Available filter options (brands, price ranges, conditions)
 *   @type int total_unfiltered_count Total items before price filtering
 * }
 */
function psfa_process_query_search($query, $category, $sort, $brand, $min_price, $max_price, $condition, $merchant, $min_saving_percent = 0, $has_deal = '', $availability = '') {
    $result = array(
        'error_message' => '',
        'results' => null,
        'filtered_results' => null,
        'filter_options' => null,
        'total_unfiltered_count' => 0,
    );
    
    $effective_sort = psfa_get_effective_sort($sort, $min_price, $max_price);
    
    // Initial search without filters to get filter options
    $unfiltered_result = psfa_fetch_and_merge_pages($query, $category, $sort, '', '', '', $has_deal, 0, $min_saving_percent, $availability, $min_price, $max_price);
    
    if (!empty($unfiltered_result['error_message'])) {
        $result['error_message'] = $unfiltered_result['error_message'];
        return $result;
    }
    
    $all_items = $unfiltered_result['merged_items'];
    $filter_options = psfa_build_filter_options($all_items, $unfiltered_result['search_refinements']);
    
    // If URL has filters (Brand, Condition, Merchant), re-fetch with filters applied server-side
    if (!empty($brand) || !empty($condition) || !empty($merchant)) {
        $filtered_result = psfa_fetch_and_merge_pages($query, $category, $effective_sort, $brand, $condition, $merchant, $has_deal, 0, $min_saving_percent, $availability, $min_price, $max_price);
        
        if (!empty($filtered_result['error_message'])) {
            $result['error_message'] = $filtered_result['error_message'];
            return $result;
        }
        
        $all_items = $filtered_result['merged_items'];
        $result['results'] = $filtered_result['page1_results'];
    } else {
        $result['results'] = $unfiltered_result['page1_results'];
    }
    
    // Store total unfiltered count BEFORE price filtering
    $result['total_unfiltered_count'] = count($all_items);
    
    // Apply client-side processing (price filtering + condition + merchant + availability + prime + savings filtering)
    $all_items = psfa_apply_client_side_processing($all_items, $min_price, $max_price, $sort, $brand, $min_saving_percent, $has_deal, $condition, $merchant, $availability);
    
    // Update results
    $result['filtered_results'] = $result['results'];
    $result['filtered_results']['SearchResult']['Items'] = $all_items;
    $result['filter_options'] = $filter_options;
    
    return $result;
}

/**
 * Process category-based search
 *
 * @param string $category
 * @param string $sort
 * @param string $brand
 * @param float $min_price
 * @param float $max_price
 * @param string $condition
 * @param string $merchant
 * @return array
 */
function psfa_process_category_search($category, $sort, $brand, $min_price, $max_price, $condition, $merchant, $min_saving_percent = 0, $has_deal = '', $availability = '') {
    $result = array(
        'error_message' => '',
        'results' => null,
        'filtered_results' => null,
        'filter_options' => null,
        'total_unfiltered_count' => 0,
    );
    
    $effective_sort = psfa_get_effective_sort($sort, $min_price, $max_price);
    $filter_options = array('brands' => array(), 'min_price' => 0, 'max_price' => 0, 'conditions' => array());
    $all_items = array();
    
    // If URL has filters (Brand, Condition, Merchant), re-fetch with filters applied server-side
    if (!empty($brand) || !empty($condition) || !empty($merchant)) {
        $filtered_result = psfa_fetch_and_merge_pages('', $category, $effective_sort, $brand, $condition, $merchant, $has_deal, 0, $min_saving_percent, $availability, $min_price, $max_price);
        
        if (!empty($filtered_result['error_message'])) {
            $result['error_message'] = $filtered_result['error_message'];
        } elseif (!empty($filtered_result['merged_items'])) {
            $all_items = $filtered_result['merged_items'];
            $result['results'] = $filtered_result['page1_results'];
            $filter_options = psfa_build_filter_options($all_items, $filtered_result['search_refinements']);
            $result['total_unfiltered_count'] = count($all_items);
        }
        
        // If results are empty but we have condition/merchant filters, fetch filter options from unfiltered category
        if (empty($all_items) && (!empty($condition) || !empty($merchant))) {
            $unfiltered_result = psfa_fetch_and_merge_pages('', $category, '', '', '', '', '', 0, 0, $availability, 0, 0);
            if (!empty($unfiltered_result['merged_items'])) {
                $filter_options = psfa_build_filter_options($unfiltered_result['merged_items'], $unfiltered_result['search_refinements']);
                $result['total_unfiltered_count'] = count($unfiltered_result['merged_items']);
            }
        }
    } else {
        // Load top products for category (no brand/condition/merchant filters)
        $category_result = psfa_fetch_and_merge_pages('', $category, $effective_sort, '', '', '', $has_deal, 0, $min_saving_percent, $availability, $min_price, $max_price);
        
        if (!empty($category_result['error_message'])) {
            $result['error_message'] = $category_result['error_message'];
        } elseif (!empty($category_result['merged_items'])) {
            $all_items = $category_result['merged_items'];
            $result['results'] = $category_result['page1_results'];
            $filter_options = psfa_build_filter_options($all_items, $category_result['search_refinements']);
        }
    }
    
    // Always build the brand dropdown from unfiltered category results (no Brand filter)
    // so users can switch brands without resetting filters. We still respect Condition/Merchant
    if (!empty($category)) {
        $brand_options_list = psfa_build_brand_options_from_category($category, $effective_sort, $condition, $merchant);
        if (!empty($brand_options_list)) {
            $existing_brands = $filter_options['brands'] ?? array();
            $filter_options['brands'] = array_unique(array_merge($existing_brands, $brand_options_list));
            sort($filter_options['brands']);
        }
    }
    
    // Store total unfiltered count BEFORE price filtering
    if (!isset($result['total_unfiltered_count'])) {
        $result['total_unfiltered_count'] = count($all_items);
    }
    
    // Apply client-side processing (price filtering + condition + merchant + availability + prime + savings filtering)
    $all_items = psfa_apply_client_side_processing($all_items, $min_price, $max_price, $sort, $brand, $min_saving_percent, $has_deal, $condition, $merchant, $availability);
    
    // Update results
    if (isset($result['results'])) {
        $result['filtered_results'] = $result['results'];
        $result['filtered_results']['SearchResult']['Items'] = $all_items;
    }
    $result['filter_options'] = $filter_options;
    
    return $result;
}

/**
 * Process POST request search (non-redirect case)
 *
 * @param array $params
 * @param string $initial_category
 * @return array
 */
function psfa_process_post_search($params, $initial_category) {
    $result = array(
        'error_message' => '',
        'results' => null,
        'filtered_results' => null,
        'filter_options' => null,
        'total_unfiltered_count' => 0,
    );
    
    $query = $params['query'];
    $category = !empty($params['category']) ? $params['category'] : $initial_category;
    $min_price = $params['price_min'];
    $max_price = $params['price_max'];
    $brand = $params['brand'];
    $sort = $params['sort'];
    $condition = $params['condition'];
    $merchant = $params['merchant'];
    $min_saving_percent = $params['min_saving_percent'] ?? 0;
    $has_deal = $params['has_deal'] ?? '';
    $availability = !empty($params['availability']) ? $params['availability'] : 'Available';
    
    if (empty($query) && empty($category)) {
        $result['error_message'] = 'Please enter a search term or select a category.';
        return $result;
    }
    
    $effective_sort = psfa_get_effective_sort($sort, $min_price, $max_price);
    
    // Fetch and merge pages
    $search_result = psfa_fetch_and_merge_pages($query, $category, $effective_sort, $brand, $condition, $merchant, $has_deal, 0, $min_saving_percent, $availability, $min_price, $max_price);
    
    if (!empty($search_result['error_message'])) {
        $result['error_message'] = $search_result['error_message'];
        return $result;
    }
    
    $all_items = $search_result['merged_items'];
    
    // Handle empty results
    if (empty($all_items)) {
        $result['filtered_results'] = $search_result['page1_results'];
        $result['filtered_results']['SearchResult']['Items'] = array();
        $result['filter_options'] = array('brands' => array(), 'min_price' => 0, 'max_price' => 0, 'conditions' => array());
        return $result;
    }
    
    // Build filter options
    $filter_options = psfa_build_filter_options($all_items, $search_result['search_refinements']);
    
    // Store total unfiltered count BEFORE price filtering
    $result['total_unfiltered_count'] = count($all_items);
    
    // Apply client-side processing (price filtering + condition + merchant + availability + prime + savings filtering)
    $all_items = psfa_apply_client_side_processing($all_items, $min_price, $max_price, $sort, $brand, $min_saving_percent, $has_deal, $condition, $merchant, $availability);
    
    // Update results
    $result['results'] = $search_result['page1_results'];
    $result['filtered_results'] = $search_result['page1_results'];
    $result['filtered_results']['SearchResult']['Items'] = $all_items;
    $result['filter_options'] = $filter_options;
    
    return $result;
}

/**
 * Main search request handler - processes all search logic and returns structured data.
 *
 * This is the primary entry point for search functionality. It:
 * - Handles POST form submissions (with redirects for clean URLs)
 * - Processes GET-based searches (from URL parameters)
 * - Executes Amazon PA-API searches
 * - Applies filters (brand, condition, merchant server-side; price client-side)
 * - Merges page 2 results for better filter options
 * - Returns structured data for the template to render
 *
 * @return array {
 *   @type string error_message Error message if any
 *   @type array|null results Raw API results
 *   @type array|null filtered_results Results after client-side filtering
 *   @type array|null filter_options Available filter options (brands, price ranges, conditions)
 *   @type array|null search_params Search parameters for pagination data attributes
 *   @type int total_unfiltered_count Total items before price filtering
 *   @type int pages_fetched Number of pages fetched (1 or 2)
 *   @type string initial_category Category from shortcode or URL
 * }
 */
function psfa_handle_search_request() {
    // Initialize return structure
    $result = array(
        'error_message' => '',
        'results' => null,
        'filtered_results' => null,
        'filter_options' => null,
        'search_params' => null,
        'total_unfiltered_count' => 0,
        'pages_fetched' => 1,
        'initial_category' => '',
    );
    
    // Handle POST requests first (may redirect)
    $post_params = psfa_handle_post_request();
    if ($post_params === null && isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST') {
        // Redirect happened, function will exit
        return $result;
    }
    
    // Get category from shortcode attributes (passed via global or GET)
    global $psfa_shortcode_category;
    $psfa_shortcode_category = isset($psfa_shortcode_category) ? $psfa_shortcode_category : '';
    
    // Normalize category
    $psfa_raw_category = '';
    if (!empty($psfa_shortcode_category)) {
        $psfa_raw_category = strtolower(trim($psfa_shortcode_category));
    // GET parameters are public URL parameters and don't require nonce verification (WordPress standard practice)
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Public URL parameter, no form submission
    } elseif (isset($_GET['category'])) {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Public URL parameter, wp_unslash and sanitize_text_field handle sanitization
        $psfa_raw_category = strtolower(trim(sanitize_text_field(wp_unslash($_GET['category']))));
    }
    
    $psfa_initial_category = '';
    if (!empty($psfa_raw_category)) {
        $psfa_initial_category = psfa_normalize_category($psfa_raw_category);
        
        // Check if category is available in free version
        if (!psfa_is_category_available($psfa_initial_category)) {
            $psfa_upgrade_url = function_exists('psfa_fs') ? psfa_fs()->get_upgrade_url() : '#';
            $result['error_message'] = psfa_render_category_upgrade_notice($psfa_upgrade_url);
            $psfa_initial_category = ''; // Reset to prevent search
        }
    }
    
    $result['initial_category'] = $psfa_initial_category;
    
    // Get shortcode attributes for defaults
    global $psfa_shortcode_atts;
    $psfa_shortcode_atts = isset($psfa_shortcode_atts) ? $psfa_shortcode_atts : array();
    
    // Get normalized search parameters
    $psfa_request_params = psfa_get_search_params();
    $psfa_url_query = $psfa_request_params['query'];
    $psfa_url_sort = $psfa_request_params['sort'];
    $psfa_url_brand = $psfa_request_params['brand'];
    $psfa_url_min_price = $psfa_request_params['min_price'];
    $psfa_url_max_price = $psfa_request_params['max_price'];
    $psfa_url_condition = $psfa_request_params['condition'];
    $psfa_url_merchant = $psfa_request_params['merchant'];
    $psfa_url_min_saving_percent = $psfa_request_params['min_saving_percent'] ?? 0;
    $psfa_url_has_deal = $psfa_request_params['has_deal'] ?? '';
    $psfa_url_availability = !empty($psfa_request_params['availability']) ? $psfa_request_params['availability'] : 'Available';
    
    // Use shortcode attributes as defaults if URL params are empty
    if (empty($psfa_url_query) && !empty($psfa_shortcode_atts['query'])) {
        $psfa_url_query = sanitize_text_field($psfa_shortcode_atts['query']);
    }
    if (empty($psfa_url_brand) && !empty($psfa_shortcode_atts['brand'])) {
        $psfa_url_brand = sanitize_text_field($psfa_shortcode_atts['brand']);
    }
    if (empty($psfa_url_sort) && !empty($psfa_shortcode_atts['sort'])) {
        $psfa_url_sort = sanitize_text_field($psfa_shortcode_atts['sort']);
    }
    
    // If POST params exist (non-redirect case), use them
    if ($post_params !== null) {
        $search_result = psfa_process_post_search($post_params, $psfa_initial_category);
        $result = array_merge($result, $search_result);
        $result['search_params'] = $post_params;
        return $result;
    }
    
    // Execute search based on request type
    // If query is provided, search by query
    if ((!isset($_SERVER['REQUEST_METHOD']) || $_SERVER['REQUEST_METHOD'] !== 'POST') && !empty($psfa_url_query)) {
        $search_result = psfa_process_query_search(
            $psfa_url_query,
            $psfa_initial_category,
            $psfa_url_sort,
            $psfa_url_brand,
            $psfa_url_min_price,
            $psfa_url_max_price,
            $psfa_url_condition,
            $psfa_url_merchant,
            $psfa_url_min_saving_percent,
            $psfa_url_has_deal,
            $psfa_url_availability
        );
        $result = array_merge($result, $search_result);
        $result['search_params'] = array(
            'query' => $psfa_url_query,
            'category' => $psfa_initial_category,
            'price_min' => $psfa_url_min_price,
            'price_max' => $psfa_url_max_price,
            'brand' => $psfa_url_brand,
            'sort' => $psfa_url_sort,
            'condition' => $psfa_url_condition,
            'merchant' => $psfa_url_merchant,
            'availability' => $psfa_url_availability,
        );
    }
    // If category is provided (and no query), show top products for category
    elseif ((!isset($_SERVER['REQUEST_METHOD']) || $_SERVER['REQUEST_METHOD'] !== 'POST') && !empty($psfa_initial_category) && empty($psfa_url_query)) {
        $search_result = psfa_process_category_search(
            $psfa_initial_category,
            $psfa_url_sort,
            $psfa_url_brand,
            $psfa_url_min_price,
            $psfa_url_max_price,
            $psfa_url_condition,
            $psfa_url_merchant,
            $psfa_url_min_saving_percent,
            $psfa_url_has_deal,
            $psfa_url_availability
        );
        $result = array_merge($result, $search_result);
        $result['search_params'] = array(
            'query' => '',
            'category' => $psfa_initial_category,
            'price_min' => $psfa_url_min_price,
            'price_max' => $psfa_url_max_price,
            'brand' => $psfa_url_brand,
            'sort' => $psfa_url_sort,
            'condition' => $psfa_url_condition,
            'merchant' => $psfa_url_merchant,
            'availability' => $psfa_url_availability,
        );
    }
    
    // Ensure search params are always set
    if ($result['search_params'] === null) {
        $result['search_params'] = array(
            'query' => $psfa_url_query,
            'category' => $psfa_initial_category,
            'price_min' => 0,
            'price_max' => 0,
            'brand' => '',
            'sort' => $psfa_url_sort,
            'condition' => '',
            'merchant' => '',
            'availability' => $psfa_url_availability,
        );
    }
    
    return $result;
}
