<?php
/**
 * AWS Signature V4 API Client
 *
 * Implements PA-API authentication using AWS Signature V4.
 * Wraps existing http.php functionality with clean interface.
 */

if (!defined('ABSPATH')) {
    exit;
}

require_once __DIR__ . '/interface-api-client.php';
require_once __DIR__ . '/http.php';
require_once __DIR__ . '/config.php';

class PSFA_AWSSig4_Client implements PSFA_API_Client_Interface {

    /**
     * API configuration
     *
     * @var array
     */
    private $config;

    /**
     * Constructor
     */
    public function __construct() {
        $this->config = psfa_get_api_config();
    }

    /**
     * Execute a SearchItems request
     *
     * @param array $payload SearchItems payload
     * @return array API response or error array
     */
    public function search_items(array $payload): array {
        if (isset($this->config['error'])) {
            return ['error' => $this->config['error']];
        }

        // Ensure required fields
        if (!isset($payload['PartnerTag'])) {
            $payload['PartnerTag'] = $this->config['associate_tag'];
        }
        if (!isset($payload['PartnerType'])) {
            $payload['PartnerType'] = 'Associates';
        }

        return psfa_execute_paapi_request($this->config, $payload);
    }

    /**
     * Execute a GetItems request
     *
     * @param array $asins List of ASINs
     * @param array $resources Resources to request
     * @return array API response or error array
     */
    public function get_items(array $asins, array $resources): array {
        if (isset($this->config['error'])) {
            return ['error' => $this->config['error']];
        }

        // Build GetItems payload
        $payload = [
            'ItemIds' => $asins,
            'PartnerTag' => $this->config['associate_tag'],
            'PartnerType' => 'Associates',
            'Resources' => $resources,
        ];

        // Modify config for GetItems operation
        $get_config = $this->config;
        $get_config['uri'] = '/paapi5/getitems';
        $get_config['amz_target'] = 'com.amazon.paapi5.v1.ProductAdvertisingAPIv1.GetItems';

        return psfa_execute_paapi_request($get_config, $payload);
    }

    /**
     * Check if client is properly configured
     *
     * @return bool|WP_Error
     */
    public function is_configured() {
        if (isset($this->config['error'])) {
            return new WP_Error('not_configured', $this->config['error']);
        }
        return true;
    }

    /**
     * Get client type identifier
     *
     * @return string
     */
    public function get_type(): string {
        return 'awssig4';
    }

    /**
     * Get associate tag
     *
     * @return string
     */
    public function get_associate_tag(): string {
        return $this->config['associate_tag'] ?? '';
    }

    /**
     * Reload configuration (useful after settings change)
     */
    public function reload_config(): void {
        $this->config = psfa_get_api_config();
    }
}
