<?php
/**
 * Premium Features Registry for Paapi Product Search for Amazon
 *
 * Defines all premium features and provides helper functions for checking
 * availability and rendering UI components.
 */

if (!defined('ABSPATH')) {
    exit;
}

// Include view helpers for psfa_is_premium()
require_once PSFA_PLUGIN_DIR . 'core/search/view-helpers.php';

/**
 * Get formatted categories description for premium features
 *
 * @return string HTML formatted description with all categories listed
 */
function psfa_get_categories_description() {
    $all_categories = array(
        'Electronics',
        'Books',
        'Fashion',
        'ToysAndGames',
        'HomeGarden',
        'TodaysDeals',
        'Automotive',
        'Baby',
        'Beauty',
        'Computers',
        'Grocery',
        'HealthPersonalCare',
        'HomeImprovement',
        'Industrial',
        'KindleStore',
        'Luggage',
        'MoviesTV',
        'Music',
        'MusicalInstruments',
        'OfficeProducts',
        'PetSupplies',
        'Software',
        'SportsOutdoors',
        'ToolsHomeImprovement',
        'VideoGames',
        'Watches'
    );
    
    $free_categories = array(
        'Electronics',
        'Books',
        'Fashion',
        'ToysAndGames',
        'HomeGarden'
    );
    
    $description = 'Access to all ' . count($all_categories) . ' Amazon categories:<br><br>';
    $description .= '<ul class="psfa-category-list">';
    
    foreach ($all_categories as $cat) {
        $is_free = in_array($cat, $free_categories);
        $cat_label = esc_html($cat);
        if ($is_free) {
            $cat_label .= ' <span class="psfa-category-free-badge">(free)</span>';
        }
        $description .= '<li class="psfa-category-item">';
        $description .= '<span class="psfa-category-checkmark">✓</span> ';
        $description .= $cat_label;
        $description .= '</li>';
    }
    
    $description .= '</ul>';
    
    return $description;
}

/**
 * Get all premium features
 *
 * @return array Array of premium feature definitions
 */
function psfa_get_all_premium_features() {
    $upgrade_url = 'https://paapiplugin.com/pricing/#plans';
    
    return array(
        'themes' => array(
            'key' => 'themes',
            'name' => 'Premium Themes',
            'description' => 'Choose from multiple beautiful themes (Ocean Blue, Modern, Minimal, Dark). Free users get the Amazon-Inspired theme.',
            'plan' => 'pro',
            'enabled' => psfa_is_premium(),
            'upgrade_url' => $upgrade_url,
            'icon' => 'dashicons-art',
        ),
        'unlimited_pagination' => array(
            'key' => 'unlimited_pagination',
            'name' => 'Unlimited Pagination',
            'description' => 'Load more products up to amazons API’s limit (Free: 5 pages limit)',
            'plan' => 'pro',
            'enabled' => psfa_is_premium(),
            'upgrade_url' => $upgrade_url,
            'icon' => 'dashicons-arrow-right-alt',
        ),
        'all_categories' => array(
            'key' => 'all_categories',
            'name' => 'All Categories',
            'short_description' => 'Access to all 26 Amazon categories. Free users get 5 categories (Electronics, Books, Fashion, Toys, Home & Garden).',
            'description' => psfa_get_categories_description(),
            'plan' => 'pro',
            'enabled' => psfa_is_premium(),
            'upgrade_url' => $upgrade_url,
            'icon' => 'dashicons-category',
        ),
        'list_display' => array(
            'key' => 'list_display',
            'name' => 'List Display Mode',
            'description' => 'Display products in a vertical list layout instead of grid. Perfect for detailed product comparisons.',
            'plan' => 'pro',
            'enabled' => psfa_is_premium(),
            'upgrade_url' => $upgrade_url,
            'icon' => 'dashicons-list-view',
        ),
        'custom_colors' => array(
            'key' => 'custom_colors',
            'name' => 'Custom Color Schemes',
            'description' => 'Fully customize colors for each theme - buttons, text, borders, and backgrounds to match your brand.',
            'plan' => 'pro',
            'enabled' => psfa_is_premium(),
            'upgrade_url' => $upgrade_url,
            'icon' => 'dashicons-admin-customizer',
        ),
        'button_customization' => array(
            'key' => 'button_customization',
            'name' => 'Button Customization',
            'description' => 'Customize the "Buy on Amazon" button text, colors, size, and style to match your site\'s design.',
            'plan' => 'pro',
            'enabled' => psfa_is_premium(),
            'upgrade_url' => $upgrade_url,
            'icon' => 'dashicons-edit',
        ),
        'card_layouts' => array(
            'key' => 'card_layouts',
            'name' => 'Custom Product Card Layouts',
            'description' => 'Choose from multiple product card design options - compact, detailed, image-focused, and more.',
            'plan' => 'pro',
            'enabled' => psfa_is_premium(),
            'upgrade_url' => $upgrade_url,
            'icon' => 'dashicons-layout',
        ),
        'font_customization' => array(
            'key' => 'font_customization',
            'name' => 'Font Customization',
            'description' => 'Choose custom fonts for product titles, descriptions, and prices from Google Fonts or system fonts.',
            'plan' => 'pro',
            'enabled' => psfa_is_premium(),
            'upgrade_url' => $upgrade_url,
            'icon' => 'dashicons-editor-textcolor',
        ),
        'product_badges' => array(
            'key' => 'product_badges',
            'name' => 'Product Badges',
            'description' => 'Display Best Seller rank badges and discount percentage labels on product cards to highlight top products and savings.',
            'plan' => 'pro',
            'enabled' => psfa_is_premium(),
            'upgrade_url' => $upgrade_url,
            'icon' => 'dashicons-awards',
        ),
        'deal_features' => array(
            'key' => 'deal_features',
            'name' => 'Amazon Deal Features',
            'description' => 'Show live deal data from Amazon\'s API: Lightning Deal badges, Prime Exclusive indicators, countdown timers, and Low Stock warnings.',
            'plan' => 'pro',
            'enabled' => psfa_is_premium(),
            'upgrade_url' => $upgrade_url,
            'icon' => 'dashicons-tag',
        ),
        'white_label' => array(
            'key' => 'white_label',
            'name' => 'White-Label Mode',
            'description' => 'Remove "Powered by PSFA" branding from search results for a fully white-labeled experience. Perfect for agencies and client sites.',
            'plan' => 'pro',
            'enabled' => psfa_is_premium(),
            'upgrade_url' => $upgrade_url,
            'icon' => 'dashicons-visibility',
        ),
    );
}

/**
 * Check if a premium feature is available
 *
 * @param string $feature_key Feature key
 * @return bool
 */
function psfa_is_feature_available($feature_key) {
    $features = psfa_get_all_premium_features();
    
    if (!isset($features[$feature_key])) {
        return false;
    }
    
    return $features[$feature_key]['enabled'];
}

/**
 * Render upgrade prompt for a feature
 *
 * @param array $feature Feature definition
 * @return string HTML for upgrade prompt
 */
function psfa_render_upgrade_prompt($feature) {
    if ($feature['enabled']) {
        return '';
    }
    
    $upgrade_url = esc_url($feature['upgrade_url']);
    $feature_name = esc_html($feature['name']);
    
    return sprintf(
        '<div class="psfa-upgrade-prompt">
            <p class="description">
                <strong>%s</strong> is available in Pro/Premium plans.
                <a href="%s" target="_blank" class="button button-primary" style="margin-left: 10px;">Upgrade Now</a>
            </p>
        </div>',
        $feature_name,
        $upgrade_url
    );
}

/**
 * Render feature card for admin interface
 *
 * @param array $feature Feature definition
 * @return string HTML for feature card
 */
function psfa_render_feature_card($feature) {
    $is_enabled = $feature['enabled'];
    $is_free_plan = isset($feature['plan']) && $feature['plan'] === 'free';
    $icon = isset($feature['icon']) ? $feature['icon'] : 'dashicons-star-filled';
    
    ob_start();
    ?>
    <div class="psfa-feature-card <?php echo $is_enabled ? 'psfa-feature-enabled' : 'psfa-feature-disabled'; ?>">
        <div class="psfa-feature-header">
            <span class="psfa-feature-icon dashicons <?php echo esc_attr($icon); ?>"></span>
            <h3><?php echo esc_html($feature['name']); ?></h3>
            <?php if ($is_free_plan): ?>
                <span class="psfa-badge-free-small">Free</span>
            <?php elseif (isset($feature['plan']) && $feature['plan']): ?>
                <span class="psfa-badge-premium-small">Premium</span>
            <?php endif; ?>
        </div>
        <div class="description"><?php echo wp_kses_post( $feature['description'] ); ?></div>
        <?php if (!$is_enabled): ?>
            <div style="margin-top: 12px; padding: 8px; background: #fff3cd; border-left: 3px solid #ffc107; border-radius: 3px; opacity: 1; position: relative;">
                <span class="description" style="font-size: 12px; color: #856404; opacity: 1;">
                    <a href="<?php echo esc_url($feature['upgrade_url']); ?>" target="_blank" style="color: #2271b1; text-decoration: none; font-weight: 600; opacity: 1;">Upgrade to Premium</a> to unlock this feature.
                </span>
            </div>
        <?php else: ?>
            <div class="psfa-feature-action">
                <span class="psfa-feature-active">✓ Active</span>
            </div>
        <?php endif; ?>
    </div>
    <?php
    return ob_get_clean();
}

