/**
 * Shared utility functions
 *
 * Common utilities used across all frontend modules.
 */

/**
 * Escape HTML to prevent XSS attacks
 * @param {string} text Text to escape
 * @returns {string} Escaped HTML
 */
export function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

/**
 * Extract numeric price from display string
 * @param {string} priceStr Price string (e.g., ".99")
 * @returns {number|null} Numeric price or null if invalid
 */
export function extractPrice(priceStr) {
    if (!priceStr || typeof priceStr !== 'string') return null;
    // Extract FIRST price only (handles ".99 (.50 / count)" format)
    priceStr = priceStr.replace(/\s*[\(\/].*$/, '');
    const clean = priceStr.replace(/[^0-9.]/g, '');
    if (clean === '' || isNaN(clean)) return null;
    return parseFloat(clean);
}

/**
 * Build URLSearchParams from container data attributes
 * @param {HTMLElement} container Products container element
 * @param {number} page Page number
 * @returns {URLSearchParams} URL parameters
 */
export function buildSearchParams(container, page) {
    const psfaAjax = window.psfaAjax || {};
    return new URLSearchParams({
        action: 'psfa_load_more',
        nonce: psfaAjax.nonce || '',
        query: container.dataset.query || '',
        category: container.dataset.category || '',
        page: page,
        price_min: container.dataset.priceMin || 0,
        price_max: container.dataset.priceMax || 0,
        brand: container.dataset.brand || '',
        sort: container.dataset.sort || '',
        condition: container.dataset.condition || '',
        merchant: container.dataset.merchant || '',
        availability: container.dataset.availability || '',
        // Premium filters
        has_deal: container.dataset.hasDeal || '',
        min_rating: container.dataset.minRating || 0,
        min_saving_percent: container.dataset.minSavingPercent || 0
    });
}

/**
 * Check if DOM is ready
 * @param {Function} callback Function to call when ready
 */
export function onReady(callback) {
    if (typeof jQuery !== 'undefined') {
        jQuery(document).ready(callback);
    } else if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', callback);
    } else {
        callback();
    }
}

// Export as object for backward compatibility (when built to IIFE)
export default {
    escapeHtml,
    extractPrice,
    buildSearchParams,
    onReady
};
