/**
 * Infinite scroll module
 * 
 * Handles AJAX pagination, product loading, and client-side sorting
 * when server-side sorting is unavailable.
 */

import { escapeHtml, extractPrice, buildSearchParams, onReady } from './utils.js';

let initialized = false;
let observer = null;

/**
 * Initialize infinite scroll functionality
 * Can be called multiple times to re-initialize
 */
export function initInfiniteScroll() {
    const productsContainer = document.getElementById('products-container');
    if (!productsContainer) {
        return; // Container not found
    }

    const searchInput = document.getElementById('amazon-search-input');
    const loader = document.getElementById('infinite-scroll-loader');
    const endMessage = document.getElementById('infinite-scroll-end');
    const upgradeNoticeEl = document.getElementById('psfa-upgrade-notice');
    const sentinel = document.getElementById('infinite-scroll-sentinel');
    const loadMoreBtn = document.getElementById('psfa-load-more-btn');
    const loadMoreWrapper = document.getElementById('psfa-load-more-wrapper');

    // Get pagination type from data attribute (default to 'infinite' for backward compatibility)
    const paginationType = productsContainer?.dataset.pagination || 'infinite';
    
    // If pagination is 'none', skip all pagination setup
    if (paginationType === 'none') {
        return;
    }

    // Initialize state from data attributes
    if (searchInput && searchInput.value && searchInput.value.trim()) {
        productsContainer.dataset.query = searchInput.value.trim();
    }

    let isLoading = false;
    let currentPage = parseInt(productsContainer?.dataset.page) || 1;
    let hasMore = productsContainer?.dataset.hasMore === '1';

    // Determine if client-side sorting is needed
    const hasBrand = productsContainer?.dataset.brand && productsContainer.dataset.brand !== '';
    const sortType = productsContainer?.dataset.sort || '';
    const hasSort = sortType !== '';
    const hasPriceFilter = (parseFloat(productsContainer?.dataset.priceMin) > 0 || parseFloat(productsContainer?.dataset.priceMax) > 0);
    const isPriceSort = sortType === 'Price:LowToHigh' || sortType === 'Price:HighToLow';
    const isOAuth = productsContainer?.dataset.authType === 'oauth';
    // OAuth/Creators API doesn't honor sortBy - always need client-side sorting for price sorts
    const needsClientSort = (isPriceSort && isOAuth) || (hasSort && hasBrand) || (hasPriceFilter && isPriceSort);

    /**
     * Create product HTML from item data
     * @param {Object} item Product item data
     * @returns {string} HTML string
     */
    function createProductHTML(item) {
        // Get customization settings from data attributes
        const container = document.getElementById('products-container');
        const buttonText = container?.dataset.buttonText || 'Buy on Amazon';
        const cardLayout = container?.dataset.cardLayout || 'default';
        const isPremium = container?.dataset.isPremium === '1';
        
        // Build badges HTML (premium only)
        let badgesStackHTML = '';
        let primeBadgeHTML = '';
        let bestsellerBadgeHTML = '';
        if (isPremium && item.img) {
            // Deal badges stack (top-left): Countdown OR Deal type (not both), Discount %, Low Stock
            const stackBadges = [];
            if (item.dealCountdown?.time_left_display) {
                const endDate = item.dealCountdown.end_time ? new Date(item.dealCountdown.end_time).toLocaleString('en-US', { month: 'short', day: 'numeric', year: 'numeric', hour: 'numeric', minute: '2-digit', hour12: true }) : '';
                const titleText = endDate ? `Deal ends on ${endDate}` : 'Deal ends soon';
                stackBadges.push(`<span class="psfa-badge-deal psfa-badge-countdown" data-end-time="${escapeHtml(item.dealCountdown.end_time || '')}" title="${escapeHtml(titleText)}">${item.dealCountdown.time_left_display}</span>`);
            } else if (item.dealBadge) {
                stackBadges.push(`<span class="psfa-badge-deal psfa-badge-lightning" title="${escapeHtml(item.dealBadge)}">${escapeHtml(item.dealBadge)}</span>`);
            }
            if (item.savingPercent > 0) {
                stackBadges.push(`<span class="psfa-badge-deal psfa-badge-discount" title="${item.savingPercent}% Off">${item.savingPercent}% OFF</span>`);
            }
            if (item.isStockScarce) {
                stackBadges.push(`<span class="psfa-badge-lowstock" title="Low Stock">Low Stock</span>`);
            }
            if (stackBadges.length > 0) {
                badgesStackHTML = `<div class="psfa-badges-stack">${stackBadges.join('')}</div>`;
            }
            // Prime Exclusive badge (top-right) - blue circle, outside stack
            if (item.isPrimeExclusive) {
                const primeImgUrl = (window.psfaAjax?.pluginUrl || '') + 'assets/img/buy-with-prime.png';
                primeBadgeHTML = `<img src="${escapeHtml(primeImgUrl)}" alt="Prime Exclusive" class="psfa-badge-prime-img" title="Prime Exclusive Deal">`;
            }
            // Best Seller badge (bottom-left), outside stack
            if (item.salesRank > 0 && item.salesRank <= 100) {
                bestsellerBadgeHTML = `<span class="psfa-badge-bestseller" title="#${item.salesRank} Best Seller">#${item.salesRank} Best Seller</span>`;
            }
        }
        
        // Build price HTML with original price strikethrough
        let priceHTML = '';
        if (item.price) {
            if (item.savingBasis) {
                priceHTML = `<div class="psfa-price psfa-price-font"><span class="psfa-original-price" style="text-decoration: line-through; color: #888; font-size: 0.85em;">${escapeHtml(item.savingBasis)}</span> ${escapeHtml(item.price)}</div>`;
            } else {
                priceHTML = `<div class="psfa-price psfa-price-font">${escapeHtml(item.price)}</div>`;
            }
        }
        
        return `
        <div class="psfa-product psfa-card-layout-${escapeHtml(cardLayout)}">
            ${item.img ? `<div class="psfa-image-column">
                <img src="${escapeHtml(item.img)}" alt="${escapeHtml(item.title)}">
                ${badgesStackHTML}
                ${primeBadgeHTML}
                ${bestsellerBadgeHTML}
            </div>` : ''}
            <div class="psfa-title-column">
                <h7 class="psfa-title-font" title="${escapeHtml(item.title)}">${escapeHtml(item.title)}</h7>
            </div>
            <div class="psfa-purchase-column">
            ${item.brand ? `<div class="psfa-brand psfa-description-font">${escapeHtml(item.brand)}</div>` : ''}
            ${priceHTML}
            <a href="${escapeHtml(item.url)}" target="_blank" rel="noopener" class="psfa-buy-button">
                ${item.buyButtonIcon ? `<img src="${escapeHtml(item.buyButtonIcon)}" alt="Buy on Amazon" class="psfa-buy-button-icon">` : ''}${escapeHtml(buttonText)}</a>
            </div>
        </div>
    `;
    }

    /**
     * Sort all products in container (client-side)
     * Only handles price sorting when Brand + Sort or Price Filter + Sort are active
     */
    function sortAllProducts() {
        if (!needsClientSort || !productsContainer) return;
        
        const sortType = productsContainer.dataset.sort;
        if (!sortType || (sortType !== 'Price:LowToHigh' && sortType !== 'Price:HighToLow')) {
            return;
        }
        
        const products = Array.from(productsContainer.querySelectorAll('.psfa-product'));
        if (products.length === 0) return;
        
        // Extract price and sort
        const productsWithPrice = products.map(product => {
            const priceEl = product.querySelector('.psfa-price');
            let priceStr = '';
            if (priceEl) {
                // Clone element and remove original price (strikethrough) to get current price only
                const clone = priceEl.cloneNode(true);
                const originalPriceEl = clone.querySelector('.psfa-original-price');
                if (originalPriceEl) {
                    originalPriceEl.remove();
                }
                priceStr = clone.textContent.trim();
            }
            const price = extractPrice(priceStr);
            return { product, price: price !== null ? price : Infinity };
        });
        
        // Sort by price
        productsWithPrice.sort((a, b) => {
            if (sortType === 'Price:LowToHigh') {
                return a.price - b.price;
            } else {
                return b.price - a.price;
            }
        });
        
        // Re-append sorted products
        const fragment = document.createDocumentFragment();
        productsWithPrice.forEach(({ product }) => {
            fragment.appendChild(product);
        });
        
        productsContainer.innerHTML = '';
        productsContainer.appendChild(fragment);
    }

    /**
     * Fetch and append products from a specific page
     * @param {number} page Page number to fetch
     * @returns {Promise<Object>} Result with items, hasMore, upgradeNotice
     */
    async function fetchPage(page) {
        const psfaAjax = window.psfaAjax || {};
        const params = buildSearchParams(productsContainer, page);
        const response = await fetch(psfaAjax.ajaxurl + '?' + params.toString());
        const data = await response.json();

        if (!data.success) {
            const errorMsg = data.data?.error || '';
            if (errorMsg.includes('ItemPage') || errorMsg.includes('InvalidParameterValue')) {
                throw new Error('INVALID_PAGE');
            }
            // Check if it's an upgrade notice (contains HTML)
            if (errorMsg.includes('psfa-category-upgrade-notice')) {
                // Display upgrade notice in the products container area
                const container = document.getElementById('products-container');
                if (container && container.parentNode) {
                    const noticeDiv = document.createElement('div');
                    noticeDiv.innerHTML = errorMsg;
                    container.parentNode.insertBefore(noticeDiv, container);
                }
                throw new Error('PREMIUM_CATEGORY');
            }
            throw new Error(errorMsg || 'Unknown error');
        }

        return {
            items: data.data?.items || [],
            hasMore: !!data.data?.has_more,
            upgradeNotice: data.data?.upgrade_notice || ''
        };
    }

    /**
     * Fetch pages sequentially until items are found or pages exhausted
     * @param {number} startPage Starting page number
     */
    async function fetchSequentialPages(startPage) {
        let page = startPage;
        let foundItems = false;
        const AMAZON_MAX_PAGE = 10;

        while (hasMore && page < AMAZON_MAX_PAGE) {
            page += 1;

            try {
                const result = await fetchPage(page);
                const { items, hasMore: serverHasMore, upgradeNotice } = result;

                // Append items if any found
                if (items.length > 0) {
                    const fragment = document.createDocumentFragment();
                    const tempDiv = document.createElement('div');
                    items.forEach((item) => {
                        tempDiv.innerHTML = createProductHTML(item);
                        fragment.appendChild(tempDiv.firstElementChild);
                    });
                    productsContainer.appendChild(fragment);
                    foundItems = true;
                    
                    // Re-sort if needed
                    if (needsClientSort) {
                        sortAllProducts();
                    }
                    
                    // Notify other modules (e.g., countdown timer) that new content was loaded
                    document.dispatchEvent(new CustomEvent('psfa:content-loaded'));
                }

                // Update pagination state
                hasMore = serverHasMore;
                currentPage = page;
                productsContainer.dataset.hasMore = hasMore ? '1' : '0';
                productsContainer.dataset.page = currentPage;

                // If we found items, stop to allow next scroll trigger
                if (foundItems) {
                    break;
                }

                // If no more pages, show end message
                if (!hasMore) {
                    showEndMessage(upgradeNotice);
                    break;
                }
            } catch (error) {
                if (error.message === 'INVALID_PAGE') {
                    hasMore = false;
                }
                console.error('Error loading products:', error);
                hasMore = false;
                break;
            }
        }

        // Cleanup
        if (loader) {
            loader.classList.remove('active');
        }
        isLoading = false;
        
        if (!hasMore) {
            if (observer && sentinel) {
                observer.unobserve(sentinel);
            }
            showEndMessage();
        } else if (observer && sentinel) {
            // Reset observation for next trigger
            observer.unobserve(sentinel);
            observer.observe(sentinel);
        }
    }

    /**
     * Show end of results message
     * @param {string} upgradeNotice Optional upgrade notice HTML
     */
    function showEndMessage(upgradeNotice) {
        if (endMessage) {
            endMessage.classList.add('active');
        }
        if (upgradeNotice && upgradeNoticeEl) {
            upgradeNoticeEl.innerHTML = upgradeNotice;
            upgradeNoticeEl.style.display = 'block';
        }
        // Hide load more button when no more results
        if (loadMoreWrapper) {
            loadMoreWrapper.style.display = 'none';
        }
    }

    /**
     * Load more products (entry point for infinite scroll and button click)
     */
    function loadMoreProducts() {
        if (isLoading || !hasMore || !productsContainer) {
            return;
        }

        isLoading = true;
        
        // Show appropriate loader based on pagination type
        if (paginationType === 'infinite' && loader) {
            loader.classList.add('active');
        } else if (paginationType === 'button' && loadMoreBtn) {
            loadMoreBtn.disabled = true;
            loadMoreBtn.textContent = 'Loading...';
        }

        fetchSequentialPages(currentPage).catch((error) => {
            console.error('Error loading products:', error);
            if (loader) {
                loader.classList.remove('active');
            }
            if (loadMoreBtn) {
                loadMoreBtn.disabled = false;
                loadMoreBtn.textContent = 'Load More Products';
            }
            isLoading = false;
            hasMore = false;
            if (observer && sentinel) {
                observer.unobserve(sentinel);
            }
        }).finally(() => {
            // Reset button state after loading
            if (paginationType === 'button' && loadMoreBtn && hasMore) {
                loadMoreBtn.disabled = false;
                loadMoreBtn.textContent = 'Load More Products';
            }
        });
    }

    // Disconnect existing observer if re-initializing
    if (initialized && observer) {
        observer.disconnect();
        observer = null;
    }

    // Initialize pagination based on type
    if (paginationType === 'infinite') {
        // Infinite scroll: use Intersection Observer
        if (productsContainer && 'IntersectionObserver' in window && sentinel) {
            const observerOptions = {
                root: null,
                rootMargin: '100px',
                threshold: 0.1
            };
            
            observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting && hasMore && !isLoading) {
                        loadMoreProducts();
                    }
                });
            }, observerOptions);
            
            observer.observe(sentinel);
        }
    } else if (paginationType === 'button') {
        // Manual pagination: use Load More button
        if (loadMoreBtn) {
            loadMoreBtn.addEventListener('click', function() {
                loadMoreProducts();
            });
        }
    }

    // Sort initial products on page load if needed.
    // Run once immediately (for environments like tests where DOM is already ready)
    // and schedule a second pass on DOM ready for real pages.
    if (needsClientSort && productsContainer) {
        // Immediate sort for current DOM state.
        sortAllProducts();

        // Defer a second sort until DOM ready to catch any products rendered later.
        onReady(() => {
            setTimeout(sortAllProducts, 100);
        });
    }
    
    initialized = true;
}

// Auto-initialize if container exists (for production)
if (typeof document !== 'undefined') {
    initInfiniteScroll();
}


