/**
 * Filters toggle module
 * 
 * Handles expand/collapse functionality for the filter panel
 * with smooth transitions.
 */

import { onReady } from './utils.js';

let clickHandler = null;

/**
 * Initialize filters toggle functionality
 * Can be called multiple times to re-initialize
 */
export function initFiltersToggle() {
    const toggleBtn = document.getElementById('toggle-filters-btn');
    const filters = document.querySelector('.psfa-filters');

    if (!toggleBtn || !filters) {
        return; // Elements not found
    }
    
    // Remove existing handler if re-initializing
    if (clickHandler) {
        toggleBtn.removeEventListener('click', clickHandler);
    }
    
    setupFiltersToggle(toggleBtn, filters);
}

function setupFiltersToggle(toggleBtn, filters) {
    /**
     * Set filter panel expanded/collapsed state
     * @param {boolean} expanded Whether filters should be expanded
     */
    function setState(expanded) {
        if (expanded) {
            filters.classList.remove('is-collapsed');
            toggleBtn.classList.add('is-active');
            toggleBtn.setAttribute('aria-expanded', 'true');
        } else {
            filters.classList.add('is-collapsed');
            toggleBtn.classList.remove('is-active');
            toggleBtn.setAttribute('aria-expanded', 'false');
        }
    }

    // Initialize state from data attribute
    const initiallyExpanded = filters.dataset.initialExpanded === '1';
    setState(initiallyExpanded);

    // Toggle on click
    clickHandler = function(e) {
        e.preventDefault();
        e.stopPropagation();
        const currentlyExpanded = toggleBtn.getAttribute('aria-expanded') === 'true';
        setState(!currentlyExpanded);
    };
    toggleBtn.addEventListener('click', clickHandler);
}

// Auto-initialize on DOM ready (for production)
if (typeof document !== 'undefined') {
    onReady(() => {
        initFiltersToggle();
    });
}


