/**
 * Live countdown timer for deal badges
 * Optimized: caches badge references and timestamps, only updates visible badges
 */

(function() {
    'use strict';

    // Cache of badge data: { element, endTimestamp, ended }
    let badgeCache = [];
    let intervalId = null;

    /**
     * Format seconds into countdown string
     */
    function formatCountdown(seconds) {
        if (seconds <= 0) return 'ENDED';
        
        const days = Math.floor(seconds / 86400);
        const hours = Math.floor((seconds % 86400) / 3600);
        const mins = Math.floor((seconds % 3600) / 60);
        const secs = seconds % 60;

        if (days > 0) {
            return `DEAL ENDS IN ${days}D ${hours}H ${mins}M ${secs}s`;
        } else if (hours > 0) {
            return `DEAL ENDS IN ${hours}H ${mins}M ${secs}s`;
        } else {
            return `DEAL ENDS IN ${mins}M ${secs}s`;
        }
    }

    /**
     * Scan DOM for countdown badges and cache their data
     */
    function scanBadges() {
        const badges = document.querySelectorAll('.psfa-badge-countdown[data-end-time]');
        badgeCache = [];
        
        badges.forEach(badge => {
            const endTime = badge.dataset.endTime;
            if (!endTime) return;
            
            const endTimestamp = Math.floor(new Date(endTime).getTime() / 1000);
            badgeCache.push({
                element: badge,
                endTimestamp: endTimestamp,
                ended: false
            });
        });

        // Stop interval if no badges
        if (badgeCache.length === 0 && intervalId) {
            clearInterval(intervalId);
            intervalId = null;
        }
    }

    /**
     * Update all cached countdown badges
     */
    function updateCountdowns() {
        if (badgeCache.length === 0) return;
        
        const now = Math.floor(Date.now() / 1000);
        let activeCount = 0;

        for (let i = 0; i < badgeCache.length; i++) {
            const item = badgeCache[i];
            if (item.ended) continue;
            
            const secondsLeft = item.endTimestamp - now;

            if (secondsLeft <= 0) {
                item.element.textContent = 'ENDED';
                item.element.classList.add('psfa-badge-ended');
                item.ended = true;
            } else {
                item.element.textContent = formatCountdown(secondsLeft);
                activeCount++;
            }
        }

        // Stop interval if all countdowns ended
        if (activeCount === 0 && intervalId) {
            clearInterval(intervalId);
            intervalId = null;
        }
    }

    /**
     * Initialize countdown timer
     */
    function init() {
        scanBadges();
        
        if (badgeCache.length > 0) {
            updateCountdowns();
            intervalId = setInterval(updateCountdowns, 1000);
        }

        // Re-scan when new content is loaded (infinite scroll)
        document.addEventListener('psfa:content-loaded', function() {
            scanBadges();
            if (badgeCache.length > 0 && !intervalId) {
                intervalId = setInterval(updateCountdowns, 1000);
            }
        });
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
