/**
 * Autocomplete module
 * 
 * Handles search input autocomplete with suggestions, keyboard navigation,
 * and form submission.
 */

import { escapeHtml } from './utils.js';

let currentSuggestions = [];
let selectedIndex = -1;
let debounceTimer;
let handlers = {
    input: null,
    keydown: null,
    click: null
};

/**
 * Initialize autocomplete functionality
 * Can be called multiple times to re-initialize
 */
export function initAutocomplete() {
    const searchInput = document.getElementById('amazon-search-input');
    const suggestionsDiv = document.getElementById('autocomplete-suggestions');
    
    if (!searchInput || !suggestionsDiv) {
        return; // Elements not found
    }
    
    // Remove existing handlers if re-initializing
    if (handlers.input) {
        searchInput.removeEventListener('input', handlers.input);
        searchInput.removeEventListener('keydown', handlers.keydown);
        document.removeEventListener('click', handlers.click);
    }
    
    initEventHandlers(searchInput, suggestionsDiv);
}

function initEventHandlers(searchInput, suggestionsDiv) {
    /**
     * Render suggestions list
     * @param {Array<string>} suggestions List of suggestion strings
     * @param {string} query Current search query
     */
    function renderSuggestions(suggestions, query) {
        if (suggestions.length === 0) {
            suggestionsDiv.style.display = 'none';
            return;
        }
        
        const queryLower = query.toLowerCase();
        suggestionsDiv.innerHTML = suggestions.map((suggestion, index) => {
            const lowerSuggestion = suggestion.toLowerCase();
            const matchIndex = lowerSuggestion.indexOf(queryLower);
            
            if (matchIndex !== -1) {
                const before = suggestion.substring(0, matchIndex);
                const match = suggestion.substring(matchIndex, matchIndex + query.length);
                const after = suggestion.substring(matchIndex + query.length);
                return `<div class="autocomplete-item" data-index="${index}" data-value="${escapeHtml(suggestion)}">` +
                    `${escapeHtml(before)}<strong>${escapeHtml(match)}</strong>${escapeHtml(after)}` +
                    `</div>`;
            }
            return `<div class="autocomplete-item" data-index="${index}" data-value="${escapeHtml(suggestion)}">` +
                `${escapeHtml(suggestion)}</div>`;
        }).join('');
        
        suggestionsDiv.style.display = 'block';
        
        // Add click handlers
        suggestionsDiv.querySelectorAll('.autocomplete-item').forEach(item => {
            item.addEventListener('click', function() {
                selectSuggestion(this.getAttribute('data-value'));
            });
        });
    }

    /**
     * Select a suggestion and submit form
     * @param {string} value Selected suggestion value
     */
    function selectSuggestion(value) {
        searchInput.value = value;
        suggestionsDiv.style.display = 'none';
        selectedIndex = -1;
        
        const form = searchInput.closest('form');
        if (form) {
            form.submit();
        }
    }

    /**
     * Update keyboard selection highlight
     * @param {NodeList} items Suggestion items
     */
    function updateSelection(items) {
        items.forEach((item, index) => {
            if (index === selectedIndex) {
                item.classList.add('selected');
                item.scrollIntoView({ block: 'nearest', behavior: 'smooth' });
            } else {
                item.classList.remove('selected');
            }
        });
    }

    /**
     * Fetch autocomplete suggestions from server
     * @param {string} query Search query
     */
    function fetchSuggestions(query) {
        const psfaAjax = window.psfaAjax || {};
        const params = new URLSearchParams({
            action: 'psfa_autocomplete',
            nonce: psfaAjax.nonce || '',
            q: query
        });
        
        fetch(psfaAjax.ajaxurl + '?' + params.toString())
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    currentSuggestions = data.data.suggestions || [];
                    selectedIndex = -1;
                    renderSuggestions(currentSuggestions, query);
                } else {
                    suggestionsDiv.style.display = 'none';
                }
            })
            .catch(error => {
                console.error('Autocomplete error:', error);
                suggestionsDiv.style.display = 'none';
            });
    }

    // Input event with debouncing
    handlers.input = function(e) {
        const query = e.target.value.trim();
        
        clearTimeout(debounceTimer);
        
        if (query.length < 2) {
            suggestionsDiv.style.display = 'none';
            return;
        }
        
        debounceTimer = setTimeout(() => fetchSuggestions(query), 300);
    };
    searchInput.addEventListener('input', handlers.input);

    // Keyboard navigation
    handlers.keydown = function(e) {
        if (suggestionsDiv.style.display === 'none') return;
        
        const items = suggestionsDiv.querySelectorAll('.autocomplete-item');
        
        if (e.key === 'ArrowDown') {
            e.preventDefault();
            selectedIndex = Math.min(selectedIndex + 1, items.length - 1);
            updateSelection(items);
        } else if (e.key === 'ArrowUp') {
            e.preventDefault();
            selectedIndex = Math.max(selectedIndex - 1, -1);
            updateSelection(items);
        } else if (e.key === 'Enter') {
            if (selectedIndex >= 0 && items[selectedIndex]) {
                e.preventDefault();
                selectSuggestion(items[selectedIndex].getAttribute('data-value'));
            } else {
                suggestionsDiv.style.display = 'none';
                selectedIndex = -1;
            }
        } else if (e.key === 'Escape') {
            e.preventDefault();
            suggestionsDiv.style.display = 'none';
            selectedIndex = -1;
        }
    };
    searchInput.addEventListener('keydown', handlers.keydown);

    // Hide suggestions when clicking outside
    handlers.click = function(e) {
        if (!searchInput.contains(e.target) && !suggestionsDiv.contains(e.target)) {
            suggestionsDiv.style.display = 'none';
        }
    };
    document.addEventListener('click', handlers.click);
}

// Auto-initialize if elements exist (for production)
if (typeof document !== 'undefined') {
    initAutocomplete();
}


