<?php
namespace OXICAT\CookiesManager\Admin;

use OXICAT\CookiesManager\Plugin;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Admin {

    /** @var Plugin */
    private $plugin;

    public function __construct( Plugin $plugin ) {
        $this->plugin = $plugin;

        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
    }

    public function add_admin_menu() : void {
        add_menu_page(
            __( 'Cookie Manager', 'oxicat-cookies-manager' ),
            __( 'Cookie Manager', 'oxicat-cookies-manager' ),
            'manage_options',
            'oxicat-cookies-manager',
            array( $this, 'admin_page' ),
            'dashicons-shield-alt',
            30
        );
    }

    public function enqueue_assets( string $hook ) : void {
        if ( 'toplevel_page_oxicat-cookies-manager' !== $hook ) {
            return;
        }

        wp_enqueue_style(
            'oxicat-cookies-manager-admin',
            OXICAT_COOKIES_MANAGER_URL . 'assets/css/admin.css',
            array(),
            OXICAT_COOKIES_MANAGER_VERSION
        );

        wp_enqueue_script(
            'oxicat-cookies-manager-admin',
            OXICAT_COOKIES_MANAGER_URL . 'assets/js/admin.js',
            array( 'jquery' ),
            OXICAT_COOKIES_MANAGER_VERSION,
            true
        );

        wp_localize_script(
            'oxicat-cookies-manager-admin',
            'OXICAT_CM',
            array(
                'ajaxUrl' => admin_url( 'admin-ajax.php' ),
                'nonce'   => wp_create_nonce( 'oxicat_cm_nonce' ),
                'i18n'    => array(
                    'detecting'         => __( 'Detecting cookies…', 'oxicat-cookies-manager' ),
                    'detectButton'      => __( '🔍 Detect cookies automatically', 'oxicat-cookies-manager' ),
                    'noCookiesDetected' => __( 'No cookies detected on this site.', 'oxicat-cookies-manager' ),
                    'clickToAdd'        => __( 'The following cookies were detected. Click on a cookie to prefill the form.', 'oxicat-cookies-manager' ),
                    'clickToAdd'        => __( 'The following cookies were detected. Click on a cookie to prefill the form.', 'oxicat-cookies-manager' ),
                    'documentButton'    => __( '📝 Document', 'oxicat-cookies-manager' ),
                    'detectedTitle'     => __( 'Detected cookies (%d)', 'oxicat-cookies-manager' ),
                    'addCookieTitle'    => __( 'Add a cookie', 'oxicat-cookies-manager' ),
                    'editCookieTitle'   => __( 'Edit cookie: %s', 'oxicat-cookies-manager' ),
                    'documentCookieTitle' => __( 'Document cookie: %s', 'oxicat-cookies-manager' ),
                    'savedOk'           => __( 'Cookie saved successfully!', 'oxicat-cookies-manager' ),
                    'confirmDelete'     => __( 'Delete this cookie?', 'oxicat-cookies-manager' ),
                    'deleteFailed'      => __( 'Delete failed.', 'oxicat-cookies-manager' ),
                    'saveFailed'        => __( 'Save failed.', 'oxicat-cookies-manager' ),
                ),
            )
        );
    }

    public function admin_page() : void {
        global $wpdb;
        $cookies = $wpdb->get_results( "SELECT * FROM {$this->plugin->table_name} ORDER BY cookie_name ASC" );
        include OXICAT_COOKIES_MANAGER_DIR . 'templates/admin-page.php';
    }

    public function get_category_badge( string $category ) : string {
        $labels = array(
            'necessary'  => __( 'Necessary', 'oxicat-cookies-manager' ),
            'functional' => __( 'Functional', 'oxicat-cookies-manager' ),
            'analytics'  => __( 'Analytics', 'oxicat-cookies-manager' ),
            'marketing'  => __( 'Marketing', 'oxicat-cookies-manager' ),
        );
        $label = isset( $labels[ $category ] ) ? $labels[ $category ] : $category;
        return '<span class="category-badge category-' . esc_attr( $category ) . '">' . esc_html( $label ) . '</span>';
    }

    public function ajax_save_cookie() : void {
        global $wpdb;

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( __( 'Permission denied.', 'oxicat-cookies-manager' ) );
        }

        check_ajax_referer( 'oxicat_cm_nonce', 'nonce' );

        $cookie_id = isset( $_POST['cookie_id'] ) ? absint( $_POST['cookie_id'] ) : 0;

        $data = array(
            'cookie_name'   => isset( $_POST['cookie_name'] ) ? sanitize_text_field( wp_unslash( $_POST['cookie_name'] ) ) : '',
            'purpose'       => isset( $_POST['purpose'] ) ? wp_kses_post( wp_unslash( $_POST['purpose'] ) ) : '',
            'duration'      => isset( $_POST['duration'] ) ? sanitize_text_field( wp_unslash( $_POST['duration'] ) ) : '',
            'provider'      => isset( $_POST['provider'] ) ? sanitize_text_field( wp_unslash( $_POST['provider'] ) ) : '',
            'category'      => isset( $_POST['category'] ) ? sanitize_text_field( wp_unslash( $_POST['category'] ) ) : '',
            'more_info_url' => isset( $_POST['more_info_url'] ) ? esc_url_raw( wp_unslash( $_POST['more_info_url'] ) ) : '',
        );

        if ( empty( $data['cookie_name'] ) ) {
            wp_send_json_error( __( 'Cookie name is required.', 'oxicat-cookies-manager' ) );
        }

        if ( $cookie_id > 0 ) {
            $result = $wpdb->update( $this->plugin->table_name, $data, array( 'id' => $cookie_id ) );
        } else {
            $result = $wpdb->insert( $this->plugin->table_name, $data );
        }

        if ( false !== $result ) {
    wp_send_json_success();
}

// If the cookie name already exists (UNIQUE), update the existing row instead of failing.
if ( ! empty( $data['cookie_name'] ) && ! empty( $wpdb->last_error ) && false !== stripos( $wpdb->last_error, 'Duplicate' ) ) {
    $existing_id = (int) $wpdb->get_var(
        $wpdb->prepare(
            "SELECT id FROM {$this->plugin->table_name} WHERE cookie_name = %s LIMIT 1",
            $data['cookie_name']
        )
    );

    if ( $existing_id > 0 ) {
        $update = $wpdb->update( $this->plugin->table_name, $data, array( 'id' => $existing_id ) );
        if ( false !== $update ) {
            wp_send_json_success();
        }
    }
}

$message = __( 'Error while saving.', 'oxicat-cookies-manager' );
if ( defined( 'WP_DEBUG' ) && WP_DEBUG && ! empty( $wpdb->last_error ) ) {
    $message .= ' ' . $wpdb->last_error;
}

wp_send_json_error( $message );

    }

    public function ajax_delete_cookie() : void {
        global $wpdb;

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( __( 'Permission denied.', 'oxicat-cookies-manager' ) );
        }

        check_ajax_referer( 'oxicat_cm_nonce', 'nonce' );

        $cookie_id = isset( $_POST['cookie_id'] ) ? absint( $_POST['cookie_id'] ) : 0;

        if ( ! $cookie_id ) {
            wp_send_json_error( __( 'Invalid cookie ID.', 'oxicat-cookies-manager' ) );
        }

        $result = $wpdb->delete( $this->plugin->table_name, array( 'id' => $cookie_id ) );

        if ( $result ) {
            wp_send_json_success();
        }

        wp_send_json_error( __( 'Error while deleting.', 'oxicat-cookies-manager' ) );
    }
}
