<?php
/*
Plugin Name: Overseas Express Shipment
Description: Plugin for using Overseas Express Shipment
Text Domain: overseas-express-shipment
Domain Path: /languages
Version: 1.0.1
Author: Overseas Express
Author URI: https://overseas.hr/
License: GPLv3
*/

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if ( ! defined( 'OVERSEAS_EXPRESS_SHIPMENT_VERSION' ) ) {
	define( 'OVERSEAS_EXPRESS_SHIPMENT_VERSION', '1.0.1' );
}

if ( ! defined( 'OVERSEAS_EXPRESS_SHIPMENT_URL' ) ) {
	define( 'OVERSEAS_EXPRESS_SHIPMENT_URL', plugin_dir_url( __FILE__ ) );
}

if ( ! defined( 'OVERSEAS_EXPRESS_SHIPMENT_PATH' ) ) {
	define( 'OVERSEAS_EXPRESS_SHIPMENT_PATH', plugin_dir_path( __FILE__ ) );
}

/**
 * Initialize WP_Filesystem and return it.
 *
 * @return WP_Filesystem_Base|false
 */
function overseas_express_get_filesystem() {
	global $wp_filesystem;

	if ( $wp_filesystem instanceof WP_Filesystem_Base ) {
		return $wp_filesystem;
	}

	require_once ABSPATH . 'wp-admin/includes/file.php';
	$initialized = WP_Filesystem();

	if ( ! $initialized || ! ( $wp_filesystem instanceof WP_Filesystem_Base ) ) {
		return false;
	}

	return $wp_filesystem;
}

/**
 * Verify WooCommerce checkout nonce for hooks that read $_POST.
 *
 * @return bool
 */
function overseas_express_verify_wc_checkout_nonce() {
	$nonce = isset( $_POST['woocommerce-process-checkout-nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['woocommerce-process-checkout-nonce'] ) ) : '';
	if ( empty( $nonce ) ) {
		return false;
	}
	return (bool) wp_verify_nonce( $nonce, 'woocommerce-process_checkout' );
}

/**
 * Frontend assets (checkout)
 */
function overseas_express_enqueue_frontend_assets() {
	if ( ! function_exists( 'is_checkout' ) || ! is_checkout() ) {
		return;
	}

	wp_enqueue_style(
		'overseas-express-leaflet',
		OVERSEAS_EXPRESS_SHIPMENT_URL . 'assets/vendor/leaflet/leaflet.css',
		[],
		OVERSEAS_EXPRESS_SHIPMENT_VERSION
	);

	wp_enqueue_style(
		'overseas-express-leaflet-geocoder',
		OVERSEAS_EXPRESS_SHIPMENT_URL . 'assets/vendor/leaflet/leaflet-control-geocoder.css',
		[ 'overseas-express-leaflet' ],
		OVERSEAS_EXPRESS_SHIPMENT_VERSION
	);

	wp_enqueue_script(
		'overseas-express-leaflet',
		OVERSEAS_EXPRESS_SHIPMENT_URL . 'assets/vendor/leaflet/leaflet.js',
		[],
		OVERSEAS_EXPRESS_SHIPMENT_VERSION,
		true
	);

	wp_enqueue_script(
		'overseas-express-leaflet-geocoder',
		OVERSEAS_EXPRESS_SHIPMENT_URL . 'assets/vendor/leaflet/leaflet-control-geocoder.js',
		[ 'overseas-express-leaflet' ],
		OVERSEAS_EXPRESS_SHIPMENT_VERSION,
		true
	);

	wp_enqueue_script(
		'overseas-express-checkout-map',
		OVERSEAS_EXPRESS_SHIPMENT_URL . 'assets/js/checkout-map.js',
		[ 'jquery', 'overseas-express-leaflet', 'overseas-express-leaflet-geocoder' ],
		OVERSEAS_EXPRESS_SHIPMENT_VERSION,
		true
	);

	$api_key = overseas_express_get_api_key();
	$points  = [];

	if ( ! empty( $api_key ) ) {
		$api_url  = 'https://api.overseas.hr/parcelshops?isActive=true&apikey=' . rawurlencode( $api_key );
		$response = wp_remote_get(
			$api_url,
			[
				'timeout'   => 15,
				'sslverify' => true,
			]
		);

		if ( ! is_wp_error( $response ) && 200 === wp_remote_retrieve_response_code( $response ) ) {
			$data = json_decode( wp_remote_retrieve_body( $response ), true );

			if ( isset( $data['data'] ) && is_array( $data['data'] ) ) {
				foreach ( $data['data'] as $point ) {
					if (
						! empty( $point['GeoLat'] ) && ! empty( $point['GeoLong'] ) &&
						0 != $point['GeoLat'] && 0 != $point['GeoLong'] &&
						! empty( $point['IsActive'] ) && true === $point['IsActive']
					) {
						$points[] = [
							'lat'     => $point['GeoLat'],
							'lng'     => $point['GeoLong'],
							'name'    => ( $point['ShortName'] ?? '' ) . ' - ' . ( $point['Address']['Place'] ?? '' ),
							'street'  => $point['Address']['Street'] ?? '',
							'zip'     => $point['Address']['ZipCode'] ?? '',
							'city'    => $point['Address']['Place'] ?? '',
							'OOHType' => $point['OOHType'] ?? 0,
						];
					}
				}
			}
		}
	}

	$blue_icon_url   = OVERSEAS_EXPRESS_SHIPMENT_URL . 'assets/images/marker-icon-blue.png';
	$red_icon_url    = OVERSEAS_EXPRESS_SHIPMENT_URL . 'assets/images/marker-icon-red.png';
	$shadow_icon_url = OVERSEAS_EXPRESS_SHIPMENT_URL . 'assets/images/marker-shadow.png';

	$localized_data = [
		'points'           => $points,
		'icons'            => [
			'blue'   => $blue_icon_url,
			'red'    => $red_icon_url,
			'shadow' => $shadow_icon_url,
		],
		'placeholder'      => __( 'Search for a location...', 'overseas-express-shipment' ),
		'shippingMethodId' => 'overseas_express_pickup_shipping',
		'mapContainerId'   => 'overseas-express-map',
		'ajaxUrl'          => admin_url( 'admin-ajax.php' ),
		'nonce'            => wp_create_nonce( 'overseas_express_pickup_nonce' ),
	];

	wp_localize_script(
		'overseas-express-checkout-map',
		'overseasExpressPickup',
		$localized_data
	);
}
add_action( 'wp_enqueue_scripts', 'overseas_express_enqueue_frontend_assets' );

/**
 * Admin assets (inline CSS for orders list button)
 */
function overseas_express_admin_enqueue_assets( $hook ) {
	if ( 'woocommerce_page_wc-orders' !== $hook ) {
		return;
	}

	wp_register_style(
		'overseas-express-admin-inline',
		false,
		[],
		OVERSEAS_EXPRESS_SHIPMENT_VERSION,
		'all'
	);
	wp_enqueue_style( 'overseas-express-admin-inline' );

	$slug = 'download_label';
	$css  = '
		.wc-action-button-' . $slug . '::after {
			font-family: inherit !important;
			content: "\2B07" !important;
			font-size: 14px;
			display: inline-block;
			line-height: 1;
		}
		.wp-admin .woocommerce-page .widefat .wc-action-button-' . $slug . ':hover {
			background-color: #7ad03a !important;
			color: #fff !important;
		}
		.wc-action-button-' . $slug . ' {
			cursor: pointer;
			text-decoration: none;
		}
	';

	wp_add_inline_style( 'overseas-express-admin-inline', $css );
}
add_action( 'admin_enqueue_scripts', 'overseas_express_admin_enqueue_assets' );

/**
 * Admin JS for main settings page
 */
function overseas_express_admin_enqueue_main_settings_script( $hook ) {
	if ( 'toplevel_page_overseas-express-settings' !== $hook ) {
		return;
	}

	wp_enqueue_script(
		'overseas-express-admin-main-settings',
		OVERSEAS_EXPRESS_SHIPMENT_URL . 'assets/js/admin-main-settings.js',
		[ 'jquery' ],
		OVERSEAS_EXPRESS_SHIPMENT_VERSION,
		true
	);

	$data = [
		'ajaxUrl'            => admin_url( 'admin-ajax.php' ),
		'nonce'              => wp_create_nonce( 'overseas_express_ajax_nonce' ),
		'loadingLabel'       => __( 'Loading...', 'overseas-express-shipment' ),
		'selectLabelText'    => __( 'Select a label', 'overseas-express-shipment' ),
		'errorFetchText'     => __( 'Error fetching layouts', 'overseas-express-shipment' ),
		'initialLabelLayout' => overseas_express_get_label_layout_id(),
	];

	wp_localize_script(
		'overseas-express-admin-main-settings',
		'overseasExpressAdmin',
		$data
	);
}
add_action( 'admin_enqueue_scripts', 'overseas_express_admin_enqueue_main_settings_script' );

/**
 * AJAX: set pickup (checkout)
 */
add_action( 'wp_ajax_overseas_express_set_pickup', 'overseas_express_set_pickup' );
add_action( 'wp_ajax_nopriv_overseas_express_set_pickup', 'overseas_express_set_pickup' );
function overseas_express_set_pickup() {
	$request_method = isset( $_SERVER['REQUEST_METHOD'] ) ? sanitize_key( wp_unslash( $_SERVER['REQUEST_METHOD'] ) ) : '';
	if ( 'post' !== strtolower( $request_method ) ) {
		wp_send_json_error( 'invalid_method' );
	}

	$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
	if ( ! wp_verify_nonce( $nonce, 'overseas_express_pickup_nonce' ) ) {
		wp_send_json_error( 'invalid_nonce' );
	}

	if ( ! function_exists( 'WC' ) || ! WC()->session ) {
		wp_send_json_error( 'no_session' );
	}

	$point   = sanitize_text_field( wp_unslash( $_POST['point'] ?? '' ) );
	$addr    = sanitize_text_field( wp_unslash( $_POST['address'] ?? '' ) );
	$zip     = sanitize_text_field( wp_unslash( $_POST['postcode'] ?? '' ) );
	$city    = sanitize_text_field( wp_unslash( $_POST['city'] ?? '' ) );
	$country = sanitize_text_field( wp_unslash( $_POST['country'] ?? 'HR' ) );

	if ( '' === $point || '' === $addr || '' === $zip || '' === $city ) {
		wp_send_json_error( 'missing_fields' );
	}

	if ( ! preg_match( '/^[A-Z]{2}$/', strtoupper( $country ) ) ) {
		$country = 'HR';
	} else {
		$country = strtoupper( $country );
	}

	WC()->session->set( 'overseas_express_pickup_point', $point );
	WC()->session->set( 'overseas_express_pickup_address', $addr );
	WC()->session->set( 'overseas_express_pickup_postcode', $zip );
	WC()->session->set( 'overseas_express_pickup_city', $city );
	WC()->session->set( 'overseas_express_pickup_country', $country );

	wp_send_json_success();
}

add_action(
	'woocommerce_before_calculate_totals',
	function () {
		if ( ! function_exists( 'WC' ) || ! WC()->session || ! WC()->customer ) {
			return;
		}

		$chosen = WC()->session->get( 'chosen_shipping_methods' );
		$method = is_array( $chosen ) ? ( $chosen[0] ?? '' ) : '';

		if ( $method !== 'overseas_express_pickup_shipping' ) {
			return;
		}

		$addr    = WC()->session->get( 'overseas_express_pickup_address' );
		$zip     = WC()->session->get( 'overseas_express_pickup_postcode' );
		$city    = WC()->session->get( 'overseas_express_pickup_city' );
		$country = WC()->session->get( 'overseas_express_pickup_country' );

		if ( empty( $addr ) || empty( $zip ) || empty( $city ) ) {
			return;
		}

		WC()->customer->set_shipping_address_1( $addr );
		WC()->customer->set_shipping_postcode( $zip );
		WC()->customer->set_shipping_city( $city );
		WC()->customer->set_shipping_country( $country ?: 'HR' );
	},
	20
);

/**
 * Save pickup point as shipping address (checkout)
 */
add_action(
	'woocommerce_checkout_create_order',
	function ( $order ) {

		// Verify WooCommerce checkout nonce (Plugin Check-friendly).
		$nonce = isset( $_POST['woocommerce-process-checkout-nonce'] )
			? sanitize_text_field( wp_unslash( $_POST['woocommerce-process-checkout-nonce'] ) )
			: '';

		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'woocommerce-process_checkout' ) ) {
			return;
		}

		// Read POST only AFTER nonce verification.
		$selected_point = isset( $_POST['overseas_express_selected_point'] )
			? sanitize_text_field( wp_unslash( $_POST['overseas_express_selected_point'] ) )
			: '';

		if ( '' === $selected_point ) {
			return;
		}

		$address  = isset( $_POST['overseas_express_address'] ) ? sanitize_text_field( wp_unslash( $_POST['overseas_express_address'] ) ) : '';
		$postcode = isset( $_POST['overseas_express_postcode'] ) ? sanitize_text_field( wp_unslash( $_POST['overseas_express_postcode'] ) ) : '';
		$city     = isset( $_POST['overseas_express_city'] ) ? sanitize_text_field( wp_unslash( $_POST['overseas_express_city'] ) ) : '';
		$country  = isset( $_POST['overseas_express_country'] ) ? sanitize_text_field( wp_unslash( $_POST['overseas_express_country'] ) ) : '';

		$order->set_shipping_address_1( $address );
		$order->set_shipping_postcode( $postcode );
		$order->set_shipping_city( $city );
		$order->set_shipping_country( $country );
		$order->add_meta_data( 'Pickup Point', $selected_point, true );
	},
	10,
	1
);

/**
 * Block checkout if pickup point is required but not selected
 */
add_action( 'woocommerce_after_checkout_validation', 'overseas_express_validate_shipping', 10, 2 );

function overseas_express_validate_shipping( $data, $errors ) {

	// ✅ Explicit WooCommerce checkout nonce verification (Plugin Check-friendly).
	$nonce = isset( $_POST['woocommerce-process-checkout-nonce'] )
		? sanitize_text_field( wp_unslash( $_POST['woocommerce-process-checkout-nonce'] ) )
		: '';

	if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'woocommerce-process_checkout' ) ) {
		$errors->add(
			'overseas-nonce',
			esc_html__( 'Security check failed. Please refresh the page and try again.', 'overseas-express-shipment' )
		);
		return;
	}

	$chosen_methods = WC()->session->get( 'chosen_shipping_methods' );
	if ( empty( $chosen_methods ) || ! is_array( $chosen_methods ) ) {
		return;
	}

	$selected_method = $chosen_methods[0];

	// Read POST only AFTER nonce verification.
	$selected_point = isset( $_POST['overseas_express_selected_point'] )
		? sanitize_text_field( wp_unslash( $_POST['overseas_express_selected_point'] ) )
		: '';

	if ( 'overseas_express_pickup_shipping' === $selected_method ) {
		if ( '' === $selected_point ) {
			$errors->add(
				'no-pickup',
				esc_html__( 'You must select a pickup point before completing your order.', 'overseas-express-shipment' )
			);
		}
	}

	if (
		'overseas_express_pickup_shipping' === $selected_method ||
		'overseas_express_flat_shipping' === $selected_method
	) {
		$city    = isset( $data['shipping_city'] ) ? sanitize_text_field( $data['shipping_city'] ) : '';
		$zipcode = isset( $data['shipping_postcode'] ) ? sanitize_text_field( $data['shipping_postcode'] ) : '';

		if ( '' === $city || '' === $zipcode ) {
			return;
		}

		$api_key  = overseas_express_get_api_key();
		$endpoint = 'https://api.overseas.hr/places';

		$url = add_query_arg(
			[
				// add_query_arg() does URL-encoding; don't double-encode.
				'name'    => $city,
				'zipcode' => $zipcode,
				'approx'  => 'false',
				'apikey'  => $api_key,
			],
			$endpoint
		);

		$response = wp_remote_get(
			$url,
			[
				'timeout' => 10,
			]
		);

		if ( is_wp_error( $response ) ) {
			$errors->add(
				'api-fail',
				esc_html__( 'Unable to validate cargo availability. Please try again.', 'overseas-express-shipment' )
			);
			return;
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( empty( $body['data'] ) ) {
			$errors->add(
				'no-service',
				esc_html__( 'The cargo company does not ship to your address.', 'overseas-express-shipment' )
			);
			return;
		}

		$row = $body['data'][0];

		if ( empty( $row['StandardAvailable'] ) || empty( $row['CargoAvailable'] ) ) {
			$errors->add(
				'not-available',
				esc_html__( 'This shipment method does not provide cargo service to this address.', 'overseas-express-shipment' )
			);
			return;
		}
	}
}


// Add admin menu
add_action( 'admin_menu', 'overseas_express_add_admin_menu' );

function overseas_express_add_admin_menu() {
	add_menu_page(
		__( 'Integration Shipment', 'overseas-express-shipment' ),
		__( 'Integration Shipment', 'overseas-express-shipment' ),
		'manage_woocommerce',
		'overseas-express-settings',
		'overseas_express_main_settings_page',
		'dashicons-location-alt',
		56
	);

	add_submenu_page(
		'overseas-express-settings',
		__( 'Pickup Shipment', 'overseas-express-shipment' ),
		__( 'Pickup Shipment', 'overseas-express-shipment' ),
		'manage_woocommerce',
		'overseas-express-pickup',
		'overseas_express_pickup_settings_page'
	);

	add_submenu_page(
		'overseas-express-settings',
		__( 'Home Delivery Shipment', 'overseas-express-shipment' ),
		__( 'Home Delivery Shipment', 'overseas-express-shipment' ),
		'manage_woocommerce',
		'overseas-express-flat',
		'overseas_express_flat_settings_page'
	);
}

function overseas_express_main_settings_page() {
	if ( isset( $_POST['overseas_express_save_main_settings'] ) ) {
		check_admin_referer( 'overseas_express_main_settings' );

		$api_key = sanitize_text_field( wp_unslash( $_POST['api_key'] ?? '' ) );

		$validation_response = wp_remote_get(
			'https://api.overseas.hr/apiuserinfo?apikey=' . rawurlencode( $api_key ),
			[
				'timeout'   => 15,
				'sslverify' => true,
			]
		);

		if ( is_wp_error( $validation_response ) ) {
			$error_message = $validation_response->get_error_message();
			echo '<div class="notice notice-error is-dismissible"><p>' .
				sprintf(
                    /* translators: %s: error message returned by the API request */
					esc_html__( 'API Key validation failed: %s', 'overseas-express-shipment' ),
					esc_html( $error_message )
				) .
				'</p></div>';
		} else {
			$status_code = wp_remote_retrieve_response_code( $validation_response );
			if ( 200 === $status_code ) {
				update_option( 'overseas_express_api_key', $api_key );
				update_option( 'overseas_express_sender_name', sanitize_text_field( wp_unslash( $_POST['sender_name'] ?? '' ) ) );
				update_option( 'overseas_express_sender_country', sanitize_text_field( wp_unslash( $_POST['sender_country'] ?? '' ) ) );
				update_option( 'overseas_express_sender_zipcode', sanitize_text_field( wp_unslash( $_POST['sender_zipcode'] ?? '' ) ) );
				update_option( 'overseas_express_sender_city', sanitize_text_field( wp_unslash( $_POST['sender_city'] ?? '' ) ) );
				update_option( 'overseas_express_sender_street', sanitize_text_field( wp_unslash( $_POST['sender_street'] ?? '' ) ) );
				update_option( 'overseas_express_sender_telephone', sanitize_text_field( wp_unslash( $_POST['sender_telephone'] ?? '' ) ) );
				update_option( 'overseas_express_sender_fax', sanitize_text_field( wp_unslash( $_POST['sender_fax'] ?? '' ) ) );
				update_option( 'overseas_express_sender_gsm', sanitize_text_field( wp_unslash( $_POST['sender_gsm'] ?? '' ) ) );
				update_option( 'overseas_express_sender_email', sanitize_email( wp_unslash( $_POST['sender_email'] ?? '' ) ) );

				if ( ! empty( $_POST['label_layout'] ) ) {
					update_option( 'overseas_express_label_layout_id', sanitize_text_field( wp_unslash( $_POST['label_layout'] ) ) );
				}

				echo '<div class="notice notice-success is-dismissible"><p>' .
					esc_html__( 'Settings saved successfully! API Key is valid.', 'overseas-express-shipment' ) .
					'</p></div>';
			} else {
				echo '<div class="notice notice-error is-dismissible"><p>' .
					esc_html__( 'Invalid API Key!', 'overseas-express-shipment' ) .
					'</p></div>';
			}
		}
	}

	$api_key          = get_option( 'overseas_express_api_key', '' );
	$label_layout_id  = get_option( 'overseas_express_label_layout_id', '' );
	$sender_name      = get_option( 'overseas_express_sender_name', '' );
	$sender_country   = get_option( 'overseas_express_sender_country', 'SI' );
	$sender_zipcode   = get_option( 'overseas_express_sender_zipcode', '' );
	$sender_city      = get_option( 'overseas_express_sender_city', '' );
	$sender_street    = get_option( 'overseas_express_sender_street', '' );
	$sender_telephone = get_option( 'overseas_express_sender_telephone', '' );
	$sender_fax       = get_option( 'overseas_express_sender_fax', '' );
	$sender_gsm       = get_option( 'overseas_express_sender_gsm', '' );
	$sender_email     = get_option( 'overseas_express_sender_email', '' );
	?>
	<div class="wrap">
		<h1><?php esc_html_e( 'Integration Shipment - Main Settings', 'overseas-express-shipment' ); ?></h1>
		<form method="post" action="">
			<?php wp_nonce_field( 'overseas_express_main_settings' ); ?>
			<table class="form-table">
				<tr>
					<th scope="row"><label for="api_key"><?php esc_html_e( 'API Key', 'overseas-express-shipment' ); ?></label></th>
					<td><input type="text" name="api_key" id="api_key" value="<?php echo esc_attr( $api_key ); ?>" class="regular-text" required /></td>
				</tr>
				<tr>
					<th scope="row"><label for="label_layout"><?php esc_html_e( 'Label Type Selection', 'overseas-express-shipment' ); ?></label></th>
					<td>
						<select name="label_layout" id="label_layout" class="regular-text" disabled>
							<option value=""><?php esc_html_e( 'Enter valid API to select label', 'overseas-express-shipment' ); ?></option>
						</select>
						<p class="description"><?php esc_html_e( 'Select label layout after entering a valid API key.', 'overseas-express-shipment' ); ?></p>
					</td>
				</tr>
				<tr><th colspan="2"><h2><?php esc_html_e( 'Sender Information', 'overseas-express-shipment' ); ?></h2></th></tr>
				<tr>
					<th scope="row"><label for="sender_name"><?php esc_html_e( 'Company Name', 'overseas-express-shipment' ); ?></label></th>
					<td><input type="text" name="sender_name" id="sender_name" value="<?php echo esc_attr( $sender_name ); ?>" class="regular-text" required /></td>
				</tr>
				<tr>
					<th scope="row"><label for="sender_country"><?php esc_html_e( 'Country Code', 'overseas-express-shipment' ); ?></label></th>
					<td>
						<select name="sender_country" id="sender_country" class="regular-text" required>
							<?php
							if ( function_exists( 'WC' ) && WC()->countries ) {
								$countries = WC()->countries->get_countries();
								foreach ( $countries as $code => $name ) {
									printf(
										'<option value="%1$s" %2$s>%3$s (%1$s)</option>',
										esc_attr( $code ),
										selected( $sender_country, $code, false ),
										esc_html( $name )
									);
								}
							}
							?>
						</select>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="sender_zipcode"><?php esc_html_e( 'Zip Code', 'overseas-express-shipment' ); ?></label></th>
					<td><input type="text" name="sender_zipcode" id="sender_zipcode" value="<?php echo esc_attr( $sender_zipcode ); ?>" class="regular-text" required /></td>
				</tr>
				<tr>
					<th scope="row"><label for="sender_city"><?php esc_html_e( 'City', 'overseas-express-shipment' ); ?></label></th>
					<td><input type="text" name="sender_city" id="sender_city" value="<?php echo esc_attr( $sender_city ); ?>" class="regular-text" required /></td>
				</tr>
				<tr>
					<th scope="row"><label for="sender_street"><?php esc_html_e( 'Street and Number', 'overseas-express-shipment' ); ?></label></th>
					<td><input type="text" name="sender_street" id="sender_street" value="<?php echo esc_attr( $sender_street ); ?>" class="regular-text" required /></td>
				</tr>
				<tr>
					<th scope="row"><label for="sender_telephone"><?php esc_html_e( 'Telephone', 'overseas-express-shipment' ); ?></label></th>
					<td><input type="tel" name="sender_telephone" id="sender_telephone" value="<?php echo esc_attr( $sender_telephone ); ?>" class="regular-text" required /></td>
				</tr>
				<tr>
					<th scope="row"><label for="sender_fax"><?php esc_html_e( 'Fax', 'overseas-express-shipment' ); ?></label></th>
					<td><input type="tel" name="sender_fax" id="sender_fax" value="<?php echo esc_attr( $sender_fax ); ?>" class="regular-text" /></td>
				</tr>
				<tr>
					<th scope="row"><label for="sender_gsm"><?php esc_html_e( 'Notify GSM', 'overseas-express-shipment' ); ?></label></th>
					<td><input type="tel" name="sender_gsm" id="sender_gsm" value="<?php echo esc_attr( $sender_gsm ); ?>" class="regular-text" required /></td>
				</tr>
				<tr>
					<th scope="row"><label for="sender_email"><?php esc_html_e( 'Notify Email', 'overseas-express-shipment' ); ?></label></th>
					<td><input type="email" name="sender_email" id="sender_email" value="<?php echo esc_attr( $sender_email ); ?>" class="regular-text" required /></td>
				</tr>
			</table>
			<?php submit_button( esc_html__( 'Save Settings', 'overseas-express-shipment' ), 'primary', 'overseas_express_save_main_settings' ); ?>
		</form>
	</div>
	<?php
}

// AJAX handler for layouts
add_action(
	'wp_ajax_overseas_express_fetch_layouts',
	function() {
		check_ajax_referer( 'overseas_express_ajax_nonce', 'nonce' );

		if ( empty( $_POST['api_key'] ) ) {
			wp_send_json_error( esc_html__( 'API key is required.', 'overseas-express-shipment' ) );
		}

		$api_key  = sanitize_text_field( wp_unslash( $_POST['api_key'] ) );
		$response = wp_remote_get(
			'https://api.overseas.hr/layouts?apikey=' . rawurlencode( $api_key ),
			[
				'timeout'   => 15,
				'sslverify' => true,
			]
		);

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( esc_html( $response->get_error_message() ) );
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( ! isset( $data['data'] ) ) {
			wp_send_json_error( esc_html__( 'API Key is Not Valid.', 'overseas-express-shipment' ) );
		}

		$layouts = array_filter(
			$data['data'],
			static function( $item ) {
				return isset( $item['layoutType'] ) && 0 === (int) $item['layoutType'];
			}
		);

		wp_send_json_success( array_values( $layouts ) );
	}
);

// Pickup Settings Page
function overseas_express_pickup_settings_page() {
	?>
	<div class="wrap">
		<h1><?php esc_html_e( 'Pickup Point Shipping Settings', 'overseas-express-shipment' ); ?></h1>

		<div class="notice notice-info">
			<p><?php esc_html_e( 'Configure pickup point shipping settings below. For API Key, Tokens and Sender information, please visit the main settings page.', 'overseas-express-shipment' ); ?></p>
		</div>

		<p>
			<?php
			printf(
                /* translators: %s is the navigation path inside WooCommerce admin */
				esc_html__( 'To configure Pickup Point shipping method, go to: %s', 'overseas-express-shipment' ),
				'<strong>' . esc_html__( 'WooCommerce → Settings → Shipping → Pickup Shipment', 'overseas-express-shipment' ) . '</strong>'
			);
			?>
		</p>

		<p>
			<a href="<?php echo esc_url( admin_url( 'admin.php?page=wc-settings&tab=shipping&section=overseas_express_pickup_shipping' ) ); ?>"
				class="button button-primary">
				<?php esc_html_e( 'Configure Pickup Point Settings', 'overseas-express-shipment' ); ?>
			</a>
		</p>

		<hr>

		<h2><?php esc_html_e( 'Current Configuration', 'overseas-express-shipment' ); ?></h2>

		<?php
		$pickup_options = get_option( 'woocommerce_overseas_express_pickup_shipping_settings', [] );

		if ( ! empty( $pickup_options ) ) {
			echo '<table class="widefat fixed striped">';
			echo '<thead><tr><th>' . esc_html__( 'Setting', 'overseas-express-shipment' ) . '</th><th>' . esc_html__( 'Value', 'overseas-express-shipment' ) . '</th></tr></thead>';
			echo '<tbody>';

			$setting_labels = [
				'pickupEnabled'      => esc_html__( 'Enabled', 'overseas-express-shipment' ),
				'pickupTitle'        => esc_html__( 'Title', 'overseas-express-shipment' ),
				'price'              => esc_html__( 'Delivery Fee', 'overseas-express-shipment' ),
				'pickupCodes'        => esc_html__( 'Zip/Post Codes', 'overseas-express-shipment' ),
				'pickupAvailability' => esc_html__( 'Availability', 'overseas-express-shipment' ),
				'freePickupShipment' => esc_html__( 'Free Shipment Threshold', 'overseas-express-shipment' ),
			];

			foreach ( $setting_labels as $key => $label ) {
				if ( isset( $pickup_options[ $key ] ) ) {
					$value = $pickup_options[ $key ];

					if ( 'pickupEnabled' === $key ) {
						$value = ( 'yes' === $value ) ? esc_html__( 'Yes', 'overseas-express-shipment' ) : esc_html__( 'No', 'overseas-express-shipment' );
					} elseif ( '' === $value ) {
						$value = esc_html__( 'Not set', 'overseas-express-shipment' );
					}

					echo '<tr><td><strong>' . esc_html( $label ) . '</strong></td><td>' . esc_html( $value ) . '</td></tr>';
				}
			}

			echo '</tbody></table>';
		} else {
			echo '<p>' . esc_html__( 'No settings configured yet.', 'overseas-express-shipment' ) . '</p>';
		}
		?>
	</div>
	<?php
}

// Flat Settings Page
function overseas_express_flat_settings_page() {
	?>
	<div class="wrap">
		<h1><?php esc_html_e( 'Home Delivery Shipping Settings', 'overseas-express-shipment' ); ?></h1>

		<div class="notice notice-info">
			<p><?php esc_html_e( 'Configure home delivery shipping settings below. For API Key, Tokens and Sender information, please visit the main settings page.', 'overseas-express-shipment' ); ?></p>
		</div>

		<p>
			<?php
			printf(
                /* translators: %s is the navigation path inside WooCommerce admin */
				esc_html__( 'To configure Home Delivery shipping method, go to: %s', 'overseas-express-shipment' ),
				'<strong>' . esc_html__( 'WooCommerce → Settings → Shipping → Home Delivery Shipment', 'overseas-express-shipment' ) . '</strong>'
			);
			?>
		</p>

		<p>
			<a href="<?php echo esc_url( admin_url( 'admin.php?page=wc-settings&tab=shipping&section=overseas_express_flat_shipping' ) ); ?>"
				class="button button-primary">
				<?php esc_html_e( 'Configure Home Delivery Settings', 'overseas-express-shipment' ); ?>
			</a>
		</p>

		<hr>

		<h2><?php esc_html_e( 'Current Configuration', 'overseas-express-shipment' ); ?></h2>

		<?php
		$flat_options = get_option( 'woocommerce_overseas_express_flat_shipping_settings', [] );

		if ( ! empty( $flat_options ) ) {
			echo '<table class="widefat fixed striped">';
			echo '<thead><tr><th>' . esc_html__( 'Setting', 'overseas-express-shipment' ) . '</th><th>' . esc_html__( 'Value', 'overseas-express-shipment' ) . '</th></tr></thead>';
			echo '<tbody>';

			$setting_labels = [
				'flatEnabled'      => esc_html__( 'Enabled', 'overseas-express-shipment' ),
				'flatTitle'        => esc_html__( 'Title', 'overseas-express-shipment' ),
				'price'            => esc_html__( 'Delivery Fee', 'overseas-express-shipment' ),
				'flatCodes'        => esc_html__( 'Zip/Post Codes', 'overseas-express-shipment' ),
				'flatAvailability' => esc_html__( 'Availability', 'overseas-express-shipment' ),
				'freeFlatShipment' => esc_html__( 'Free Shipment Threshold', 'overseas-express-shipment' ),
			];

			foreach ( $setting_labels as $key => $label ) {
				if ( isset( $flat_options[ $key ] ) ) {
					$value = $flat_options[ $key ];

					if ( 'flatEnabled' === $key ) {
						$value = ( 'yes' === $value ) ? esc_html__( 'Yes', 'overseas-express-shipment' ) : esc_html__( 'No', 'overseas-express-shipment' );
					} elseif ( '' === $value ) {
						$value = esc_html__( 'Not set', 'overseas-express-shipment' );
					}

					echo '<tr><td><strong>' . esc_html( $label ) . '</strong></td><td>' . esc_html( $value ) . '</td></tr>';
				}
			}

			echo '</tbody></table>';
		} else {
			echo '<p>' . esc_html__( 'No settings configured yet.', 'overseas-express-shipment' ) . '</p>';
		}
		?>
	</div>
	<?php
}

/**
 * Helpers
 */
function overseas_express_get_sender_data() {
	return [
		'name'            => get_option( 'overseas_express_sender_name', '' ),
		'countryCode'     => get_option( 'overseas_express_sender_country', 'SI' ),
		'zipcode'         => get_option( 'overseas_express_sender_zipcode', '' ),
		'city'            => get_option( 'overseas_express_sender_city', '' ),
		'streetAndNumber' => get_option( 'overseas_express_sender_street', '' ),
		'telephone'       => get_option( 'overseas_express_sender_telephone', '' ),
		'fax'             => get_option( 'overseas_express_sender_fax', '' ),
		'notifyGSM'       => get_option( 'overseas_express_sender_gsm', '' ),
		'notifyEmail'     => get_option( 'overseas_express_sender_email', '' ),
	];
}

function overseas_express_get_api_key() {
	return get_option( 'overseas_express_api_key', '' );
}

function overseas_express_get_label_layout_id() {
	return get_option( 'overseas_express_label_layout_id', '' );
}

/**
 * Shipping Methods: Pickup
 */
if ( ! class_exists( 'OverseasExpress_Shipping_Options' ) ) {

	function overseas_express_shipping_methods_init() {

		class OverseasExpress_Shipping_Options extends WC_Shipping_Method {

			public function __construct() {
				$this->id                  = 'overseas_express_pickup_shipping';
				$this->method_title        = __( 'Pickup Shipment', 'overseas-express-shipment' );
				$this->title               = __( 'Pickup Shipment', 'overseas-express-shipment' );
				$this->options_array_label = 'overseas_express_shipping_options';
				$this->method_description  = __( 'Pickup Shipment Description', 'overseas-express-shipment' );

				add_action( 'woocommerce_update_options_shipping_' . $this->id, [ $this, 'process_admin_options' ] );
				add_action( 'woocommerce_update_options_shipping_' . $this->id, [ $this, 'process_shipping_options' ] );
				$this->init();
			}

			public function init() {
				$this->init_form_fields();
				$this->init_settings();

				$this->pickupEnabled      = $this->get_option( 'pickupEnabled' );
				$this->pickupTitle        = __( 'Pickup Shipment', 'overseas-express-shipment' );
				$this->pickupType         = 'fixed';
				$this->pickupFee          = $this->get_option( 'price' );
				$this->pickupCodes        = $this->get_option( 'pickupCodes' );
				$this->pickupAvailability = $this->get_option( 'pickupAvailability' );
				$this->pickupCountries    = $this->get_option( 'pickupCountries' );
				$this->freePickupShipment = $this->get_option( 'freePickupShipment' );

				$this->get_shipping_options();

				add_filter( 'woocommerce_shipping_methods', [ $this, 'add_overseas_express_shipping_methods' ] );
				add_action( 'woocommerce_checkout_update_order_meta', [ $this, 'overseas_express_field_update_shipping_order_meta' ], 10, 2 );

				if ( ! is_admin() ) {
					add_action( 'woocommerce_cart_shipping_method_full_label', [ $this, 'negative_pickup_label' ], 10, 2 );
				}

				if ( is_admin() ) {
					add_action( 'woocommerce_admin_order_data_after_shipping_address', [ $this, 'overseas_express_display_shipping_admin_order_meta' ], 10, 2 );
				}
			}

			public function calculate_shipping( $package = [] ) {
				$shipping_total = 0;
				$pickupFee      = ( trim( (string) $this->pickupFee ) === '' ) ? 0 : (float) $this->pickupFee;
				$cart_total     = WC()->cart->cart_contents_total;

				if ( $cart_total >= (float) $this->freePickupShipment ) {
					$shipping_total = 0;
				} else {
					if ( 'fixed' === $this->pickupType ) {
						$shipping_total = $pickupFee;
					}

					if ( 'percent' === $this->pickupType ) {
						$shipping_total = $package['contents_cost'] * ( $pickupFee / 100 );
					}

					if ( 'product' === $this->pickupType ) {
						foreach ( $package['contents'] as $item_id => $values ) {
							$_product = $values['data'];
							if ( $values['quantity'] > 0 && $_product->needs_shipping() ) {
								$shipping_total += $pickupFee * $values['quantity'];
							}
						}
					}
				}

				$shipping_total_with_tax = $shipping_total;
				$tax_total               = 0;
				$label_suffix            = $this->pickupTitle;
				$taxes                   = [];

				if ( wc_tax_enabled() ) {
					$tax_rates = WC_Tax::get_shipping_tax_rates();
					$taxes     = WC_Tax::calc_shipping_tax( $shipping_total, $tax_rates );
					$tax_total = array_sum( $taxes );
					$shipping_total_with_tax += $tax_total;

					$suffix_template = get_option( 'woocommerce_price_display_suffix', '' );

					$price_excl_tax = wc_price( $shipping_total );
					$price_incl_tax = wc_price( $shipping_total_with_tax );
					$tax_amount     = wc_price( $tax_total );

					$search         = [ '{tax_amount}', '{price_including_tax}', '{price_excluding_tax}' ];
					$replace        = [ $tax_amount, $price_incl_tax, $price_excl_tax ];
					$dynamic_suffix = str_replace( $search, $replace, $suffix_template );

					if ( ! empty( $dynamic_suffix ) ) {
						$label_suffix .= ' (' . $dynamic_suffix . ')';
					}
				}

				$rate = [
					'id'    => $this->id,
					'label' => $label_suffix,
					'cost'  => $shipping_total,
					'taxes' => $taxes,
				];

				$this->add_rate( $rate );
			}

			public function init_form_fields() {
				$this->form_fields = [
					'pickupEnabled'      => [
						'title'   => esc_html__( 'Enable', 'overseas-express-shipment' ),
						'type'    => 'checkbox',
						'label'   => esc_html__( 'Enable Pickup Point', 'overseas-express-shipment' ),
						'default' => 'no',
					],
					'pickupCodes'        => [
						'title'       => esc_html__( 'Zip/Post Codes', 'overseas-express-shipment' ),
						'type'        => 'textarea',
						'description' => esc_html__( 'What zip/post codes can use the local pickup option? Separate codes with a comma. Accepts wildcards, e.g. P* will match a postcode of PE30.', 'overseas-express-shipment' ),
						'default'     => '',
						'desc_tip'    => true,
						'placeholder' => '12345, 56789, etc',
					],
					'price'              => [
						'title'       => esc_html__( 'Pickup Point Delivery Fee/Discount', 'overseas-express-shipment' ),
						'type'        => 'price',
						'description' => esc_html__( 'What fee do you want to charge for local pickup? Enter negative value for discount. Leave blank to disable.', 'overseas-express-shipment' ),
						'default'     => '',
						'desc_tip'    => true,
						'placeholder' => wc_format_localized_price( 0 ),
					],
					'pickupAvailability' => [
						'title'   => esc_html__( 'Method availability', 'overseas-express-shipment' ),
						'type'    => 'select',
						'default' => 'all',
						'class'   => 'availability',
						'options' => [
							'all'      => esc_html__( 'All allowed countries', 'overseas-express-shipment' ),
							'specific' => esc_html__( 'Specific Countries', 'overseas-express-shipment' ),
						],
					],
					'pickupCountries'    => [
						'title'             => esc_html__( 'Specific Countries', 'overseas-express-shipment' ),
						'type'              => 'multiselect',
						'class'             => 'chosen_select',
						'css'               => 'width: 450px;',
						'default'           => '',
						'options'           => function_exists( 'WC' ) && WC()->countries ? WC()->countries->get_shipping_countries() : [],
						'custom_attributes' => [
							'data-placeholder' => esc_html__( 'Select some countries', 'overseas-express-shipment' ),
						],
					],
					'freePickupShipment' => [
						'title'       => esc_html__( 'Custom Free Shipment', 'overseas-express-shipment' ),
						'type'        => 'price',
						'description' => esc_html__( 'Enter the users price. For example if it is greater than $50, shipping will be free.', 'overseas-express-shipment' ),
						'default'     => '',
						'desc_tip'    => true,
					],
				];
			}

			public function admin_options() {
				?>
				<h3><?php echo esc_html( $this->method_title ); ?></h3>
				<p><?php esc_html_e( 'Pickup Shipment Description', 'overseas-express-shipment' ); ?></p>
				<table class="form-table">
					<?php $this->generate_settings_html(); ?>
				</table>
				<?php
			}

			public function is_available( $package ) {
				$pickup_enabled = ( 'no' === $this->pickupEnabled );
				if ( $pickup_enabled ) {
					return false;
				}

				$pickupCodes = [];
				if ( ! empty( $this->pickupCodes ) ) {
					foreach ( explode( ',', $this->pickupCodes ) as $code ) {
						$pickupCodes[] = $this->clean( $code );
					}
				}

				if ( is_array( $pickupCodes ) && ! empty( $pickupCodes ) ) {
					$found_match = false;

					if ( in_array( $this->clean( $package['destination']['postcode'] ), $pickupCodes, true ) ) {
						$found_match = true;
					}

					if ( ! $found_match ) {
						$customer_postcode = $this->clean( $package['destination']['postcode'] );
						foreach ( $pickupCodes as $c ) {
							$pattern = '/^' . str_replace( '_', '[0-9a-zA-Z]', preg_quote( $c, '/' ) ) . '$/i';
							if ( preg_match( $pattern, $customer_postcode ) ) {
								$found_match = true;
								break;
							}
						}
					}

					if ( ! $found_match ) {
						$customer_postcode        = $this->clean( $package['destination']['postcode'] );
						$customer_postcode_length = strlen( $customer_postcode );

						for ( $i = 0; $i <= $customer_postcode_length; $i++ ) {
							if ( in_array( $customer_postcode, $pickupCodes, true ) ) {
								$found_match = true;
							}
							$customer_postcode = substr( $customer_postcode, 0, -2 ) . '*';
						}
					}

					if ( ! $found_match ) {
						return false;
					}
				}

				if ( 'specific' === $this->pickupAvailability ) {
					$ship_to_countries = $this->pickupCountries;
				} else {
					$ship_to_countries = function_exists( 'WC' ) && WC()->countries ? array_keys( WC()->countries->get_shipping_countries() ) : [];
				}

				if ( is_array( $ship_to_countries ) && ! empty( $ship_to_countries ) ) {
					if ( ! in_array( $package['destination']['country'], $ship_to_countries, true ) ) {
						return false;
					}
				}

				// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
				return apply_filters( 'woocommerce_shipping_' . $this->id . '_is_available', true, $package );
			}

			public function clean( $code ) {
				return str_replace( '-', '', sanitize_title( $code ) ) . ( strstr( $code, '*' ) ? '*' : '' );
			}

			public function validate_shipping_options_table_field( $key ) { // phpcs:ignore
				return false;
			}

            public function process_shipping_options() {
                $options = [];

                // 1) Nonce verification (add this nonce field in your form).
                $nonce = isset( $_POST[ $this->id . '_options_nonce' ] )
                    ? sanitize_text_field( wp_unslash( $_POST[ $this->id . '_options_nonce' ] ) )
                    : '';

                if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, $this->id . '_save_options' ) ) {
                    return;
                }

                // 2) Read POST only after nonce verification.
                $key = $this->id . '_options';

                if ( isset( $_POST[ $key ] ) ) {
                    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                    $raw = wp_unslash( $_POST[ $key ] );

                    if ( is_array( $raw ) ) {
                        // Sanitize deeply (covers nested arrays).
                        $options = map_deep( $raw, 'wc_clean' );
                    }
                }

                update_option( $this->options_array_label, $options );
                $this->get_shipping_options();
            }


			public function get_shipping_options() {
				$this->shipping_options = array_filter( (array) get_option( $this->options_array_label ) );
			}

            public function overseas_express_field_update_shipping_order_meta( $order_id, $posted ) {

                // Verify WooCommerce checkout nonce BEFORE reading $_POST.
                $nonce = isset( $_POST['woocommerce-process-checkout-nonce'] )
                    ? sanitize_text_field( wp_unslash( $_POST['woocommerce-process-checkout-nonce'] ) )
                    : '';

                if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'woocommerce-process_checkout' ) ) {
                    return;
                }

                if ( is_array( $posted['shipping_method'] ) && in_array( $this->id, $posted['shipping_method'], true ) ) {
                    if ( isset( $_POST['shipping_option'] ) && '' !== $_POST['shipping_option'] ) {
                        $shipping_option = sanitize_text_field( wp_unslash( $_POST['shipping_option'] ) );
                        update_post_meta( $order_id, 'overseas_express_shipping_option', $shipping_option );

                        if ( WC()->session ) {
                            WC()->session->_chosen_shipping_option = $shipping_option; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
                        }
                    }
                } else {
                    if ( WC()->session ) {
                        $chosen_method = WC()->session->get( 'chosen_shipping_methods' );
                        $chosen_option = WC()->session->_chosen_shipping_option ?? ''; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
                        if ( is_array( $chosen_method ) && in_array( $this->id, $chosen_method, true ) && $chosen_option ) {
                            update_post_meta( $order_id, 'overseas_express_shipping_option', sanitize_text_field( $chosen_option ) );
                        }
                    }
                }
            }


			public function overseas_express_display_shipping_admin_order_meta( $order ) {
				$selected_option = get_post_meta( $order->get_id(), 'overseas_express_shipping_option', true );
				if ( $selected_option ) {
					echo '<p><strong>' . esc_html( $this->pickupTitle ) . ':</strong> ' . esc_html( $selected_option ) . '</p>';
				}
			}

			public function add_overseas_express_shipping_methods( $methods ) {
				$methods[] = $this;
				return $methods;
			}

			public function negative_pickup_label( $label, $method ) {
				if ( ! is_admin() && 'overseas_express_pickup_shipping' === $method->id ) {
					$label = $method->label . ' : ' . wc_price( $method->cost );
				}
				return $label;
			}
		}

		new OverseasExpress_Shipping_Options();
	}

	add_action( 'woocommerce_shipping_init', 'overseas_express_shipping_methods_init' );

	/**
	 * AJAX: save selected option
	 *
	 * NOTE: Plugin Check requires nonce verification. Your JS should send `nonce`
	 * generated with `wp_create_nonce('overseas_express_pickup_nonce')` (or a dedicated one).
	 */
	add_action( 'wp_ajax_overseas_express_save_selected', 'overseas_express_save_selected_option' );
	add_action( 'wp_ajax_nopriv_overseas_express_save_selected', 'overseas_express_save_selected_option' );
	function overseas_express_save_selected_option() {
		$nonce = isset( $_REQUEST['nonce'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'overseas_express_pickup_nonce' ) ) {
			wp_die( esc_html__( 'Invalid nonce.', 'overseas-express-shipment' ) );
		}

		if ( isset( $_GET['shipping_option'] ) && ! empty( $_GET['shipping_option'] ) ) {
			$selected_option = sanitize_text_field( wp_unslash( $_GET['shipping_option'] ) );

			$allowed = [ 'pickup', 'flat' ];
			if ( ! in_array( $selected_option, $allowed, true ) ) {
				wp_die( esc_html__( 'Invalid selection.', 'overseas-express-shipment' ) );
			}
			if ( WC()->session ) {
				WC()->session->_chosen_shipping_option = $selected_option; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
			}
		}
		wp_die();
	}

	/**
	 * Admin order buttons (Create/Update/Print)
	 */
	add_action( 'woocommerce_admin_order_data_after_order_details', 'overseas_express_add_order_buttons' );
	function overseas_express_add_order_buttons( $order ) {
		$order_id = $order->get_id();

		$shipment_id = get_post_meta( $order_id, '_overseas_shipment_id', true );
		$cargo_ids   = get_post_meta( $order_id, '_overseas_cargo_ids', true );

		if ( empty( $cargo_ids ) ) {
			$cargo_ids = [];
		} elseif ( is_string( $cargo_ids ) ) {
			$maybe_unserialized = maybe_unserialize( $cargo_ids );
			if ( is_array( $maybe_unserialized ) ) {
				$cargo_ids = $maybe_unserialized;
			} else {
				$maybe_json = json_decode( $cargo_ids, true );
				$cargo_ids  = is_array( $maybe_json ) ? $maybe_json : [ $cargo_ids ];
			}
		} elseif ( ! is_array( $cargo_ids ) ) {
			$cargo_ids = (array) $cargo_ids;
		}

		if ( $shipment_id ) {
			$update_url = wp_nonce_url(
				admin_url( 'admin-post.php?action=update_overseas_shipment&order_id=' . (int) $order_id ),
				'overseas_express_update_shipment_' . (int) $order_id,
				'_wpnonce'
			);

			$print_label_url = wp_nonce_url(
				admin_url( 'admin-post.php?action=print_overseas_label&order_id=' . (int) $order_id ),
				'overseas_express_print_label_' . (int) $order_id,
				'_wpnonce'
			);

			$first_cargo_id = ! empty( $cargo_ids ) ? reset( $cargo_ids ) : '';

			echo '<a href="' . esc_url( $update_url ) . '" style="margin-right: 10px; margin-top:15px" class="button button-primary">' . esc_html__( 'Update Shipment', 'overseas-express-shipment' ) . '</a>';

			if ( ! empty( $first_cargo_id ) ) {
				$tracking_url = 'https://inet.overseas.hr/auth?id=' . rawurlencode( (string) $first_cargo_id );
				echo ' <a href="' . esc_url( $tracking_url ) . '" target="_blank" rel="noopener noreferrer" style="margin-right: 10px; margin-top:15px" class="button button-secondary">' . esc_html__( 'Show Tracking', 'overseas-express-shipment' ) . '</a>';
			} else {
				echo ' <button type="button" style="margin-right: 10px; margin-top:15px" class="button" disabled title="' . esc_attr__( 'No tracking ID found', 'overseas-express-shipment' ) . '">' . esc_html__( 'Show Tracking', 'overseas-express-shipment' ) . '</button>';
			}

			echo ' <a href="' . esc_url( $print_label_url ) . '" target="_blank" rel="noopener noreferrer" style="margin-right: 10px; margin-top:15px" class="button">' . esc_html__( 'Print Label', 'overseas-express-shipment' ) . '</a>';
		} else {
			$create_url = wp_nonce_url(
				admin_url( 'admin-post.php?action=create_overseas_express_shipment&order_id=' . (int) $order_id ),
				'overseas_express_create_shipment_' . (int) $order_id,
				'_wpnonce'
			);

			echo '<p style="margin-top: 10px;">
				<a href="' . esc_url( $create_url ) . '" style="margin-right: 10px; margin-top:15px" class="button button-primary">' . esc_html__( 'Create Integration Shipment', 'overseas-express-shipment' ) . '</a>
			</p>';
		}
	}

	function overseas_express_admin_error_redirect( $order_id, $message ) {
		$url = admin_url(
			'post.php?post=' . (int) $order_id .
			'&action=edit&shipment_error=' . rawurlencode( (string) $message )
		);
		wp_safe_redirect( $url );
		exit;
	}

	add_action( 'admin_post_create_overseas_express_shipment', 'overseas_express_handle_create_shipment' );
	function overseas_express_handle_create_shipment() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'Unauthorized', 'overseas-express-shipment' ) );
		}

		$order_id = isset( $_GET['order_id'] ) ? absint( $_GET['order_id'] ) : 0;
		if ( ! $order_id ) {
			overseas_express_admin_error_redirect( 0, 'Invalid order ID.' );
		}

		$nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'overseas_express_create_shipment_' . $order_id ) ) {
			wp_die( esc_html__( 'Security check failed.', 'overseas-express-shipment' ) );
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			overseas_express_admin_error_redirect( $order_id, 'Order not found.' );
		}

		$api_key     = overseas_express_get_api_key();
		$sender_data = overseas_express_get_sender_data();

		if ( empty( $api_key ) ) {
			overseas_express_admin_error_redirect( $order_id, 'Missing API Key. Please configure it in Integration Shipment Settings.' );
		}

		if ( empty( $sender_data['name'] ) ) {
			overseas_express_admin_error_redirect( $order_id, 'Missing Sender Information. Please configure it in Integration Shipment Settings.' );
		}

		$cod_value = 0;
		if ( 'cod' === $order->get_payment_method() ) {
			$cod_value = (float) $order->get_total();
		}

		$collies = [];
		$i       = 1;
		foreach ( $order->get_items() as $item_id => $item ) {
			$product      = $item->get_product();
			$quantity     = $item->get_quantity();
			$product_name = $item->get_name();
			$weight       = $product ? (float) $product->get_weight() : 0;
			$sku          = $product ? $product->get_sku() : '';
			$ref_prefix   = 'COLLIE' . $i . '_';

			$collies[] = [
				'unitAmount' => $weight > 0 ? $weight * $quantity : 0,
				'ref1'       => $ref_prefix . 'REF',
				'ref2'       => $ref_prefix . 'ORD' . $order_id,
				'ref3'       => $ref_prefix . 'INV' . $order_id,
				'refInt'     => $ref_prefix . 'INT' . $order_id,
				'remark'     => $product_name,
				'barcode'    => $sku ? $sku : 'ORDER' . $order_id . '_ITEM' . $item_id,
			];
			$i++;
		}

		$body = [
			'cosignee'            => [
				'name'            => trim( $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name() ),
				'countryCode'     => $order->get_shipping_country(),
				'zipcode'         => $order->get_shipping_postcode(),
				'city'            => $order->get_shipping_city(),
				'streetAndNumber' => $order->get_shipping_address_1(),
				'telephone'       => $order->get_billing_phone(),
				'fax'             => '',
				'notifyGSM'       => $order->get_billing_phone(),
				'notifyEmail'     => $order->get_billing_email(),
			],
			'sender'              => $sender_data,
			'cosigneeNotifyType'  => 'Email',
			'isSenderNonCustomer' => false,
			'unitAmount'          => array_sum( wp_list_pluck( $collies, 'unitAmount' ) ),
			'ref1'                => 'REF' . $order_id,
			'ref2'                => 'ORD' . $order->get_order_number(),
			'ref3'                => 'INV' . $order_id,
			'refInt'              => 'INT' . $order_id,
			'exWorksType'         => 1,
			'cargo'               => true,
			'returnDocument'      => true,
			'remark'              => 'Handle with care',
			'codValue'            => $cod_value,
			'codCurrency'         => 0,
			'collies'             => $collies,
			'noSticker'           => false,
		];

		$api_url  = 'https://api.overseas.hr/createshipment?apikey=' . rawurlencode( $api_key );
		$response = wp_remote_post(
			$api_url,
			[
				'headers' => [ 'Content-Type' => 'application/json' ],
				'body'    => wp_json_encode( $body ),
				'timeout' => 30,
			]
		);

		if ( is_wp_error( $response ) ) {
			overseas_express_admin_error_redirect( $order_id, 'Shipment API error: ' . $response->get_error_message() );
		}

		$status_code   = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );
		$response_data = json_decode( $response_body, true );

		if ( $status_code >= 200 && $status_code < 300 && ! empty( $response_data['shipmentid'] ) ) {
			update_post_meta( $order_id, '_overseas_shipment_id', sanitize_text_field( (string) $response_data['shipmentid'] ) );

			if ( ! empty( $response_data['CargoIDs'] ) ) {
				$cargo_ids = is_array( $response_data['CargoIDs'] ) ? $response_data['CargoIDs'] : [ $response_data['CargoIDs'] ];
				$cargo_ids = array_map(
					static function( $v ) {
						return sanitize_text_field( (string) $v );
					},
					$cargo_ids
				);
				update_post_meta( $order_id, '_overseas_cargo_ids', $cargo_ids );
			}

			$order->update_status(
				'completed',
				esc_html__( 'Shipment created successfully and order marked as completed.', 'overseas-express-shipment' )
			);
            $redirect_url = add_query_arg(
                array(
                    'post'             => (int) $order_id,
                    'action'           => 'edit',
                    'shipment_created' => 1,
                    '_wpnonce'         => wp_create_nonce( 'oes_shipment_notice' ),
                ),
                admin_url( 'post.php' )
            );

            wp_safe_redirect( $redirect_url );
            exit;
		} else {
			$error_msg = ! empty( $response_data['error']['message'] ?? '' ) ? (string) $response_data['error']['message'] : (string) $response_body;

			$redirect_url = add_query_arg(
                array(
                    'post'          => (int) $order_id,
                    'action'        => 'edit',
                    'shipment_error'=> rawurlencode( (string) $error_msg ), // or pass unslashed string; add_query_arg will encode
                    '_wpnonce'      => wp_create_nonce( 'oes_shipment_notice' ),
                ),
                admin_url( 'post.php' )
            );

            wp_safe_redirect( $redirect_url );
            exit;
		}

		exit;
	}

	add_action( 'admin_post_update_overseas_shipment', 'overseas_express_handle_update_shipment' );
	function overseas_express_handle_update_shipment() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'Unauthorized', 'overseas-express-shipment' ) );
		}

		$order_id = isset( $_GET['order_id'] ) ? absint( $_GET['order_id'] ) : 0;
		if ( ! $order_id ) {
			wp_die( esc_html__( 'Invalid order ID', 'overseas-express-shipment' ) );
		}

		$nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'overseas_express_update_shipment_' . $order_id ) ) {
			wp_die( esc_html__( 'Security check failed.', 'overseas-express-shipment' ) );
		}

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			wp_die( esc_html__( 'Order not found', 'overseas-express-shipment' ) );
		}

		$api_key     = overseas_express_get_api_key();
		$sender_data = overseas_express_get_sender_data();
		$shipment_id = get_post_meta( $order_id, '_overseas_shipment_id', true );

		if ( ! $shipment_id ) {
			wp_die( esc_html__( 'Shipment not found for this order', 'overseas-express-shipment' ) );
		}

		$cod_value = 0;
		if ( 'cod' === $order->get_payment_method() ) {
			$cod_value = (float) $order->get_total();
		}

		$collies = [];
		$i       = 1;
		foreach ( $order->get_items() as $item_id => $item ) {
			$product      = $item->get_product();
			$quantity     = $item->get_quantity();
			$product_name = $item->get_name();
			$weight       = $product ? (float) $product->get_weight() : 0;
			$sku          = $product ? $product->get_sku() : '';
			$ref_prefix   = 'COLLIE' . $i . '_';

			$collies[] = [
				'unitAmount' => $weight > 0 ? $weight * $quantity : 0,
				'ref1'       => $ref_prefix . 'REF',
				'ref2'       => $ref_prefix . 'ORD' . $order_id,
				'ref3'       => $ref_prefix . 'INV' . $order_id,
				'refInt'     => $ref_prefix . 'INT' . $order_id,
				'remark'     => $product_name,
				'barcode'    => $sku ? $sku : 'ORDER' . $order_id . '_ITEM' . $item_id,
			];
			$i++;
		}

		$body = [
			'cosignee'            => [
				'name'            => trim( $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name() ),
				'countryCode'     => $order->get_shipping_country(),
				'zipcode'         => $order->get_shipping_postcode(),
				'city'            => $order->get_shipping_city(),
				'streetAndNumber' => $order->get_shipping_address_1(),
				'telephone'       => $order->get_billing_phone(),
				'fax'             => '',
				'notifyGSM'       => $order->get_billing_phone(),
				'notifyEmail'     => $order->get_billing_email(),
			],
			'sender'              => $sender_data,
			'cosigneeNotifyType'  => 'Email',
			'isSenderNonCustomer' => false,
			'unitAmount'          => array_sum( wp_list_pluck( $collies, 'unitAmount' ) ),
			'ref1'                => 'REF' . $order_id,
			'ref2'                => 'ORD' . $order->get_order_number(),
			'ref3'                => 'INV' . $order_id,
			'refInt'              => 'INT' . $order_id,
			'exWorksType'         => 1,
			'cargo'               => true,
			'returnDocument'      => true,
			'remark'              => esc_html__( 'Handle with care', 'overseas-express-shipment' ),
			'codValue'            => $cod_value,
			'codCurrency'         => 0,
			'collies'             => $collies,
			'noSticker'           => false,
		];

		$api_url = 'https://api.overseas.hr/updateshipment?shipmentId=' . rawurlencode( (string) $shipment_id ) . '&apikey=' . rawurlencode( $api_key );

		$response = wp_remote_post(
			$api_url,
			[
				'headers' => [ 'Content-Type' => 'application/json' ],
				'body'    => wp_json_encode( $body ),
				'timeout' => 30,
			]
		);

		if ( is_wp_error( $response ) ) {
			wp_safe_redirect(
				admin_url(
					'post.php?post=' . $order_id . '&action=edit&shipment_error=' . rawurlencode( $response->get_error_message() )
				)
			);
			exit;
		}

		$status_code   = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );

		if ( $status_code >= 200 && $status_code < 300 ) {
            $redirect_url = add_query_arg(
                array(
                    'post'             => (int) $order_id,
                    'action'           => 'edit',
                    'shipment_updated' => 1,
                    '_wpnonce'         => wp_create_nonce( 'oes_shipment_notice' ),
                ),
                admin_url( 'post.php' )
            );

            wp_safe_redirect( $redirect_url );
            exit;
		} else {
            $redirect_url = add_query_arg(
                array(
                    'post'          => (int) $order_id,
                    'action'        => 'edit',
                    'shipment_error'=> (string) $response_body,
                    '_wpnonce'      => wp_create_nonce( 'oes_shipment_notice' ),
                ),
                admin_url( 'post.php' )
            );

            wp_safe_redirect( $redirect_url );
            exit;
		}

		exit;
	}

	add_action( 'admin_post_print_overseas_label', 'overseas_express_handle_print_label' );
	function overseas_express_handle_print_label() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'Unauthorized', 'overseas-express-shipment' ), 'Error', [ 'response' => 403 ] );
		}

		$order_id = isset( $_GET['order_id'] ) ? absint( $_GET['order_id'] ) : 0;
		if ( ! $order_id ) {
			wp_die( esc_html__( 'Invalid order ID', 'overseas-express-shipment' ) );
		}

		$nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'overseas_express_print_label_' . $order_id ) ) {
			wp_die( esc_html__( 'Security check failed.', 'overseas-express-shipment' ) );
		}

		$shipment_id = get_post_meta( $order_id, '_overseas_shipment_id', true );
		if ( ! $shipment_id ) {
			wp_die( esc_html__( 'No shipment found for this order', 'overseas-express-shipment' ) );
		}

		$api_key = overseas_express_get_api_key();
		if ( empty( $api_key ) ) {
			wp_die( esc_html__( 'API Key is missing. Please configure it in Integration Shipment settings.', 'overseas-express-shipment' ) );
		}

		$label_layout_id = overseas_express_get_label_layout_id();
		if ( empty( $label_layout_id ) ) {
			wp_die( esc_html__( 'Label Layout ID is missing. Please select a label layout in Integration Shipment settings.', 'overseas-express-shipment' ) );
		}

		$api_url = 'https://api.overseas.hr/pdfinternal?layoutid=' . rawurlencode( $label_layout_id ) . '&apikey=' . rawurlencode( $api_key );
		$body    = wp_json_encode( [ (int) $shipment_id ] );

		$response = wp_remote_post(
			$api_url,
			[
				'headers' => [ 'Content-Type' => 'application/json' ],
				'body'    => $body,
				'timeout' => 30,
			]
		);

		if ( is_wp_error( $response ) ) {
			wp_die( esc_html__( 'API request failed: ', 'overseas-express-shipment' ) . esc_html( $response->get_error_message() ) );
		}

		$response_body = wp_remote_retrieve_body( $response );
		$data          = json_decode( $response_body, true );

		if ( empty( $data['labelsbase64'] ) ) {
			// Allow <br><pre> while still escaping the API body.
			$msg = esc_html__( 'No PDF data received from API.', 'overseas-express-shipment' ) . '<br><pre>' . esc_html( (string) $response_body ) . '</pre>';
			wp_die( wp_kses_post( $msg ) );
		}

		$decoded_pdf = base64_decode( (string) $data['labelsbase64'] );
		if ( ! $decoded_pdf ) {
			wp_die( esc_html__( 'Failed to decode PDF data.', 'overseas-express-shipment' ) );
		}

		header( 'Content-Type: application/pdf' );
		header( 'Content-Disposition: attachment; filename="shipment-label-' . (int) $shipment_id . '.pdf"' );
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Binary data must be output raw.
		echo $decoded_pdf;
		exit;
	}

	/**
	 * Orders list action: Print Label
	 */
	add_filter(
		'woocommerce_admin_order_actions',
		function( $actions, $order ) {
			$shipment_id = get_post_meta( $order->get_id(), '_overseas_shipment_id', true );

			if ( $shipment_id ) {
				$url = wp_nonce_url(
					admin_url( 'admin-post.php?action=print_overseas_label&order_id=' . (int) $order->get_id() ),
					'overseas_express_print_label_' . (int) $order->get_id(),
					'_wpnonce'
				);

				$actions['print_overseas_label'] = [
					'url'    => $url,
					'name'   => esc_html__( 'Print Label', 'overseas-express-shipment' ),
					'action' => 'download_label',
					'target' => '_blank',
				];
			}

			return $actions;
		},
		10,
		2
	);

	/**
	 * Bulk labels
	 */
	add_filter( 'bulk_actions-woocommerce_page_wc-orders', 'overseas_express_bulk_labels_bulk_actions' );
	function overseas_express_bulk_labels_bulk_actions( $bulk_actions ) {
		$bulk_actions['print_bulk_labels'] = esc_html__( 'Print Labels', 'overseas-express-shipment' );
		return $bulk_actions;
	}

	add_filter(
		'handle_bulk_actions-woocommerce_page_wc-orders',
		function( $redirect_to, $action, $post_ids ) {
			if ( 'print_bulk_labels' !== $action ) {
				return $redirect_to;
			}

			$shipment_ids   = [];
			$failed_orders  = [];
			$success_orders = [];

			foreach ( $post_ids as $order_id ) {
				$order_id    = (int) $order_id;
				$shipment_id = get_post_meta( $order_id, '_overseas_shipment_id', true );

				if ( ! $shipment_id ) {
					$failed_orders[] = $order_id;
					continue;
				}

				$shipment_ids[]   = (int) $shipment_id;
				$success_orders[] = $order_id;
			}

			$result = [
				'success' => $success_orders,
				'failed'  => $failed_orders,
				'pdf_url' => '',
			];

			if ( ! empty( $shipment_ids ) ) {
				$api_key = overseas_express_get_api_key();
				if ( empty( $api_key ) ) {
					wp_die( esc_html__( 'API Key is missing. Please configure it in Integration Shipment settings.', 'overseas-express-shipment' ) );
				}

				$label_layout_id = overseas_express_get_label_layout_id();
				if ( empty( $label_layout_id ) ) {
					wp_die( esc_html__( 'Label Layout ID is missing. Please select a label layout in Integration Shipment settings.', 'overseas-express-shipment' ) );
				}

				$api_url = 'https://api.overseas.hr/pdfinternal?layoutid=' . rawurlencode( $label_layout_id ) . '&apikey=' . rawurlencode( $api_key );
				$body    = wp_json_encode( $shipment_ids );

				$response = wp_remote_post(
					$api_url,
					[
						'headers' => [ 'Content-Type' => 'application/json' ],
						'body'    => $body,
						'timeout' => 30,
					]
				);

				if ( ! is_wp_error( $response ) ) {
					$data = json_decode( wp_remote_retrieve_body( $response ), true );

					if ( ! empty( $data['labelsbase64'] ) ) {
						$pdf_binary = base64_decode( (string) $data['labelsbase64'] );
						if ( $pdf_binary ) {
							$upload_dir = wp_upload_dir();
							$file_name  = 'shipment-labels-' . time() . '-' . wp_rand( 1000, 9999 ) . '.pdf';
							$file_path  = trailingslashit( $upload_dir['basedir'] ) . $file_name;

							$fs = overseas_express_get_filesystem();
							if ( $fs ) {
								$fs->put_contents( $file_path, $pdf_binary, FS_CHMOD_FILE );
								$result['pdf_url'] = admin_url(
									'admin-post.php?action=download_bulk_labels&file=' . rawurlencode( $file_name ) .
									'&_wpnonce=' . wp_create_nonce( 'download_bulk_labels' )
								);
							}
						}
					}
				}
			}

			set_transient( 'bulk_labels_result_' . get_current_user_id(), $result, 60 );
			return wp_get_referer();
		},
		10,
		3
	);

	add_action(
		'admin_notices',
		function() {
			$user_id = get_current_user_id();
			$result  = get_transient( 'bulk_labels_result_' . $user_id );

			if ( ! $result ) {
				return;
			}

			delete_transient( 'bulk_labels_result_' . $user_id );

			if ( ! empty( $result['success'] ) ) {
				$msg = esc_html__( 'Labels created for orders: ', 'overseas-express-shipment' ) . implode( ', ', array_map( 'intval', $result['success'] ) );
				if ( ! empty( $result['pdf_url'] ) ) {
					$msg .= ' — <a href="' . esc_url( $result['pdf_url'] ) . '" target="_blank" rel="noopener noreferrer">' . esc_html__( 'Download PDF', 'overseas-express-shipment' ) . '</a>';
				}
				echo '<div class="notice notice-success"><p>' . wp_kses_post( $msg ) . '</p></div>';
			}

			if ( ! empty( $result['failed'] ) ) {
				echo '<div class="notice notice-error"><p>' .
					esc_html__( 'Failed to create labels for orders: ', 'overseas-express-shipment' ) .
					esc_html( implode( ', ', array_map( 'intval', $result['failed'] ) ) ) .
					'</p></div>';
			}
		}
	);

	add_action(
		'admin_post_download_bulk_labels',
		function() {
			if ( ! current_user_can( 'manage_woocommerce' ) ) {
				wp_die( esc_html__( 'Unauthorized access.', 'overseas-express-shipment' ) );
			}

			$nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
			if ( ! $nonce || ! wp_verify_nonce( $nonce, 'download_bulk_labels' ) ) {
				wp_die( esc_html__( 'Security check failed.', 'overseas-express-shipment' ) );
			}

			if ( empty( $_GET['file'] ) ) {
				wp_die( esc_html__( 'No file specified.', 'overseas-express-shipment' ) );
			}

			$file_name = sanitize_file_name( wp_unslash( $_GET['file'] ) );
			if ( $file_name !== wp_basename( $file_name ) ) {
				wp_die( esc_html__( 'Invalid file name.', 'overseas-express-shipment' ) );
			}

			$upload_dir = wp_upload_dir();
			$file_path  = trailingslashit( $upload_dir['basedir'] ) . $file_name;

			$fs = overseas_express_get_filesystem();
			if ( ! $fs ) {
				wp_die( esc_html__( 'Filesystem not available.', 'overseas-express-shipment' ) );
			}

			if ( ! $fs->exists( $file_path ) ) {
				wp_die( esc_html__( 'File not found.', 'overseas-express-shipment' ) );
			}

			$pdf_data = $fs->get_contents( $file_path );
			if ( false === $pdf_data || '' === $pdf_data ) {
				wp_die( esc_html__( 'Failed to read file.', 'overseas-express-shipment' ) );
			}

			$safe_filename = sanitize_file_name( basename( $file_path ) );

			header( 'Content-Description: File Transfer' );
			header( 'Content-Type: application/pdf' );
			header( 'Content-Disposition: attachment; filename="' . $safe_filename . '"' );
			header( 'Content-Length: ' . strlen( $pdf_data ) );

			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Binary PDF output.
			echo $pdf_data;

			$fs->delete( $file_path );
			exit;
		}
	);

	/**
	 * Admin notices
	 */
    add_action(
        'admin_notices',
        function () {

            // Only users who can manage WooCommerce should see these.
            if ( ! current_user_can( 'manage_woocommerce' ) ) {
                return;
            }

            // Only show on order edit screen (prevents notices anywhere else).
            $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
            if ( ! $screen || 'post' !== $screen->base || 'shop_order' !== $screen->post_type ) {
                return;
            }

            // Verify notice nonce BEFORE reading flags.
            $nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
            if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'oes_shipment_notice' ) ) {
                return;
            }

            if ( isset( $_GET['shipment_created'] ) ) {
                echo '<div class="notice notice-success is-dismissible"><p>✅ ' .
                    esc_html__( 'Integration Shipment created successfully!', 'overseas-express-shipment' ) .
                '</p></div>';
                return;
            }

            if ( isset( $_GET['shipment_updated'] ) ) {
                echo '<div class="notice notice-success is-dismissible"><p>✅ ' .
                    esc_html__( 'Integration Shipment updated successfully!', 'overseas-express-shipment' ) .
                '</p></div>';
                return;
            }

            if ( isset( $_GET['shipment_error'] ) ) {
                $shipment_error = sanitize_text_field( wp_unslash( $_GET['shipment_error'] ) );
                echo '<div class="notice notice-error is-dismissible"><p>❌ ' .
                    esc_html__( 'Shipment error: ', 'overseas-express-shipment' ) .
                    esc_html( $shipment_error ) .
                '</p></div>';
                return;
            }
        }
    );
}

/**
 * Shipping Methods: Flat/Home Delivery
 */
if ( ! class_exists( 'OverseasExpress_Flat_Shipping_Options' ) ) {

	function overseas_express_flat_shipping_methods_init() {

		class OverseasExpress_Flat_Shipping_Options extends WC_Shipping_Method {

			public function __construct() {
				$this->id                 = 'overseas_express_flat_shipping';
				$this->method_title       = __( 'Home Delivery Shipment', 'overseas-express-shipment' );
				$this->title              = __( 'Home Delivery Shipment', 'overseas-express-shipment' );
				$this->method_description = __( 'Home Delivery Shipment Description', 'overseas-express-shipment' );

				add_action( 'woocommerce_update_options_shipping_' . $this->id, [ $this, 'process_admin_options' ] );
				$this->init();
			}

			public function init() {
				$this->init_form_fields();
				$this->init_settings();

				$this->flatEnabled      = $this->get_option( 'flatEnabled' );
				$this->flatTitle        = __( 'Home Delivery Shipment', 'overseas-express-shipment' );
				$this->flatType         = 'fixed';
				$this->flatFee          = $this->get_option( 'price' );
				$this->flatCodes        = $this->get_option( 'flatCodes' );
				$this->flatAvailability = $this->get_option( 'flatAvailability' );
				$this->flatCountries    = $this->get_option( 'flatCountries' );
				$this->freeFlatShipment = $this->get_option( 'freeFlatShipment' );

				add_filter( 'woocommerce_shipping_methods', [ $this, 'add_overseas_express_flat_shipping_methods' ] );
				if ( ! is_admin() ) {
					add_action( 'woocommerce_cart_shipping_method_full_label', [ $this, 'negative_flat_label' ], 10, 2 );
				}
			}

			public function calculate_shipping( $package = [] ) {
				$shipping_total = 0;
				$cart_total     = WC()->cart->cart_contents_total;
				$flatFee        = ( trim( (string) $this->flatFee ) === '' ) ? 0 : (float) $this->flatFee;

				if ( $cart_total >= (float) $this->freeFlatShipment ) {
					$shipping_total = 0;
				} else {
					if ( 'fixed' === $this->flatType ) {
						$shipping_total = $flatFee;
					}

					if ( 'percent' === $this->flatType ) {
						$shipping_total = $package['contents_cost'] * ( $flatFee / 100 );
					}

					if ( 'product' === $this->flatType ) {
						foreach ( $package['contents'] as $item_id => $values ) {
							$_product = $values['data'];

							if ( $values['quantity'] > 0 && $_product->needs_shipping() ) {
								$shipping_total += $flatFee * $values['quantity'];
							}
						}
					}
				}

				$shipping_total_with_tax = $shipping_total;
				$tax_total               = 0;
				$label_suffix            = $this->flatTitle;
				$taxes                   = [];

				if ( wc_tax_enabled() ) {
					$tax_rates = WC_Tax::get_shipping_tax_rates();
					$taxes     = WC_Tax::calc_shipping_tax( $shipping_total, $tax_rates );
					$tax_total = array_sum( $taxes );
					$shipping_total_with_tax += $tax_total;

					$suffix_template = get_option( 'woocommerce_price_display_suffix', '' );

					$price_excl_tax = wc_price( $shipping_total );
					$price_incl_tax = wc_price( $shipping_total_with_tax );
					$tax_amount     = wc_price( $tax_total );

					$search         = [ '{tax_amount}', '{price_including_tax}', '{price_excluding_tax}' ];
					$replace        = [ $tax_amount, $price_incl_tax, $price_excl_tax ];
					$dynamic_suffix = str_replace( $search, $replace, $suffix_template );

					if ( ! empty( $dynamic_suffix ) ) {
						$label_suffix .= ' (' . $dynamic_suffix . ')';
					}
				}

				$rate = [
					'id'    => $this->id,
					'label' => $label_suffix,
					'cost'  => $shipping_total,
					'taxes' => $taxes,
				];

				$this->add_rate( $rate );
			}

			public function init_form_fields() {
				$this->form_fields = [
					'flatEnabled'      => [
						'title'   => esc_html__( 'Enable', 'overseas-express-shipment' ),
						'type'    => 'checkbox',
						'label'   => esc_html__( 'Enable Home Delivery', 'overseas-express-shipment' ),
						'default' => 'no',
					],
					'flatCodes'        => [
						'title'       => esc_html__( 'Zip/Post Codes', 'overseas-express-shipment' ),
						'type'        => 'textarea',
						'description' => esc_html__( 'What zip/post codes can use the local pickup option? Separate codes with a comma. Accepts wildcards, e.g. P* will match a postcode of PE30.', 'overseas-express-shipment' ),
						'default'     => '',
						'desc_tip'    => true,
						'placeholder' => '12345, 56789, etc',
					],
					'price'            => [
						'title'       => esc_html__( 'Home Delivery Fee/Discount', 'overseas-express-shipment' ),
						'type'        => 'price',
						'description' => esc_html__( 'What fee do you want to charge for local pickup? Enter negative value for discount. Leave blank to disable.', 'overseas-express-shipment' ),
						'default'     => '',
						'desc_tip'    => true,
						'placeholder' => '0',
					],
					'flatAvailability' => [
						'title'   => esc_html__( 'Method availability', 'overseas-express-shipment' ),
						'type'    => 'select',
						'default' => 'all',
						'class'   => 'availability',
						'options' => [
							'all'      => esc_html__( 'All allowed countries', 'overseas-express-shipment' ),
							'specific' => esc_html__( 'Specific Countries', 'overseas-express-shipment' ),
						],
					],
					'flatCountries'    => [
						'title'             => esc_html__( 'Specific Countries', 'overseas-express-shipment' ),
						'type'              => 'multiselect',
						'class'             => 'chosen_select',
						'css'               => 'width: 450px;',
						'default'           => '',
						'options'           => function_exists( 'WC' ) && WC()->countries ? WC()->countries->get_shipping_countries() : [],
						'custom_attributes' => [
							'data-placeholder' => esc_html__( 'Select some countries', 'overseas-express-shipment' ),
						],
					],
					'freeFlatShipment' => [
						'title'       => esc_html__( 'Custom Free Shipment', 'overseas-express-shipment' ),
						'type'        => 'price',
						'description' => esc_html__( 'Enter the users price. For example if it is greater than $50, shipping will be free.', 'overseas-express-shipment' ),
						'default'     => '',
						'desc_tip'    => true,
					],
				];
			}

			public function admin_options() {
				?>
				<h3><?php echo esc_html( $this->method_title ); ?></h3>
				<p><?php esc_html_e( 'Home Delivery Shipment Description', 'overseas-express-shipment' ); ?></p>
				<table class="form-table">
					<?php $this->generate_settings_html(); ?>
				</table>
				<?php
			}

			public function is_available( $package ) {
				$flat_enabled = ( 'no' === $this->flatEnabled );
				if ( $flat_enabled ) {
					return false;
				}

				$flatCodes = [];
				if ( ! empty( $this->flatCodes ) ) {
					foreach ( explode( ',', $this->flatCodes ) as $code ) {
						$flatCodes[] = $this->clean( $code );
					}
				}

				if ( is_array( $flatCodes ) && ! empty( $flatCodes ) ) {
					$found_match = false;

					if ( in_array( $this->clean( $package['destination']['postcode'] ), $flatCodes, true ) ) {
						$found_match = true;
					}

					if ( ! $found_match ) {
						$customer_postcode = $this->clean( $package['destination']['postcode'] );
						foreach ( $flatCodes as $c ) {
							$pattern = '/^' . str_replace( '_', '[0-9a-zA-Z]', preg_quote( $c, '/' ) ) . '$/i';
							if ( preg_match( $pattern, $customer_postcode ) ) {
								$found_match = true;
								break;
							}
						}
					}

					if ( ! $found_match ) {
						$customer_postcode        = $this->clean( $package['destination']['postcode'] );
						$customer_postcode_length = strlen( $customer_postcode );

						for ( $i = 0; $i <= $customer_postcode_length; $i++ ) {
							if ( in_array( $customer_postcode, $flatCodes, true ) ) {
								$found_match = true;
							}
							$customer_postcode = substr( $customer_postcode, 0, -2 ) . '*';
						}
					}

					if ( ! $found_match ) {
						return false;
					}
				}

				if ( 'specific' === $this->flatAvailability ) {
					$ship_to_countries = $this->flatCountries;
				} else {
					$ship_to_countries = function_exists( 'WC' ) && WC()->countries ? array_keys( WC()->countries->get_shipping_countries() ) : [];
				}

				if ( is_array( $ship_to_countries ) && ! empty( $ship_to_countries ) ) {
					if ( ! in_array( $package['destination']['country'], $ship_to_countries, true ) ) {
						return false;
					}
				}

				// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
				return apply_filters( 'woocommerce_shipping_' . $this->id . '_is_available', true, $package );
			}

			public function clean( $code ) {
				return str_replace( '-', '', sanitize_title( $code ) ) . ( strstr( $code, '*' ) ? '*' : '' );
			}

			public function add_overseas_express_flat_shipping_methods( $methods ) {
				$methods[] = $this;
				return $methods;
			}

			public function negative_flat_label( $label, $method ) {
				if ( ! is_admin() && 'overseas_express_flat_shipping' === $method->id ) {
					$label = $method->label . ' : ' . wc_price( $method->cost );
				}
				return $label;
			}
		}

		new OverseasExpress_Flat_Shipping_Options();
	}

	add_action( 'woocommerce_shipping_init', 'overseas_express_flat_shipping_methods_init' );
}