(function ($) {
  'use strict';

  function log()  { try { console.log.apply(console, ['[OE MAP]'].concat([].slice.call(arguments))); } catch (e) {} }
  function warn() { try { console.warn.apply(console, ['[OE MAP]'].concat([].slice.call(arguments))); } catch (e) {} }
  function error(){ try { console.error.apply(console, ['[OE MAP]'].concat([].slice.call(arguments))); } catch (e) {} }

  var leafletMap = null;
  var leafletMarkersLayer = null;
  var isMapOpen = false;
  var lastSelectedMethod = null;

  // Prevent auto-opening map during order submit / rerenders
  var isSubmittingOrder = false;

  // Only allow one auto-open on initial load when pickup is selected and no point
  var didInitialAutoOpen = false;

  // Cache selected point so DOM rerenders don't make it look empty
  var cachedSelectedPointName = '';

  /**
   * Escape HTML to prevent XSS
   */
  function escapeHtml(text) {
    var map = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#039;'
    };
    return String(text || '').replace(/[&<>"']/g, function(m) { return map[m]; });
  }

  function getCfg() {
    if (typeof overseasExpressPickup === 'undefined') return null;
    return overseasExpressPickup;
  }

  function getAjaxUrl() {
    var cfg = getCfg();
    if (cfg && cfg.ajaxUrl) return cfg.ajaxUrl;
    if (window.ajaxurl) return window.ajaxurl;
    return '/wp-admin/admin-ajax.php';
  }

  // ----- Pickup notice UI -----
  function ensurePickupNotice() {
    if ($('#oe-pickup-notice').length) return;

    var css =
      '<style id="oe-pickup-notice-css">' +
      '#oe-pickup-notice{margin:12px 0;padding:10px 12px;border:1px solid #c3e6cb;background:#d4edda;color:#155724;border-radius:6px;display:none;font-size:13px;}' +
      '</style>';

    if (!$('#oe-pickup-notice-css').length) $('head').append(css);

    // Place it in a sensible spot for classic OR blocks checkout
    var $target = $('.woocommerce-shipping-methods').first();
    if (!$target.length) $target = $('.wc-block-components-shipping-rates-control').first();
    if (!$target.length) $target = $('.woocommerce-notices-wrapper').first();
    if (!$target.length) $target = $('form.checkout').first();
    if (!$target.length) $target = $('body');

    $('<div id="oe-pickup-notice" role="status" aria-live="polite"></div>').insertBefore($target);
  }

  function buildPickupNoticeText(point) {
    var name = point && point.name ? point.name : '';
    var street = point && point.street ? point.street : '';
    var zip = point && point.zip ? point.zip : '';
    var city = point && point.city ? point.city : '';

    var parts = [];
    if (name) parts.push(name);
    if (street) parts.push(street);

    var cityLine = (zip || city) ? (zip + (zip && city ? ' ' : '') + city) : '';
    if (cityLine) parts.push(cityLine);

    return 'Pickup Point Selected: ' + parts.join(' - ');
  }

  function showPickupNotice(point) {
    ensurePickupNotice();
    var text = buildPickupNoticeText(point);
    $('#oe-pickup-notice').text(text).stop(true, true).fadeIn(150);
  }

  function hidePickupNotice() {
    $('#oe-pickup-notice').stop(true, true).fadeOut(150);
  }
  // ---------------------------

  function getSelectedShippingMethod() {
    // Classic checkout
    var $checked = $('input[name^="shipping_method"]:checked');
    if ($checked.length) return $checked.val();

    $checked = $('input[type="radio"][name*="shipping"]:checked');
    if ($checked.length) return $checked.val();

    var $hidden = $('input[type="hidden"][name^="shipping_method"]');
    if ($hidden.length) return $hidden.first().val();

    // Blocks
    var $blockChecked = $('.wc-block-components-radio-control__option input:checked');
    if ($blockChecked.length) return $blockChecked.val();

    var $selectedOpt = $('.wc-block-components-radio-control__option.is-selected input');
    if ($selectedOpt.length) return $selectedOpt.val();

    return null;
  }

  function isPickupShippingMethod(methodValue) {
    var cfg = getCfg();
    if (!cfg) return false;
    var methodId = cfg.shippingMethodId || 'overseas_express_pickup_shipping';
    return methodValue && methodValue.indexOf(methodId) !== -1;
  }

  function hasSelectedPickupPoint() {
    var $field = $('#overseas_express_selected_point');
    if ($field.length) {
      var v = ($field.val() || '').trim();
      if (v) {
        cachedSelectedPointName = v;
        return true;
      }
      return false;
    }
    return !!(cachedSelectedPointName && cachedSelectedPointName.trim());
  }

  function clearSelectedPickupPoint() {
    cachedSelectedPointName = '';
    $('#overseas_express_selected_point').val('');
    $('#oe_address').val('');
    $('#oe_postcode').val('');
    $('#oe_city').val('');
  }

  function ensurePickupOverlay() {
    if ($('#oe-pickup-overlay').length) return;

    var css =
      '<style id="oe-pickup-overlay-css">' +
      '#oe-pickup-overlay{position:fixed;inset:0;z-index:99999;display:none;}' +
      '#oe-pickup-backdrop{position:absolute;inset:0;background:rgba(0,0,0,.55);cursor:pointer;}' +
      '#oe-pickup-modal{position:relative;max-width:1000px;width:92vw;max-height:85vh;margin:5vh auto;background:#fff;border-radius:12px;box-shadow:0 10px 35px rgba(0,0,0,.35);overflow:hidden;display:flex;flex-direction:column;}' +
      '#oe-pickup-header{display:flex;align-items:center;justify-content:space-between;padding:12px 14px;border-bottom:1px solid #eee;background:#f9f9f9;}' +
      '#oe-pickup-title{margin:0;font-size:16px;line-height:1.2;font-weight:600;}' +
      '#oe-close-map{border:0;background:transparent;font-size:26px;line-height:1;cursor:pointer;padding:0 6px;color:#666;transition:color .2s;}' +
      '#oe-close-map:hover{color:#000;}' +
      '#oe-pickup-body{padding:12px;flex:1;overflow:auto;}' +
      '</style>';

    if (!$('#oe-pickup-overlay-css').length) $('head').append(css);

    var html =
      '<div id="oe-pickup-overlay" aria-hidden="true">' +
        '<div id="oe-pickup-backdrop"></div>' +
        '<div id="oe-pickup-modal" role="dialog" aria-modal="true">' +
          '<div id="oe-pickup-header">' +
            '<h3 id="oe-pickup-title">Select a pickup point</h3>' +
            '<button type="button" id="oe-close-map" aria-label="Close">×</button>' +
          '</div>' +
          '<div id="oe-pickup-body">' +
            '<div id="overseas-express-map" style="width:100%;height:420px;"></div>' +
          '</div>' +
        '</div>' +
      '</div>';

    $('body').append(html);

    $('#oe-close-map, #oe-pickup-backdrop').on('click', function () {
      hideOverlay();
    });

    $(document).on('keydown.oePickup', function (e) {
      if (e.key === 'Escape' && isMapOpen) hideOverlay();
    });
  }

  function showOverlay() {
    ensurePickupOverlay();
    $('#oe-pickup-overlay').fadeIn(200).attr('aria-hidden', 'false');
    $('body').css('overflow', 'hidden');
    isMapOpen = true;
    log('Modal opened');
  }

  function hideOverlay() {
    $('#oe-pickup-overlay').fadeOut(200).attr('aria-hidden', 'true');
    $('body').css('overflow', '');
    isMapOpen = false;
    log('Modal closed');
  }

  // Optional: update classic checkout visible fields (cosmetic)
  function setClassicShippingFields(point) {
    if (!$('#shipping_address_1').length) return; // blocks checkout often won't have these
    $('#shipping_address_1').val(point.street || '').trigger('change');
    $('#shipping_postcode').val(point.zip || '').trigger('change');
    $('#shipping_city').val(point.city || '').trigger('change');
    $('#shipping_country').val('HR').trigger('change'); // adjust if dynamic
  }

  function savePickupToSession(point) {
    var cfg = getCfg();
    return $.post(getAjaxUrl(), {
      action: 'overseas_express_set_pickup',
      nonce: cfg && cfg.nonce ? cfg.nonce : '', // Added nonce for security
      point: point.name || '',
      address: point.street || '',
      postcode: point.zip || '',
      city: point.city || '',
      country: 'HR'
    });
  }

  function triggerCheckoutUpdate() {
    $(document.body).trigger('update_checkout');
    $(document.body).trigger('wc_fragment_refresh');
  }

  function initMap() {
    var cfg = getCfg();
    if (!cfg) { error('overseasExpressPickup config missing'); return; }
    if (!window.L) { error('Leaflet not loaded'); return; }

    var points = cfg.points || [];
    var icons  = cfg.icons || {};
    var mapId  = cfg.mapContainerId || 'overseas-express-map';

    var $mapDiv = $('#' + mapId);
    if (!$mapDiv.length) { error('Map container not found:', mapId); return; }

    if (!points.length) {
      warn('No pickup points available');
      $mapDiv.html('<div style="padding:20px;text-align:center;color:#666;">No pickup points available at this time.</div>');
      return;
    }

    // If map already exists, just refresh
    if (leafletMap) {
      setTimeout(function () { try { leafletMap.invalidateSize(); } catch (e) {} }, 250);
      return;
    }

    var blueIcon = new L.Icon({
      iconUrl: icons.blue || '',
      shadowUrl: icons.shadow || '',
      iconSize: [25, 41],
      iconAnchor: [12, 41],
      popupAnchor: [1, -34],
      shadowSize: [41, 41]
    });

    var redIcon = new L.Icon({
      iconUrl: icons.red || '',
      shadowUrl: icons.shadow || '',
      iconSize: [25, 41],
      iconAnchor: [12, 41],
      popupAnchor: [1, -34],
      shadowSize: [41, 41]
    });

    var firstPoint = points[0];

    leafletMap = L.map(mapId, {
      center: [firstPoint.lat, firstPoint.lng],
      zoom: 7,
      scrollWheelZoom: true
    });

    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
      maxZoom: 19,
      attribution: '© OpenStreetMap contributors'
    }).addTo(leafletMap);

    leafletMarkersLayer = L.layerGroup().addTo(leafletMap);

    points.forEach(function (p, index) {
      if (!p.lat || !p.lng) return;

      var icon = (parseInt(p.OOHType, 10) === 1) ? redIcon : blueIcon;
      var marker = L.marker([p.lat, p.lng], { icon: icon }).addTo(leafletMarkersLayer);

      // Escape all user-provided data for security (defense in depth)
      var popupContent =
        '<div style="min-width:200px;">' +
          '<strong style="font-size:14px;">' + escapeHtml(p.name || 'Pickup Point') + '</strong><br>' +
          '<small style="color:#666;">' +
            escapeHtml(p.street || '') + '<br>' +
            escapeHtml(p.zip || '') + ' ' + escapeHtml(p.city || '') +
          '</small><br>' +
          '<button class="oe-select-point" data-index="' + parseInt(index, 10)  + '" style="margin-top:8px;padding:6px 12px;background:#0073aa;color:#fff;border:0;border-radius:4px;cursor:pointer;font-size:13px;">Select this point</button>' +
        '</div>';

      marker.bindPopup(popupContent);
      marker.on('click', function () { marker.openPopup(); });
    });

    // Delegated click (namespaced)
    $(document).off('click.oeSelectPoint').on('click.oeSelectPoint', '.oe-select-point', function () {
      var cfg2 = getCfg();
      var pts2 = (cfg2 && cfg2.points) ? cfg2.points : points;

      var index = parseInt($(this).data('index'), 10);
      
      // Validate index before accessing array
      if (isNaN(index) || index < 0 || index >= pts2.length) {
        warn('Invalid point index:', index);
        return;
      }

      var point = pts2[index];
      if (!point || typeof point !== 'object') {
        warn('Invalid point data at index:', index);
        return;
      }

      log('Point selected:', point.name);

      // Hidden fields (for order create hook)
      cachedSelectedPointName = escapeHtml(point.name || '');
      $('#overseas_express_selected_point').val(escapeHtml(point.name || ''));
      $('#oe_address').val(escapeHtml(point.street || ''));
      $('#oe_postcode').val(escapeHtml(point.zip || ''));
      $('#oe_city').val(escapeHtml(point.city || ''));

      // Show pickup selection message
      showPickupNotice(point);

      // Cosmetic (classic)
      setClassicShippingFields(point);

      // Save to session (Blocks-safe) then recalc
      savePickupToSession(point)
        .done(function () {
          log('Pickup saved to session via AJAX');
        })
        .fail(function (xhr) {
          warn('AJAX save pickup failed', xhr && xhr.responseText);
        })
        .always(function () {
          hideOverlay();
          triggerCheckoutUpdate();
        });
    });

    // Optional geocoder
    if (L.Control && typeof L.Control.geocoder !== 'undefined') {
      L.Control.geocoder({
        defaultMarkGeocode: false,
        placeholder: cfg.placeholder || 'Search location...',
        collapsed: false
      })
        .on('markgeocode', function (e) {
          leafletMap.setView(e.geocode.center, 13);
        })
        .addTo(leafletMap);
    }

    setTimeout(function () {
      try { leafletMap.invalidateSize(); } catch (e) {}
    }, 300);
  }

  function openPickupMap() {
    if (isSubmittingOrder) {
      log('Not opening map: order submission in progress');
      return;
    }
    showOverlay();
    setTimeout(initMap, 150);
  }

  function checkShippingMethod() {
    var selectedValue = getSelectedShippingMethod();
    var isPickup = isPickupShippingMethod(selectedValue);
    var wasPickup = isPickupShippingMethod(lastSelectedMethod);

    // Switch away from pickup
    if (selectedValue !== lastSelectedMethod && wasPickup && !isPickup) {
      clearSelectedPickupPoint();
      hidePickupNotice();
      if (isMapOpen) hideOverlay();
    }

    // Switch to pickup -> open map if no point selected
    if (selectedValue !== lastSelectedMethod && !wasPickup && isPickup) {
      if (!hasSelectedPickupPoint()) {
        setTimeout(openPickupMap, 200);
      } else {
        // If a point was already selected (cached/hidden), keep notice visible (best-effort)
        // We don't know full address here without extra data, so only show name if available.
        ensurePickupNotice();
        if (cachedSelectedPointName) {
          $('#oe-pickup-notice').text('Pickup Point Selected: ' + cachedSelectedPointName).show();
        }
      }
    }

    // Initial load only: pickup selected and no point -> open once
    if (!didInitialAutoOpen && lastSelectedMethod === null && isPickup && !hasSelectedPickupPoint()) {
      didInitialAutoOpen = true;
      setTimeout(openPickupMap, 200);
    } else if (lastSelectedMethod === null) {
      didInitialAutoOpen = true;
      // If point is already set on load, show a minimal notice
      if (isPickup && hasSelectedPickupPoint()) {
        ensurePickupNotice();
        $('#oe-pickup-notice').text('Pickup Point Selected: ' + cachedSelectedPointName).show();
      }
    }

    // IMPORTANT: do NOT auto-open when method didn't change (prevents place-order rerender reopening)
    lastSelectedMethod = selectedValue;
  }

  function markSubmittingOrder() {
    isSubmittingOrder = true;
    clearTimeout(window.__oeSubmitTimer);
    window.__oeSubmitTimer = setTimeout(function () {
      isSubmittingOrder = false;
    }, 8000);
  }

  function unmarkSubmittingOrder() {
    isSubmittingOrder = false;
  }

  $(function () {
    log('=== Overseas Express Pickup initialized ===');

    // Ensure notice container exists early
    ensurePickupNotice();

    // Initial checks
    setTimeout(checkShippingMethod, 100);
    setTimeout(checkShippingMethod, 500);
    setTimeout(checkShippingMethod, 1500);

    // Woo events
    $(document.body).on('updated_checkout updated_shipping_method', function () {
      setTimeout(checkShippingMethod, 250);
    });

    // Classic shipping change
    $(document).on('change', 'input[name^="shipping_method"]', function () {
      setTimeout(checkShippingMethod, 100);
    });

    // Blocks shipping change (common)
    $(document).on('change', '.wc-block-components-radio-control__option input[type="radio"], input.wc-block-components-radio-control__input', function () {
      setTimeout(checkShippingMethod, 100);
    });

    // Submit guard: capture submit (click + Enter)
    document.addEventListener('submit', function (e) {
      var form = e.target;
      if (!form) return;
      if (form.matches && (form.matches('form.checkout') || form.matches('.wc-block-checkout__form'))) {
        markSubmittingOrder();
      }
    }, true);

    // Also guard on place order clicks (extra)
    $(document).on('click', '#place_order, button[name="woocommerce_checkout_place_order"], .wc-block-components-checkout-place-order-button', function () {
      markSubmittingOrder();
    });

    // If errors happen, allow map logic again
    $(document.body).on('checkout_error', function () {
      unmarkSubmittingOrder();
    });
  });

})(jQuery);