<?php

if ( ! function_exists( 'oosdm_register_submenu_page' ) ) {

	function oosdm_register_submenu_page() {
		add_submenu_page(
			'edit.php?post_type=product',
			__( 'Out of Stock Visibility Manager', 'out-of-stock-display-manager-for-woocommerce' ),
			__( 'Out of Stock Visibility', 'out-of-stock-display-manager-for-woocommerce' ),
			'manage_woocommerce',
			'out-of-stock-display-manager-for-woocommerce',
			'oosdm_load_template'
		);
	}
	add_action( 'admin_menu', 'oosdm_register_submenu_page' );
}

if ( ! function_exists( 'oosdm_load_template' ) ) {

	function oosdm_load_template() {
		$page = ( isset( $_GET['page'] ) && 'out-of-stock-display-manager-for-woocommerce' === sanitize_text_field( wp_unslash( $_GET['page'] ) ) ) ? 'oosdm-products-table' : '';
		if ( $page ) {
			include_once OOSDM_PLUGIN_DIR . '/' . $page . '.php';
		}
	}

}

if ( ! function_exists( 'oosdm_current_mode' ) ) {
	/**
	 * Get human-readable visibility mode from product visibility terms.
	 *
	 * @param array $terms Terms from product_visibility.
	 * @return string
	 */
	function oosdm_current_mode( $terms ) {
		if ( empty( $terms ) || is_wp_error( $terms ) ) {
			return __( 'Unknown', 'out-of-stock-display-manager-for-woocommerce' );
		}

		$slugs = array();
		foreach ( $terms as $term ) {
			$slugs[] = $term->slug;
		}

		$has_catalog = in_array( 'exclude-from-catalog', $slugs, true );
		$has_search  = in_array( 'exclude-from-search', $slugs, true );

		if ( $has_catalog && $has_search ) {
			return __( 'Hidden (shop + search)', 'out-of-stock-display-manager-for-woocommerce' );
		}
		if ( $has_catalog && ! $has_search ) {
			return __( 'Search only', 'out-of-stock-display-manager-for-woocommerce' );
		}
		if ( ! $has_catalog && $has_search ) {
			return __( 'Shop only', 'out-of-stock-display-manager-for-woocommerce' );
		}

		return __( 'Visible (shop + search)', 'out-of-stock-display-manager-for-woocommerce' );
	}
}

/**
 * Enqueue admin assets on plugin page.
 */
function oosdm_admin_assets( $hook ) {
	$screen = get_current_screen();
	if ( empty( $screen ) || 'product_page_out-of-stock-display-manager-for-woocommerce' !== $screen->id ) {
		return;
	}

	$css_version = file_exists( OOSDM_PLUGIN_DIR . 'assets/admin.css' ) ? filemtime( OOSDM_PLUGIN_DIR . 'assets/admin.css' ) : OOSDM_PLUGIN_VERSION;
	$js_version  = file_exists( OOSDM_PLUGIN_DIR . 'assets/admin.js' ) ? filemtime( OOSDM_PLUGIN_DIR . 'assets/admin.js' ) : OOSDM_PLUGIN_VERSION;

	wp_enqueue_style( 'oosdm-admin', OOSDM_PLUGIN_URI . 'assets/admin.css', array(), $css_version );
	wp_enqueue_script( 'oosdm-admin', OOSDM_PLUGIN_URI . 'assets/admin.js', array( 'jquery' ), $js_version, true );
	wp_localize_script(
		'oosdm-admin',
		'oosdmVars',
		array(
			'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
			'nonce'     => wp_create_nonce( 'oosdm_update' ),
			'i18n'      => array(
				'processing'    => __( 'Processing...', 'out-of-stock-display-manager-for-woocommerce' ),
				'success'       => __( 'Visibility updated!', 'out-of-stock-display-manager-for-woocommerce' ),
				'error'         => __( 'Update failed. Please refresh and try again.', 'out-of-stock-display-manager-for-woocommerce' ),
				'selectWarning' => __( 'Select at least one product for bulk update.', 'out-of-stock-display-manager-for-woocommerce' ),
				'selectVisibility' => __( 'Choose a visibility option before applying.', 'out-of-stock-display-manager-for-woocommerce' ),
			),
		)
	);
}
add_action( 'admin_enqueue_scripts', 'oosdm_admin_assets' );

if ( function_exists( 'add_theme_support' ) ) {
	add_theme_support( 'post-thumbnails' );
	add_image_size( 'outofstock-custom-thumb', 40, 40 ); // 40px thumb for table.
}

/**
 * Keep WooCommerce global hide-out-of-stock setting disabled while plugin is active.
 */
function oosdm_force_global_visibility_setting() {
	if ( function_exists( 'update_option' ) ) {
		update_option( 'woocommerce_hide_out_of_stock_items', 'no' );
	}
}
add_action( 'init', 'oosdm_force_global_visibility_setting', 5 );

/**
 * Determine if overrides should apply on the current product query.
 *
 * @param WP_Query $query Query instance.
 * @return bool
 */
function oosdm_should_apply_visibility_overrides( $query ) {
	if ( ! function_exists( 'wc_hide_out_of_stock_products' ) || ! function_exists( 'wc_get_product_visibility_term_ids' ) ) {
		return false;
	}

	if ( is_admin() && ! wp_doing_ajax() ) {
		return false;
	}

	if ( ! wc_hide_out_of_stock_products() ) {
		return false;
	}

	if ( empty( $query ) || ! is_a( $query, 'WP_Query' ) ) {
		return false;
	}

	return true;
}

/**
 * Remove WooCommerce's "outofstock" exclusion so our overrides can show selected items.
 */
function oosdm_filter_tax_query( $tax_query, $query ) {
	if ( ! oosdm_should_apply_visibility_overrides( $query ) ) {
		return $tax_query;
	}

	$term_ids = wc_get_product_visibility_term_ids();
	if ( empty( $term_ids['outofstock'] ) ) {
		return $tax_query;
	}

	if ( ! is_array( $tax_query ) ) {
		$tax_query = array();
	}

	foreach ( $tax_query as $index => $tax ) {
		if (
			isset( $tax['taxonomy'], $tax['terms'], $tax['operator'] ) &&
			'product_visibility' === $tax['taxonomy'] &&
			'NOT IN' === $tax['operator'] &&
			in_array( $term_ids['outofstock'], (array) $tax['terms'], true )
		) {
			unset( $tax_query[ $index ] );
		}
	}

	return array_values( $tax_query );
}
add_filter( 'woocommerce_product_query_tax_query', 'oosdm_filter_tax_query', 20, 2 );

/**
 * Add meta query that whitelists allowed out-of-stock products and keeps in-stock items visible.
 */
function oosdm_filter_meta_query( $meta_query, $query ) {
	if ( ! oosdm_should_apply_visibility_overrides( $query ) ) {
		return $meta_query;
	}

	if ( ! is_array( $meta_query ) ) {
		$meta_query = array();
	}

	// Remove WooCommerce's default "_stock_status = instock" clause so we can supply our own.
	$clean_meta_query = array();
	foreach ( $meta_query as $key => $clause ) {
		if ( 'relation' === $key ) {
			$clean_meta_query[ $key ] = $clause;
			continue;
		}

		if ( is_array( $clause ) && isset( $clause['key'] ) && '_stock_status' === $clause['key'] ) {
			continue;
		}

		$clean_meta_query[] = $clause;
	}
	$meta_query = $clean_meta_query;

	$allowed_values = array( 'visible' );
	if ( $query->is_search() ) {
		$allowed_values[] = 'search';
	} else {
		$allowed_values[] = 'catalog';
	}

	$meta_query[] = array(
		'relation' => 'OR',
		array(
			'key'   => '_stock_status',
			'value' => 'instock',
		),
		array(
			'relation' => 'AND',
			array(
				'key'   => '_stock_status',
				'value' => 'outofstock',
			),
			array(
				'key'     => '_oosdm_visibility',
				'value'   => $allowed_values,
				'compare' => 'IN',
			),
		),
	);

	return $meta_query;
}
add_filter( 'woocommerce_product_query_meta_query', 'oosdm_filter_meta_query', 20, 2 );

/**
 * Backstop override: force removal of WooCommerce stock/tax exclusions on the main product queries.
 */
function oosdm_pre_get_posts_override( $query ) {
	if ( ! oosdm_should_apply_visibility_overrides( $query ) ) {
		return;
	}

	// Only touch product/archive/search queries on the frontend.
	if (
		( ! $query->is_main_query() && ! $query->get( 'wc_query' ) )
		|| ( ! $query->is_post_type_archive( 'product' ) && ! $query->is_search() && 'product' !== $query->get( 'post_type' ) )
	) {
		return;
	}

	$term_ids = wc_get_product_visibility_term_ids();

	$tax_query = $query->get( 'tax_query' );
	if ( ! is_array( $tax_query ) ) {
		$tax_query = array();
	}

	// Remove core outofstock exclusion.
	foreach ( $tax_query as $index => $tax ) {
		if (
			isset( $tax['taxonomy'], $tax['terms'], $tax['operator'] ) &&
			'product_visibility' === $tax['taxonomy'] &&
			'NOT IN' === $tax['operator'] &&
			isset( $term_ids['outofstock'] ) &&
			in_array( $term_ids['outofstock'], (array) $tax['terms'], true )
		) {
			unset( $tax_query[ $index ] );
		}
	}
	$query->set( 'tax_query', array_values( $tax_query ) );

	$meta_query = $query->get( 'meta_query' );
	if ( ! is_array( $meta_query ) ) {
		$meta_query = array();
	}

	// Remove default stock status filter.
	$clean_meta_query = array();
	foreach ( $meta_query as $key => $clause ) {
		if ( 'relation' === $key ) {
			$clean_meta_query[ $key ] = $clause;
			continue;
		}

		if ( is_array( $clause ) && isset( $clause['key'] ) && '_stock_status' === $clause['key'] ) {
			continue;
		}

		$clean_meta_query[] = $clause;
	}
	$meta_query = $clean_meta_query;

	$allowed_values = array( 'visible' );
	if ( $query->is_search() ) {
		$allowed_values[] = 'search';
	} else {
		$allowed_values[] = 'catalog';
	}

	$meta_query[] = array(
		'relation' => 'OR',
		array(
			'key'   => '_stock_status',
			'value' => 'instock',
		),
		array(
			'relation' => 'AND',
			array(
				'key'   => '_stock_status',
				'value' => 'outofstock',
			),
			array(
				'key'     => '_oosdm_visibility',
				'value'   => $allowed_values,
				'compare' => 'IN',
			),
		),
	);

	$query->set( 'meta_query', $meta_query );
}
add_action( 'pre_get_posts', 'oosdm_pre_get_posts_override', 9 );

/**
 * Last-resort override for WooCommerce visibility checks on single products.
 * Ensures allowed out-of-stock products remain visible even when the global hide is on.
 *
 * @param bool        $visible   Current visibility decision.
 * @param int         $product_id Product ID.
 * @return bool
 */
function oosdm_force_product_visibility( $visible, $product_id ) {
	if ( ! function_exists( 'wc_hide_out_of_stock_products' ) || ! wc_hide_out_of_stock_products() ) {
		return $visible;
	}

	$status = get_post_meta( $product_id, '_stock_status', true );
	if ( 'outofstock' !== $status ) {
		return $visible;
	}

	$allowed = get_post_meta( $product_id, '_oosdm_visibility', true );
	if ( empty( $allowed ) ) {
		return $visible;
	}

	// Respect search vs catalog context.
	if ( is_search() && in_array( $allowed, array( 'visible', 'search' ), true ) ) {
		return true;
	}

	if ( ! is_search() && in_array( $allowed, array( 'visible', 'catalog' ), true ) ) {
		return true;
	}

	return $visible;
}
add_filter( 'woocommerce_product_is_visible', 'oosdm_force_product_visibility', 20, 2 );



if ( ! function_exists( 'oosdm_update_update_prod_display' ) ) {

	add_action( 'wp_ajax_oosdm_update_update_prod_display', 'oosdm_update_update_prod_display' );

	function oosdm_update_update_prod_display() {
		if ( ! current_user_can( 'manage_woocommerce' ) && ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized request.', 'out-of-stock-display-manager-for-woocommerce' ) ), 403 );
		}

		check_ajax_referer( 'oosdm_update', 'nonce' );

		$raw_ids = isset( $_POST['id'] ) ? wp_unslash( $_POST['id'] ) : array(); // phpcs:ignore WordPress.Security.NonceVerification.Missing
		$ids     = array();
		if ( is_array( $raw_ids ) ) {
			foreach ( $raw_ids as $id ) {
				$id = absint( $id );
				if ( $id && 'product' === get_post_type( $id ) ) {
					$ids[] = $id;
				}
			}
		} else {
			$id = absint( $raw_ids );
			if ( $id && 'product' === get_post_type( $id ) ) {
				$ids[] = $id;
			}
		}

		if ( empty( $ids ) ) {
			wp_send_json_error( array( 'message' => __( 'No valid products provided.', 'out-of-stock-display-manager-for-woocommerce' ) ), 400 );
		}

		$visibility = isset( $_POST['visibility'] ) ? sanitize_key( wp_unslash( $_POST['visibility'] ) ) : '';
		$allowed    = array( 'visible', 'catalog', 'search', 'hidden' );
		if ( ! in_array( $visibility, $allowed, true ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid visibility option.', 'out-of-stock-display-manager-for-woocommerce' ) ), 400 );
		}

		$term_catalog = get_term_by( 'slug', 'exclude-from-catalog', 'product_visibility' );
		$term_search  = get_term_by( 'slug', 'exclude-from-search', 'product_visibility' );
		$term_oos     = get_term_by( 'slug', 'outofstock', 'product_visibility' );

		if ( ! $term_catalog || ! $term_search || ! $term_oos ) {
			wp_send_json_error( array( 'message' => __( 'Required visibility terms are missing.', 'out-of-stock-display-manager-for-woocommerce' ) ), 400 );
		}

		$terms_array = array(
			$term_catalog->term_id,
			$term_search->term_id,
			$term_oos->term_id,
		);

		oosdm_product_remove_display( $ids, $terms_array );

		foreach ( $ids as $id ) {
			update_post_meta( $id, '_oosdm_visibility', $visibility );
			switch ( $visibility ) {
				case 'hidden':
					wp_set_object_terms( $id, $term_catalog->term_id, 'product_visibility', true );
					wp_set_object_terms( $id, $term_search->term_id, 'product_visibility', true );
					wp_set_object_terms( $id, $term_oos->term_id, 'product_visibility', true );
					break;
				case 'search':
					wp_set_object_terms( $id, $term_catalog->term_id, 'product_visibility', true );
					wp_set_object_terms( $id, $term_oos->term_id, 'product_visibility', true );
					break;
				case 'catalog':
					wp_set_object_terms( $id, $term_search->term_id, 'product_visibility', true );
					wp_set_object_terms( $id, $term_oos->term_id, 'product_visibility', true );
					break;
				case 'visible':
				default:
					wp_set_object_terms( $id, $term_oos->term_id, 'product_visibility', true );
					break;
			}

			// Persist changes at the DB/lookup-table level so catalog/search honor the selection immediately.
			if ( function_exists( 'wc_delete_product_transients' ) ) {
				wc_delete_product_transients( $id );
			}
			if ( function_exists( 'wc_update_product_lookup_tables' ) ) {
				wc_update_product_lookup_tables( $id );
			}
		}

		wp_send_json_success(
			array(
				'updated'    => $ids,
				'visibility' => $visibility,
			)
		);
	}
}


if( !function_exists('oosdm_product_remove_display')){

	function oosdm_product_remove_display($param_ids,$terms_array){

		foreach( (array) $param_ids as $id){
			$id = absint( $id );
			if ( ! $id ) {
				continue;
			}
			foreach( (array) $terms_array as $term_id){
				$term_id = absint( $term_id );
				if ( $term_id ) {
					wp_remove_object_terms($id,$term_id,'product_visibility');
				}
			}
		}
	}

}


if( !function_exists('oosdm_selected_display')){

	function oosdm_selected_display( $array, $attr_val = '' ){

		if ( empty( $array ) || is_wp_error( $array ) ) {
			return '';
		}

		if($attr_val == 'visible'){
			if($array[0]->slug == 'outofstock' && !isset($array[1])){
				return "selected";
			}
		}elseif($attr_val == 'catalog') {
			$arr = array();
			foreach($array as $k=>$val){	
				$arr[] = $array[$k]->slug;
			}
			
			if(in_array('exclude-from-search',$arr) && !in_array('exclude-from-catalog', $arr)){
				return "selected";
			}
		}elseif($attr_val == 'search') {
			$arr = array();
			foreach($array as $k=>$val){	
				$arr[] = $val->slug;
			}
			if(!in_array('exclude-from-search',$arr) && in_array('exclude-from-catalog', $arr)){
				return "selected";
			}
		}else{
			$arr = array();
			foreach($array as $k=>$val){	
				$arr[] = $val->slug;
			}
			if(in_array('exclude-from-search',$arr) && in_array('exclude-from-catalog', $arr)){
				return "selected";
			}
		}
		
	}

}
