<?php
/**
 * Shortcode handler for ouRSVP forms
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class OurSVP_Shortcode {

    /**
     * Counter for multiple instances on same page
     */
    private static $instances = 0;

    /**
     * Constructor
     */
    public function __construct() {
        add_shortcode('oursvp', array($this, 'render_shortcode'));
        // Note: Widget script is loaded inline using the async loader pattern
    }

    /**
     * Render the shortcode
     *
     * Usage: [oursvp event_id="abc123" show_back_btn="true" language="spanish"]
     *
     * @param array $atts Shortcode attributes
     * @return string HTML output
     */
    public function render_shortcode($atts) {
        // Increment instance counter for unique IDs
        self::$instances++;
        $instance_id = 'oursvp-' . self::$instances;

        // Parse attributes with defaults
        $atts = shortcode_atts(array(
            'event_id' => '',
            'show_back_btn' => 'false',
            'show_full_name' => 'false',
            'language' => 'english',
            'limit_search' => 'false',
            // Style parameters (CSS colors)
            'bg_color' => '',
            'heading_color' => '',
            'text_color' => '',
            'date_color' => '',
            'accent_color' => '',
            'button_bg_color' => '',
            'button_text_color' => '',
            'button_border_color' => '',
            'button_border_radius' => '',
            'button_border_width' => '',
        ), $atts, 'oursvp');

        // Validate required attributes
        if (empty($atts['event_id'])) {
            return $this->render_error(__('Error: event_id is required. Please add your ouRSVP event ID.', 'oursvp-event-rsvp-forms'));
        }

        // Validate event_id format (basic alphanumeric check)
        if (!preg_match('/^[a-zA-Z0-9_-]+$/', $atts['event_id'])) {
            return $this->render_error(__('Error: Invalid event_id format.', 'oursvp-event-rsvp-forms'));
        }

        // Validate language
        $allowed_languages = array('english', 'spanish');
        if (!in_array($atts['language'], $allowed_languages)) {
            $atts['language'] = 'english';
        }

        // Convert string booleans to actual booleans for JavaScript
        $show_back_btn = filter_var($atts['show_back_btn'], FILTER_VALIDATE_BOOLEAN);
        $show_full_name = filter_var($atts['show_full_name'], FILTER_VALIDATE_BOOLEAN);
        $limit_search = filter_var($atts['limit_search'], FILTER_VALIDATE_BOOLEAN);

        // Build the output
        ob_start();
        ?>
        <div id="<?php echo esc_attr($instance_id); ?>" class="oursvp-container" style="width: 100%; min-height: 400px;"></div>
        <script>
        (function() {
            'use strict';

            // ouRSVP widget loader pattern - creates queue and loads script
            (function (w, d, s, o, f, js, fjs) {
                // Set up the loader object with queue if it doesn't exist
                w[o] = w[o] || function () { (w[o].q = w[o].q || []).push(arguments) };

                // Only inject script if not already loaded
                if (!d.getElementById(o)) {
                    js = d.createElement(s);
                    fjs = d.getElementsByTagName(s)[0];
                    js.id = o;
                    js.src = f;
                    js.async = 1;
                    if (fjs && fjs.parentNode) {
                        fjs.parentNode.insertBefore(js, fjs);
                    } else {
                        d.body.appendChild(js);
                    }
                }
            }(window, document, 'script', 'oursvp', '<?php echo esc_js(OURSVP_WIDGET_URL); ?>'));

            // Initialize this specific instance
            var element = document.getElementById('<?php echo esc_js($instance_id); ?>');
            if (element) {
                oursvp('init', {
                    element: element,
                    eventId: '<?php echo esc_js($atts['event_id']); ?>',
                    showBackBtn: <?php echo $show_back_btn ? 'true' : 'false'; ?>,
                    showFullName: <?php echo $show_full_name ? 'true' : 'false'; ?>,
                    language: '<?php echo esc_js($atts['language']); ?>',
                    limitSearch: <?php echo $limit_search ? 'true' : 'false'; ?>
                });

                // Apply custom styling via CSS variables
                <?php if (!empty($atts['bg_color'])): ?>
                document.body.style.setProperty('--bg-color', '<?php echo esc_js($atts['bg_color']); ?>');
                <?php endif; ?>
                <?php if (!empty($atts['heading_color'])): ?>
                document.body.style.setProperty('--heading-color', '<?php echo esc_js($atts['heading_color']); ?>');
                <?php endif; ?>
                <?php if (!empty($atts['text_color'])): ?>
                document.body.style.setProperty('--text-color', '<?php echo esc_js($atts['text_color']); ?>');
                <?php endif; ?>
                <?php if (!empty($atts['date_color'])): ?>
                document.body.style.setProperty('--date-color', '<?php echo esc_js($atts['date_color']); ?>');
                <?php endif; ?>
                <?php if (!empty($atts['accent_color'])): ?>
                document.body.style.setProperty('--accent-color', '<?php echo esc_js($atts['accent_color']); ?>');
                <?php endif; ?>
                <?php if (!empty($atts['button_bg_color'])): ?>
                document.body.style.setProperty('--button-bg-color', '<?php echo esc_js($atts['button_bg_color']); ?>');
                <?php endif; ?>
                <?php if (!empty($atts['button_text_color'])): ?>
                document.body.style.setProperty('--button-text-color', '<?php echo esc_js($atts['button_text_color']); ?>');
                <?php endif; ?>
                <?php if (!empty($atts['button_border_color'])): ?>
                document.body.style.setProperty('--border-color', '<?php echo esc_js($atts['button_border_color']); ?>');
                <?php endif; ?>
                <?php if (!empty($atts['button_border_radius'])): ?>
                document.body.style.setProperty('--button-shape', '<?php echo esc_js($atts['button_border_radius']); ?>');
                <?php endif; ?>
                <?php if (!empty($atts['button_border_width'])): ?>
                document.body.style.setProperty('--button-border-width', '<?php echo esc_js($atts['button_border_width']); ?>');
                <?php endif; ?>
            } else {
                console.error('ouRSVP: Container element not found');
            }
        })();
        </script>
        <?php
        return ob_get_clean();
    }

    /**
     * Render error message
     *
     * @param string $message Error message
     * @return string HTML error output
     */
    private function render_error($message) {
        return sprintf(
            '<div class="oursvp-error" style="padding: 1rem; background: #fee2e2; border: 1px solid #fca5a5; border-radius: 4px; color: #991b1b; font-family: -apple-system, BlinkMacSystemFont, \'Segoe UI\', Roboto, sans-serif;">%s</div>',
            esc_html($message)
        );
    }
}

// Initialize the shortcode
new OurSVP_Shortcode();
