<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly

$api_key    = get_option('orufy_api_key');
$api_secret = get_option('orufy_api_secret');
$last_api_response = get_transient('orufy_booking_last_api_response');

$events = [];

if (is_array($last_api_response) && isset($last_api_response['events'])) {
    $events = $last_api_response['events'];
}

// If API keys exist and events not cached, fetch again (only on this page load)
if ($api_key && $api_secret && empty($events)) {
    $response = wp_remote_get('https://bookings.orufy.com/api/v1/bookings/api/event', [
        'headers' => [
            'x-api-key'    => $api_key,
            'x-api-secret' => $api_secret,
        ],
        'timeout' => 10,
    ]);

    if (! is_wp_error($response)) {
        $body = wp_remote_retrieve_body($response);
        $json = json_decode($body, true);

        if (! empty($json['isSuccess']) && ! empty($json['events'])) {
            $events = array_filter($json['events'], fn($e) => ! empty($e['name']));
            set_transient('orufy_booking_last_api_response', $json, 60 * 60);
        }
    }
}

$saved_event = get_option('orufy_selected_event_data');
?>

<div class="wrap">
    <h2><?php esc_html_e('Orufy Booking Settings', 'orufy-bookings'); ?></h2>

    <?php
    if ($result = get_transient('orufy_booking_api_result')) {
        echo '<div class="notice notice-info"><p>' . esc_html($result) . '</p></div>';
        delete_transient('orufy_booking_api_result');
    }
    ?>

    <!-- Save API Keys -->
    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
        <input type="hidden" name="action" value="orufy_booking_save_settings">
        <?php wp_nonce_field('orufy_booking_save_settings'); ?>

        <table class="form-table">
            <tr>
                <th><label for="orufy_api_key"><?php esc_html_e('API Key', 'orufy-bookings'); ?></label></th>
                <td><input type="text" name="orufy_api_key" id="orufy_api_key" value="<?php echo esc_attr($api_key); ?>" class="regular-text" /></td>
            </tr>
            <tr>
                <th><label for="orufy_api_secret"><?php esc_html_e('API Secret', 'orufy-bookings'); ?></label></th>
                <td><input type="text" name="orufy_api_secret" id="orufy_api_secret" value="<?php echo esc_attr($api_secret); ?>" class="regular-text" /></td>
            </tr>
        </table>

        <p><input type="submit" class="button button-primary" value="<?php esc_attr_e('Save and Connect', 'orufy-bookings'); ?>" /></p>
    </form>

    <!-- Select Event -->
    <?php if (! empty($events)) : ?>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin-top: 30px;">
            <input type="hidden" name="action" value="orufy_booking_select_event">
            <?php wp_nonce_field('orufy_booking_select_event'); ?>

            <table class="form-table">
                <tr>
                    <th><label for="orufy_selected_event"><?php esc_html_e('Select Event', 'orufy-bookings'); ?></label></th>
                    <td>
                        <select name="orufy_selected_event" id="orufy_selected_event" class="regular-text">
                            <option value=""><?php echo esc_html__('-- Select an EVENT --', 'orufy-bookings'); ?></option>
                            <?php foreach ($events as $event) : ?>
                                <?php if (($event['status'] ?? '') === 'ACTIVE') : ?>
                                    <option value="<?php echo esc_attr($event['_id'] ?? ''); ?>"
                                        <?php selected($event['_id'] ?? '', $saved_event['id'] ?? ''); ?>>
                                        <?php echo esc_html($event['name'] ?? ''); ?>
                                    </option>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </select>
                    </td>
                </tr>
            </table>
            <p><input type="submit" class="button button-secondary" value="<?php esc_attr_e('Save Selected Event', 'orufy-bookings'); ?>" /></p>
        </form>

        <?php if (! empty($saved_event)) : ?>
            <div style="margin-top: 30px;">
                <h3><?php esc_html_e('Shortcodes for Selected Event', 'orufy-bookings'); ?></h3>
                <p><?php esc_html_e('Use these anywhere in posts, pages, or templates:', 'orufy-bookings'); ?></p>
                <code>[orufy-booking-inline]</code><br><br>
                <code>[orufy-booking-popup-widget]</code><br><br>
                <code>[orufy-booking-popup-text]</code>
            </div>
        <?php endif; ?>
    <?php else : ?>
        <p><?php esc_html_e("No events available. Save valid API keys and try 'Save and Connect' to fetch events.", 'orufy-bookings'); ?></p>
    <?php endif; ?>
</div>