<?php
/**
 * Plugin Name: Orphanix Media Cleanup
 * Description: Clean, optimize, and manage your WordPress media library by detecting unused, broken, and dynamically used media files.
 * Plugin URI: https://wordpress.org/plugins/orphanix-media-cleanup/
 * Version: 1.0.0
 * Author: Atique Ullah
 * Author URI: https://www.linkedin.com/in/atiqueullahlimon
 * Text Domain: orphanix-media-cleanup
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'ORPHANIX_VERSION', '1.0.0' );
define( 'ORPHANIX_PATH', plugin_dir_path( __FILE__ ) );
define( 'ORPHANIX_URL', plugin_dir_url( __FILE__ ) );

// Load core classes before hooks
require_once ORPHANIX_PATH . 'includes/core/class-orphanix-activator.php';
require_once ORPHANIX_PATH . 'includes/core/class-orphanix-deactivator.php';

// Autoloader
require_once ORPHANIX_PATH . 'includes/core/class-orphanix-loader.php';

// Activation / Deactivation
register_activation_hook( __FILE__, array( 'ORPHANIX_Activator', 'activate' ) );
register_deactivation_hook( __FILE__, array( 'ORPHANIX_Deactivator', 'deactivate' ) );

// Bootstrap
function orphanix_init() {
	$loader = new ORPHANIX_Loader();
	$loader->run();
}
add_action( 'plugins_loaded', 'orphanix_init' );

function orphanix_enqueue_admin_assets( $hook_suffix ) {
	$is_plugin_screen = ( 'plugins.php' === $hook_suffix );
	$is_orphanix_screen    = ( false !== strpos( $hook_suffix, 'orphanix' ) );

	if ( ! $is_plugin_screen && ! $is_orphanix_screen ) {
		return;
	}

	$admin_css_ver = file_exists( ORPHANIX_PATH . 'assets/css/admin.css' ) ? (string) filemtime( ORPHANIX_PATH . 'assets/css/admin.css' ) : ORPHANIX_VERSION;
	$admin_js_ver  = file_exists( ORPHANIX_PATH . 'assets/js/admin.js' ) ? (string) filemtime( ORPHANIX_PATH . 'assets/js/admin.js' ) : ORPHANIX_VERSION;

	wp_register_style(
		'orphanix-admin',
		ORPHANIX_URL . 'assets/css/admin.css',
		array(),
		$admin_css_ver
	);
	wp_register_script(
		'orphanix-admin',
		ORPHANIX_URL . 'assets/js/admin.js',
		array( 'jquery' ),
		$admin_js_ver,
		true
	);

	wp_enqueue_style( 'orphanix-admin' );
	wp_enqueue_script( 'orphanix-admin' );
}
add_action( 'admin_enqueue_scripts', 'orphanix_enqueue_admin_assets' );

// Add plugin action links
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'orphanix_plugin_action_links' );

function orphanix_plugin_action_links( $links ) {
	$dashboard_url = admin_url( 'admin.php?page=orphanix-dashboard' );
	$settings_url  = admin_url( 'admin.php?page=orphanix-settings' );

	$action_links = array(
		'dashboard' => '<a href="' . esc_url( $dashboard_url ) . '">' . esc_html__( 'Dashboard', 'orphanix-media-cleanup' ) . '</a>',
		'settings'  => '<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'Settings', 'orphanix-media-cleanup' ) . '</a>',
	);

	return array_merge( $action_links, $links );
}
