<?php
if ( ! class_exists( 'WP_List_Table' ) ) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class ORPHANIX_Scan_Items_Table extends WP_List_Table {
    protected $scan_id;

    public function __construct($scan_id) {
        $this->scan_id = $scan_id;
        parent::__construct([
            'singular' => 'item',
            'plural'   => 'items',
            'ajax'     => false,
        ]);
    }

    public function get_columns() {
        return [
            'thumb'         => __('Thumb','orphanix-media-cleanup'),
            'attachment_id' => __('Origin ID','orphanix-media-cleanup'),
            'file_path'     => __('Title & Path','orphanix-media-cleanup'),
            'directory_type'=> __('Directory','orphanix-media-cleanup'),
            'alt_text'      => __('Alt Tag','orphanix-media-cleanup'),
            'file_size'     => __('Size','orphanix-media-cleanup'),
            'usage_context' => __('Belongs To','orphanix-media-cleanup'),
            'status'        => __('Status','orphanix-media-cleanup'),
            'actions'       => __('Actions','orphanix-media-cleanup'),
        ];
    }

    public function prepare_items() {
        $repo = new ORPHANIX_Scan_Items_Repository();
        $this->items = $repo->all_items($this->scan_id);
        $this->_column_headers = [$this->get_columns(), [], []];
    }

    public function column_thumb($item) {
        if ($item['attachment_id']) {
            return wp_get_attachment_image($item['attachment_id'], 'thumbnail');
        }
        return '&ndash;';
    }

    public function column_usage_context($item) {
        return $this->render_belongs_to($item);
    }

    public function column_default($item, $column_name) {
        switch ($column_name) {
            case 'attachment_id':
            case 'file_path':
            case 'directory_type':
            case 'file_size':
            case 'status':
                return esc_html($item[$column_name]);
            case 'alt_text':
                // Display actual alt text if available, otherwise show "Missing" badge
                if (!empty($item[$column_name])) {
                    return esc_html($item[$column_name]);
                }
                return '<span data-orphanix-style="orphanix-s-0086">Missing</span>';
            case 'actions':
                $trash_url = wp_nonce_url(admin_url('admin.php?page=orphanix-results&trash_item='.$item['id']), 'orphanix_trash_item');
                $delete_url = wp_nonce_url(admin_url('admin.php?page=orphanix-results&delete_item='.$item['id']), 'orphanix_delete_item');
                return sprintf('<a href="%s">%s</a> | <a href="%s" class="danger">%s</a>',
                    esc_url($trash_url),
                    __('Trash','orphanix-media-cleanup'),
                    esc_url($delete_url),
                    __('Delete','orphanix-media-cleanup')
                );
            default:
                return '&ndash;';

        }
    }

    /**
     * Render "Belongs To" column with post titles, sources, and database locations
     */
    private function render_belongs_to($item) {
        $usage_context = $item['usage_context'] ?? '{}';
        
        // Decode JSON if it's a string
        if (is_string($usage_context)) {
            $context = json_decode($usage_context, true);
        } else {
            $context = $usage_context;
        }

        $context = is_array($context) ? $context : [];
        $post_ids = $context['post_ids'] ?? [];
        $sources = $context['sources'] ?? [];
        $count = $context['count'] ?? 0;

        // If no usage found, show "Orphan" badge
        if (empty($post_ids) && empty($sources)) {
            return '<span data-orphanix-style="orphanix-s-0087">Orphan</span>';
        }

        // Map source slugs to readable labels with colors
        $source_map = [
            'featured_image' => ['Featured Image', '#e74c3c'],
            'post_content' => ['Post Content', '#3498db'],
            'custom_field' => ['Custom Field', '#9b59b6'],
            'gutenberg' => ['Gutenberg Block', '#f39c12'],
            'elementor' => ['Elementor', '#4ecdc4'],
            'divi' => ['Divi', '#2ecc71'],
            'wpbakery' => ['WPBakery', '#e91e63'],
            'acf' => ['ACF', '#ff6b6b'],
            'metabox' => ['MetaBox', '#4a90e2'],
            'pods' => ['Pods', '#50c878'],
            'woocommerce' => ['WooCommerce', '#9c27b0'],
            'sliders' => ['Sliders', '#ff9800'],
            'widget' => ['Widget', '#795548'],
            'menu' => ['Menu', '#607d8b'],
            'theme' => ['Theme', '#455a64'],
            'deep_db' => ['Database', '#2c3e50'],
        ];

        $html = '<div data-orphanix-style="orphanix-s-0088">';

        // Separate numeric post IDs from table names
        $numeric_post_ids = [];
        $table_names = [];

        foreach ($post_ids as $pid) {
            if (is_numeric($pid)) {
                $numeric_post_ids[] = intval($pid);
            } else {
                $table_names[] = $pid;
            }
        }

        // Display post titles with links
        if (!empty($numeric_post_ids)) {
            $html .= '<div data-orphanix-style="orphanix-s-0089">';
            
            foreach (array_unique($numeric_post_ids) as $post_id) {
                $post = get_post($post_id);
                if ($post) {
                    $post_type_obj = get_post_type_object($post->post_type);
                    $post_type_label = $post_type_obj ? $post_type_obj->labels->singular_name : ucfirst($post->post_type);
                    
                    // Color coding for post types
                    $type_colors = [
                        'post' => '#0073aa',
                        'page' => '#23282d',
                        'product' => '#96588a',
                    ];
                    $bg_color = $type_colors[$post->post_type] ?? '#6c757d';

                    $edit_url = get_edit_post_link($post_id);
                    $post_title = !empty($post->post_title) ? $post->post_title : sprintf('(ID: %d)', $post_id);

                    $html .= '<div data-orphanix-style="orphanix-s-0154">';
                    $html .= '<a href="' . esc_url($edit_url) . '" data-orphanix-style="orphanix-s-0091">' . esc_html($post_title) . '</a>';
                    $html .= '<span data-orphanix-style="orphanix-s-0155">' . esc_html($post_type_label) . '</span>';
                    $html .= '</div>';
                }
            }
            
            $html .= '</div>';
        }

        // Display sources as badges
        if (!empty($sources)) {
            $html .= '<div data-orphanix-style="orphanix-s-0156">';
            
            foreach ($sources as $source) {
                $source_info = $source_map[$source] ?? [$source, '#95a5a6'];
                list($label, $color) = $source_info;

                $html .= '<span data-orphanix-style="orphanix-s-0157">' . esc_html($label) . '</span>';
            }
            
            $html .= '</div>';
        }

        // Display database table names
        if (!empty($table_names)) {
            $html .= '<div data-orphanix-style="orphanix-s-0095">';
            $html .= '<strong data-orphanix-style="orphanix-s-0158">Tables:</strong> ' . esc_html(implode(', ', $table_names));
            $html .= '</div>';
        }

        $html .= '</div>';

        // Allow necessary HTML tags with style attributes
        $allowed_html = [
            'div' => ['style' => true],
            'span' => ['style' => true],
            'a' => ['href' => true, 'style' => true],
            'strong' => ['style' => true],
        ];

        return wp_kses($html, $allowed_html);
    }

    public function get_bulk_actions() {
        return [
            'trash'  => __('Move to Trash','orphanix-media-cleanup'),
            'delete' => __('Delete Permanently','orphanix-media-cleanup'),
        ];
    }
}
