<?php
/**
 * Path Validator
 * 
 * Validates all filesystem paths to ensure they're safe
 * COMPLIANCE: Operational Rules 12 (Safety & Security)
 */

class ORPHANIX_Path_Validator {
    
    /**
     * Validate that a path is inside the uploads directory
     * 
     * RULE: No deletion outside uploads
     * 
     * @param string $file_path Path to validate
     * @return true|WP_Error True if valid, WP_Error if invalid
     */
    public static function validate_upload_path( $file_path ) {
        $uploads = wp_get_upload_dir();
        $base = realpath( $uploads['basedir'] );
        $real = realpath( $file_path );

        // Check if realpath() succeeded
        if ( ! $real ) {
            return new WP_Error( 'invalid_path', __( 'Invalid file location.', 'orphanix-media-cleanup' ) );
        }

        // Check if path is inside uploads directory
        if ( strpos( $real, $base ) !== 0 ) {
            return new WP_Error( 'invalid_path', __( 'File is outside uploads directory.', 'orphanix-media-cleanup' ) );
        }

        return true;
    }

    /**
     * Check if a file is a symlink
     * 
     * RULE: Not symlink when doing destructive operations
     * 
     * @param string $file_path Path to check
     * @return bool True if symlink, false otherwise
     */
    public static function is_symlink( $file_path ) {
        return is_link( $file_path );
    }

    /**
     * Validate that a path is safe for destructive operations
     * 
     * Checks:
     * 1. Path is inside uploads directory
     * 2. Path is not a symlink
     * 3. Path exists
     * 
     * RULE: Validates before any deletion
     * 
     * @param string $file_path Path to validate
     * @return true|WP_Error True if valid, WP_Error if invalid
     */
    public static function validate_for_deletion( $file_path ) {
        // Validate path is in uploads
        $upload_check = self::validate_upload_path( $file_path );
        if ( is_wp_error( $upload_check ) ) {
            return $upload_check;
        }

        // Check if path exists
        if ( ! file_exists( $file_path ) ) {
            return new WP_Error( 'file_not_found', __( 'File does not exist.', 'orphanix-media-cleanup' ) );
        }

        // Check if it's a symlink (prevent accidental deletion of target)
        if ( self::is_symlink( $file_path ) ) {
            return new WP_Error( 'is_symlink', __( 'Cannot delete symlinks.', 'orphanix-media-cleanup' ) );
        }

        return true;
    }

    /**
     * Get the real path after validation
     * 
     * @param string $file_path Path to resolve
     * @return string|WP_Error Real path or WP_Error
     */
    public static function get_real_path( $file_path ) {
        $validation = self::validate_upload_path( $file_path );
        if ( is_wp_error( $validation ) ) {
            return $validation;
        }

        return realpath( $file_path );
    }
}

