<?php
/**
 * Result Resolver
 * 
 * Aggregates detector results and resolves final usage status
 */

class ORPHANIX_Result_Resolver {
    
    /**
     * Resolve usage status from multiple detector results
     * 
     * @param array $detector_results Array of results from all detectors
     * @return array {
     *     @type bool $used Final usage status
     *     @type array $used_by Combined list of post IDs
     *     @type array $sources List of detection sources
     * }
     */
    public function resolve( $detector_results ) {
        $used = false;
        $used_by = [];
        $sources = [];

        foreach ( $detector_results as $result ) {
            if ( empty( $result ) || ! is_array( $result ) ) {
                continue;
            }

            // If any detector finds usage, mark as used
            if ( ! empty( $result['used'] ) ) {
                $used = true;
            }

            // Aggregate post IDs
            if ( ! empty( $result['used_by'] ) && is_array( $result['used_by'] ) ) {
                $used_by = array_merge( $used_by, $result['used_by'] );
            }

            // Track detection sources
            if ( ! empty( $result['source'] ) ) {
                $sources[] = $result['source'];
            }
        }

        // Remove duplicates and null values
        $used_by = array_values( array_unique( array_filter( $used_by ) ) );
        $sources = array_values( array_unique( array_filter( $sources ) ) );

        return [
            'used' => $used,
            'used_by' => $used_by,
            'sources' => $sources
        ];
    }

    /**
     * Determine final status string
     * 
     * @param bool $used Whether media is used
     * @param bool $in_media_library Whether file is in media library
     * @return string Status: 'used', 'unused', or 'orphan'
     */
    public function get_status( $used, $in_media_library ) {
        if ( $used ) {
            return 'used';
        }

        if ( ! $in_media_library ) {
            return 'orphan';
        }

        return 'unused';
    }

    /**
     * Format usage context for storage
     * 
     * @param array $used_by Array of post IDs
     * @param array $sources Array of detection sources
     * @return string JSON encoded context
     */
    public function format_usage_context( $used_by, $sources ) {
        return json_encode([
            'post_ids' => $used_by,
            'sources' => $sources,
            'count' => count( $used_by )
        ]);
    }
}
