<?php
/**
 * Detector Registry
 * 
 * Manages conditional loading of detectors based on scan settings
 */

class ORPHANIX_Detector_Registry {
    private $detectors = [];
    private $settings = [];
    private $scan_type = '';

    public function __construct( $scan_type, $settings ) {
        $this->scan_type = $scan_type;
        $this->settings = $settings;
        $this->load_detectors();
    }

    /**
     * Load enabled detectors based on settings
     */
    private function load_detectors() {
        // Core detectors (controlled by scan settings)
        if ( ! empty( $this->settings['featured'] ) ) {
            $this->detectors[] = new ORPHANIX_Featured_Image_Detector();
        }

        if ( ! empty( $this->settings['content'] ) ) {
            $this->detectors[] = new ORPHANIX_Content_Detector();
        }

        if ( ! empty( $this->settings['meta'] ) ) {
            $this->detectors[] = new ORPHANIX_Meta_Detector();
        }

        // Get compatibility settings
        $compat_settings = get_option( 'orphanix_scan_compat', $this->get_default_compat_settings() );

        // Page builder detectors
        if ( ! empty( $compat_settings['scan_gutenberg'] ) && $compat_settings['scan_gutenberg'] ) {
            $this->detectors[] = new ORPHANIX_Gutenberg_Detector();
        }

        if ( ! empty( $compat_settings['scan_elementor'] ) && $compat_settings['scan_elementor'] ) {
            $this->detectors[] = new ORPHANIX_Elementor_Detector();
        }


        // Custom field plugin detectors
        if ( ! empty( $compat_settings['scan_acf'] ) && $compat_settings['scan_acf'] ) {
            $this->detectors[] = new ORPHANIX_ACF_Detector();
        }


        // eCommerce & Media detectors
        if ( ! empty( $compat_settings['scan_woocommerce'] ) && $compat_settings['scan_woocommerce'] ) {
            $this->detectors[] = new ORPHANIX_WooCommerce_Detector();
        }


        // Core WordPress detectors
        if ( ! empty( $compat_settings['scan_widgets'] ) && $compat_settings['scan_widgets'] ) {
            $this->detectors[] = new ORPHANIX_Widget_Detector();
        }

        if ( ! empty( $compat_settings['scan_menus'] ) && $compat_settings['scan_menus'] ) {
            $this->detectors[] = new ORPHANIX_Menu_Detector();
        }

        // Theme file detector (Deep Scan only)
        if ( $this->scan_type === 'deep' && ! empty( $this->settings['theme'] ) && ! empty( $compat_settings['scan_theme_files'] ) && $compat_settings['scan_theme_files'] ) {
            $this->detectors[] = new ORPHANIX_Theme_File_Detector();
        }

        // Deep DB detector (Deep Scan only)
        if ( $this->scan_type === 'deep' && ! empty( $this->settings['deep_scan'] ) ) {
            $this->detectors[] = new ORPHANIX_Deep_DB_Detector();
        }

        // Filter hook for extensibility
        $this->detectors = apply_filters(
            'orphanix_detectors',
            $this->detectors,
            $this->scan_type,
            $this->settings
        );

        // Remove unavailable detectors
        $this->detectors = array_filter( $this->detectors, function( $detector ) {
            return $detector->is_available();
        });
    }

    /**
     * Get default compatibility settings
     * 
     * @return array Default settings with all enabled except theme_files
     */
    private function get_default_compat_settings() {
        return [
            'scan_gutenberg'   => true,
            'scan_elementor'   => true,
            'scan_acf'         => true,
            'scan_woocommerce' => true,
            'scan_widgets'     => true,
            'scan_menus'       => true,
            'scan_theme_files' => false,
        ];
    }

    /**
     * Get all loaded detectors
     * 
     * @return array Array of detector instances
     */
    public function get_detectors() {
        return $this->detectors;
    }

    /**
     * Get detector count
     * 
     * @return int Number of enabled detectors
     */
    public function count() {
        return count( $this->detectors );
    }
}
