<?php
class ORPHANIX_Deep_Scan {
    protected $manager;

    public function __construct() {
        $this->manager = new ORPHANIX_Scan_Manager();
    }

    public function run($settings = []) {
        // Default settings for deep scan (includes theme files)
        $settings = wp_parse_args($settings, [
            'featured' => 1,
            'content' => 1,
            'custom' => 1,
            'theme' => 0,
            'deep_scan' => 1,
        ]);

        $scan_id = $this->manager->start_scan('deep','media',$settings);

        $registry_settings = [
            'featured' => ! empty($settings['featured']),
            'content' => ! empty($settings['content']),
            'meta' => ! empty($settings['custom']),
            'theme' => ! empty($settings['theme']),
            'deep_scan' => true,
        ];

        // Initialize detector registry with settings
        $registry = new ORPHANIX_Detector_Registry('deep', $registry_settings);
        $detectors = $registry->get_detectors();
        $resolver = new ORPHANIX_Result_Resolver();

        $attachments = get_posts([
            'post_type' => 'attachment',
            'posts_per_page' => -1,
            'post_status' => 'inherit',
            'fields' => 'ids',
        ]);

        $processed = 0; $used = 0; $orphan = 0; $unused = 0;

        foreach ($attachments as $attachment_id) {
            $file = get_attached_file($attachment_id);
            $file_url = wp_get_attachment_url($attachment_id);

            if (!$file || !file_exists($file)) {
                $status = 'orphan';
                $usage_context = [ 'issues' => ['file_missing'] ];
                $file_size = 0;
                $orphan++;
            } else {
                $resolved = $this->resolve_usage($attachment_id, $file_url, $detectors, $resolver);
                $status = $resolved['used'] ? 'used' : 'not_used';
                $usage_context = [
                    'post_ids' => $resolved['used_by'],
                    'sources' => $resolved['sources'],
                    'count' => count($resolved['used_by']),
                ];
                $file_size = filesize($file);
                if ($status === 'used') {
                    $used++;
                } else {
                    $unused++;
                }
            }

            $this->manager->add_item($scan_id, [
                'attachment_id' => $attachment_id,
                'file_path'     => $file,
                'file_url'      => $file_url,
                'file_size'     => $file_size,
                'alt_text'      => get_post_meta($attachment_id,'_wp_attachment_image_alt',true),
                'directory_type'=> 'media',
                'usage_context' => $usage_context,
                'status'        => $status,
            ]);

            $processed++;
        }

        $this->manager->complete_scan($scan_id, [
            'processed' => $processed,
            'used'      => $used,
            'orphan'    => $orphan + $unused,
        ]);

        return $scan_id;
    }

    public function run_async($scan_id, $settings = []) {
        global $wpdb;

        // Default settings for deep scan
        $settings = wp_parse_args($settings, [
            'featured' => 1,
            'content' => 1,
            'custom' => 1,
            'theme' => 0,
            'deep_scan' => 1,
        ]);

        $registry_settings = [
            'featured' => ! empty($settings['featured']),
            'content' => ! empty($settings['content']),
            'meta' => ! empty($settings['custom']),
            'theme' => ! empty($settings['theme']),
            'deep_scan' => true,
        ];

        if ( class_exists( 'ORPHANIX_Logger' ) ) {
            ORPHANIX_Logger::log( 'Deep Scan Starting', [
                'scan_id' => $scan_id,
                'settings' => $settings,
            ] );
        }

        // Initialize detector registry with settings
        $registry = new ORPHANIX_Detector_Registry('deep', $registry_settings);
        $detectors = $registry->get_detectors();
        $resolver = new ORPHANIX_Result_Resolver();

        if ( class_exists( 'ORPHANIX_Logger' ) ) {
            ORPHANIX_Logger::log( 'Deep Scan - Detectors loaded', [ 'count' => count($detectors) ] );
        }
        
        $attachments = get_posts([
            'post_type' => 'attachment',
            'posts_per_page' => -1,
            'post_status' => 'inherit',
            'fields' => 'ids',
        ]);

        $total = count($attachments);
        $started_at = current_time('timestamp');
        $processed = 0;
        $used = 0;
        $orphan = 0;
        $unused = 0;
        $in_library = 0;
        $used_in_library = 0;
        $used_not_in_library = 0;

        // Update scan with total files
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->update("{$wpdb->prefix}orphanix_scans", [
            'total_files' => $total,
        ], ['id' => $scan_id]);

        foreach ($attachments as $attachment_id) {
            $file = get_attached_file($attachment_id);
            $file_url = wp_get_attachment_url($attachment_id);

            $in_library++;

            if (!$file || !file_exists($file)) {
                $status = 'orphan';
                $usage_context = [ 'issues' => ['file_missing'] ];
                $file_size = 0;
                $orphan++;
            } else {
                $resolved = $this->resolve_usage($attachment_id, $file_url, $detectors, $resolver);
                $status = $resolved['used'] ? 'used' : 'not_used';
                $usage_context = [
                    'post_ids' => $resolved['used_by'],
                    'sources' => $resolved['sources'],
                    'count' => count($resolved['used_by']),
                ];
                $file_size = filesize($file);
                if ($status === 'used') {
                    $used++;
                    $used_in_library++;
                } else {
                    $unused++;
                }
            }

            $this->manager->add_item($scan_id, [
                'attachment_id' => $attachment_id,
                'file_path'     => $file,
                'file_url'      => $file_url,
                'file_size'     => $file_size,
                'alt_text'      => get_post_meta($attachment_id,'_wp_attachment_image_alt',true),
                'directory_type'=> 'media',
                'usage_context' => $usage_context,
                'status'        => $status,
            ]);

            $processed++;

            // Update progress
            $percentage = round(($processed / $total) * 100);
            set_transient('orphanix_scan_' . $scan_id . '_progress', [
                'scan_id' => $scan_id,
                'status' => 'running',
                'total' => $total,
                'processed' => $processed,
                'percentage' => $percentage,
                'current_file' => $file ? basename($file) : '',
                'started_at' => $started_at,
                'used' => $used,
                'orphan' => $orphan + $unused,
                'in_library' => $in_library,
                'used_in_library' => $used_in_library,
                'used_not_in_library' => $used_not_in_library,
            ], 3600);

            // Update database progress
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->update("{$wpdb->prefix}orphanix_scans", [
                'processed_files' => $processed,
                'used_files' => $used,
                'orphan_files' => $orphan + $unused,
            ], ['id' => $scan_id]);
        }

        $this->manager->complete_scan($scan_id, [
            'processed' => $processed,
            'used'      => $used,
            'orphan'    => $orphan + $unused,
        ]);

        if ( class_exists( 'ORPHANIX_Logger' ) ) {
            ORPHANIX_Logger::log( 'Deep Scan Completed', [
                'scan_id' => $scan_id,
                'total' => $total,
                'used' => $used,
                'orphan' => $orphan,
            ] );
        }

        // Update final progress
        set_transient('orphanix_scan_' . $scan_id . '_progress', [
            'scan_id' => $scan_id,
            'status' => 'completed',
            'total' => $total,
            'processed' => $processed,
            'percentage' => 100,
            'current_file' => '',
            'started_at' => $started_at,
            'used' => $used,
            'orphan' => $orphan + $unused,
            'in_library' => $in_library,
            'used_in_library' => $used_in_library,
            'used_not_in_library' => $used_not_in_library,
        ], 3600);

        return $scan_id;
    }


    private function is_used_by_detectors($file, $attachment_id, $detectors = []) {
        // Backward-compatible alias
        $file_url = $attachment_id ? wp_get_attachment_url($attachment_id) : $file;
        $resolved = $this->resolve_usage($attachment_id, $file_url, $detectors, new ORPHANIX_Result_Resolver());
        return $resolved['used'];
    }

    private function resolve_usage($attachment_id, $file_url, $detectors = [], $resolver = null) {
        if (!$resolver) {
            $resolver = new ORPHANIX_Result_Resolver();
        }

        if (empty($detectors)) {
            global $wpdb;
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_content LIKE %s",
                '%' . $wpdb->esc_like($file_url) . '%'
            ));
            return [
                'used' => $count > 0,
                'used_by' => [],
                'sources' => $count > 0 ? ['post_content'] : []
            ];
        }

        $detector_results = [];
        foreach ($detectors as $detector) {
            $detector_results[] = $detector->detect($attachment_id, $file_url);
        }

        return $resolver->resolve($detector_results);
    }

    private function scan_directory($dir) {
        $rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($dir));
        $files = [];
        foreach ($rii as $file) {
            if ($file->isDir()) continue;
            $files[] = $file->getPathname();
        }
        return $files;
    }
}

