<?php
class ORPHANIX_Broken_Scan {
    protected $manager;
    protected $scan_mode = 'regular'; // 'regular' or 'deep'
    protected $settings = [];

    public function __construct() {
        $this->manager = new ORPHANIX_Scan_Manager();
    }

    /**
     * Run broken media scan with specified settings
     * 
     * @param array $settings Scan configuration
     * @return int Scan ID
     */
    public function run($settings = []) {
        $this->settings = wp_parse_args($settings, [
            'scan_featured' => true,
            'scan_content' => false,
            'scan_custom_fields' => false,
            'scan_image_sizes' => false,
            'scan_page_builders' => false,
            'scan_theme_files' => false,
            'http_validation' => false,
        ]);

        $this->scan_mode = isset($settings['scan_mode']) ? $settings['scan_mode'] : 'regular';

        $scan_id = $this->manager->start_scan($this->scan_mode, 'broken', $this->settings);

        if ($this->scan_mode === 'deep') {
            $this->run_deep_scan($scan_id);
        } else {
            $this->run_regular_scan($scan_id);
        }

        $this->manager->complete_scan($scan_id, $this->get_final_stats($scan_id));

        return $scan_id;
    }

    /**
     * Run regular broken scan (Media Library only)
     */
    protected function run_regular_scan($scan_id) {
        $attachments = get_posts([
            'post_type' => 'attachment',
            'posts_per_page' => -1,
            'post_status' => 'inherit',
        ]);

        foreach ($attachments as $attachment) {
            $this->check_attachment($scan_id, $attachment);
        }
    }

    /**
     * Run deep broken scan (Media Library + Filesystem)
     */
    protected function run_deep_scan($scan_id) {
        // First scan Media Library
        $this->run_regular_scan($scan_id);

        // Then scan filesystem for orphans
        if ($this->settings['scan_theme_files']) {
            $this->scan_filesystem_orphans($scan_id);
        }
    }

    /**
     * Check individual attachment for broken status
     */
    protected function check_attachment($scan_id, $attachment) {
        $file_path = get_attached_file($attachment->ID);
        $metadata = wp_get_attachment_metadata($attachment->ID);
        $status = 'valid';
        $issues = [];
        $usage_context = [
            'issues' => [],
            'usage' => [
                'post_ids' => [],
                'sources' => [],
                'count' => 0,
            ],
        ];

        // Check if original file exists
        if (!file_exists($file_path)) {
            $status = 'broken';
            $issues[] = 'Original file missing';
        } elseif (!is_readable($file_path)) {
            $status = 'broken';
            $issues[] = 'File not readable';
        }

        // Check generated image sizes (if enabled)
        if ($this->settings['scan_image_sizes'] && !empty($metadata['sizes'])) {
            $missing_sizes = $this->check_image_sizes($file_path, $metadata);
            if (!empty($missing_sizes)) {
                if ($status === 'valid') {
                    $status = 'broken_partial';
                }
                $issues[] = 'Missing sizes: ' . implode(', ', $missing_sizes);
            }
        }

        // HTTP validation (if enabled)
        if ($this->settings['http_validation'] && $status === 'valid') {
            $url = wp_get_attachment_url($attachment->ID);
            if (!$this->validate_http_url($url)) {
                $status = 'broken';
                $issues[] = 'URL returns error (404/410)';
            }
        }

        // Check if referenced in content/builders/meta (Deep Scan only)
        if ($this->scan_mode === 'deep') {
            $scan_content = !empty($this->settings['scan_content']);
            $scan_custom = !empty($this->settings['scan_custom_fields']);
            $scan_page_builders = !empty($this->settings['scan_page_builders']);

            if ($scan_content || $scan_custom || $scan_page_builders) {
                $registry_settings = [
                    'featured' => !empty($this->settings['scan_featured']),
                    'content' => $scan_content,
                    'meta' => $scan_custom,
                    'theme' => !empty($this->settings['scan_theme_files']),
                    'deep_scan' => true,
                ];

                $registry = new ORPHANIX_Detector_Registry('deep', $registry_settings);
                $detectors = $registry->get_detectors();

                // Filter detectors based on selected groups
                $detectors = array_filter($detectors, function($detector) use ($scan_content, $scan_custom, $scan_page_builders) {
                    $class = get_class($detector);
                    if (!$scan_page_builders && in_array($class, ['ORPHANIX_Gutenberg_Detector','ORPHANIX_Elementor_Detector','ORPHANIX_Divi_Detector','ORPHANIX_WPBakery_Detector'], true)) {
                        return false;
                    }
                    if (!$scan_custom && in_array($class, ['ORPHANIX_ACF_Detector','ORPHANIX_MetaBox_Detector','ORPHANIX_Pods_Detector'], true)) {
                        return false;
                    }
                    if (!$scan_content && in_array($class, ['ORPHANIX_WooCommerce_Detector','ORPHANIX_Sliders_Detector','ORPHANIX_Widget_Detector','ORPHANIX_Menu_Detector','ORPHANIX_Content_Detector'], true)) {
                        return false;
                    }
                    return true;
                });

                $resolver = new ORPHANIX_Result_Resolver();
                $detector_results = [];
                $file_url = wp_get_attachment_url($attachment->ID);

                foreach ($detectors as $detector) {
                    $detector_results[] = $detector->detect($attachment->ID, $file_url);
                }

                $resolved = $resolver->resolve($detector_results);
                $usage_context['usage'] = [
                    'post_ids' => $resolved['used_by'],
                    'sources' => $resolved['sources'],
                    'count' => count($resolved['used_by']),
                ];

                if (!empty($resolved['sources'])) {
                    $issues[] = 'Referenced in: ' . implode(', ', $resolved['sources']);
                    if ($status !== 'valid') {
                        $issues[] = 'Referenced but file missing';
                    }
                }
            }
        }

        $usage_context['issues'] = $issues;

        $this->manager->add_item($scan_id, [
            'attachment_id' => $attachment->ID,
            'file_path' => $file_path,
            'file_url' => wp_get_attachment_url($attachment->ID),
            'file_size' => file_exists($file_path) ? filesize($file_path) : 0,
            'alt_text' => get_post_meta($attachment->ID, '_wp_attachment_image_alt', true),
            'directory_type' => 'media_library',
            'usage_context' => $usage_context,
            'status' => $status,
        ]);
    }

    /**
     * Check if all generated image sizes exist
     */
    protected function check_image_sizes($original_file, $metadata) {
        $missing = [];
        
        if (empty($metadata['sizes'])) {
            return $missing;
        }

        $upload_dir = wp_upload_dir();
        $base_dir = dirname($original_file);

        foreach ($metadata['sizes'] as $size_name => $size_data) {
            $size_file = $base_dir . '/' . $size_data['file'];
            if (!file_exists($size_file)) {
                $missing[] = $size_name;
            }
        }

        return $missing;
    }

    /**
     * Validate if URL is accessible via HTTP
     */
    protected function validate_http_url($url) {
        $response = wp_remote_head($url, [
            'timeout' => 5,
            'redirection' => 5,
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        $code = wp_remote_retrieve_response_code($response);
        return $code >= 200 && $code < 400;
    }

    /**
     * Check if attachment is referenced in post content
     */
    protected function check_content_references($attachment_id) {
        global $wpdb;

        $url = wp_get_attachment_url($attachment_id);
        $filename = basename($url);

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_status = 'publish' 
            AND (post_content LIKE %s OR post_content LIKE %s)",
            '%' . $wpdb->esc_like($url) . '%',
            '%' . $wpdb->esc_like($filename) . '%'
        ));

        return $count > 0 ? ['post_content'] : [];
    }

    /**
     * Scan filesystem for orphaned broken files
     */
    protected function scan_filesystem_orphans($scan_id) {
        $upload_dir = wp_upload_dir();
        $base_path = $upload_dir['basedir'];

        // Recursively scan uploads directory
        $files = $this->scan_directory($base_path);

        foreach ($files as $file) {
            // Check if file has corresponding attachment
            $attachment_id = attachment_url_to_postid($file);
            
            if (!$attachment_id) {
                // Orphan file - check if it's broken
                if (!is_readable($file)) {
                    $this->manager->add_item($scan_id, [
                        'attachment_id' => null,
                        'file_path' => $file,
                        'file_url' => str_replace($base_path, $upload_dir['baseurl'], $file),
                        'file_size' => filesize($file),
                        'alt_text' => '',
                        'directory_type' => 'filesystem',
                        'usage_context' => ['Orphan file - not readable'],
                        'status' => 'orphan_broken',
                    ]);
                }
            }
        }
    }

    /**
     * Recursively scan directory for files
     */
    protected function scan_directory($dir, &$results = []) {
        if (!is_dir($dir)) {
            return $results;
        }

        $files = scandir($dir);
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }

            $path = $dir . '/' . $file;

            if (is_dir($path)) {
                $this->scan_directory($path, $results);
            } else {
                // Only check media file types
                $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
                $media_exts = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'pdf', 'mp4', 'mp3', 'svg'];
                
                if (in_array($ext, $media_exts)) {
                    $results[] = $path;
                }
            }
        }

        return $results;
    }

    /**
     * Get final scan statistics
     */
    protected function get_final_stats($scan_id) {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $stats = $wpdb->get_row($wpdb->prepare(
            "SELECT 
                COUNT(*) as processed,
                SUM(CASE WHEN status = 'valid' THEN 1 ELSE 0 END) as valid,
                SUM(CASE WHEN status = 'broken' THEN 1 ELSE 0 END) as broken,
                SUM(CASE WHEN status = 'broken_partial' THEN 1 ELSE 0 END) as broken_partial,
                SUM(CASE WHEN status LIKE %s THEN 1 ELSE 0 END) as orphan
            FROM {$wpdb->prefix}orphanix_scan_items
            WHERE scan_id = %d",
            '%orphan%',
            $scan_id
        ), ARRAY_A);

        return [
            'processed' => (int) $stats['processed'],
            'used' => (int) $stats['valid'],
            'broken' => (int) ($stats['broken'] + $stats['broken_partial']),
            'orphan' => (int) $stats['orphan'],
        ];
    }
}


