<?php
/**
 * Elementor Detector
 * 
 * Detects media in Elementor page builder data
 * Performance: Medium (JSON parsing)
 */

class ORPHANIX_Elementor_Detector implements ORPHANIX_Usage_Detector {
    
    public function detect( $attachment_id, $file_url ) {
        global $wpdb;

        if ( ! $this->is_available() ) {
            return [
                'used' => false,
                'used_by' => [],
                'source' => 'elementor'
            ];
        }

        $used_by = [];
        $filename = basename( $file_url );

        // Search for Elementor data containing the attachment ID or filename
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $results = $wpdb->get_results( $wpdb->prepare(
            "SELECT post_id, meta_value FROM {$wpdb->postmeta} 
             WHERE meta_key = '_elementor_data' 
             AND (meta_value LIKE %s OR meta_value LIKE %s)",
            '%"id":' . intval( $attachment_id ) . '%',
            '%' . $wpdb->esc_like( $filename ) . '%'
        ));

        foreach ( $results as $row ) {
            // Verify the match by parsing JSON
            $data = json_decode( $row->meta_value, true );
            if ( $data && $this->search_elementor_data( $data, $attachment_id, $filename ) ) {
                $used_by[] = $row->post_id;
            }
        }

        return [
            'used' => ! empty( $used_by ),
            'used_by' => array_unique( $used_by ),
            'source' => 'elementor'
        ];
    }

    /**
     * Recursively search Elementor data structure
     */
    private function search_elementor_data( $data, $attachment_id, $filename ) {
        if ( ! is_array( $data ) ) {
            return false;
        }

        foreach ( $data as $key => $value ) {
            // Check for ID match
            if ( $key === 'id' && intval( $value ) === intval( $attachment_id ) ) {
                return true;
            }

            // Check for URL match
            if ( is_string( $value ) && strpos( $value, $filename ) !== false ) {
                return true;
            }

            // Recursive check
            if ( is_array( $value ) && $this->search_elementor_data( $value, $attachment_id, $filename ) ) {
                return true;
            }
        }

        return false;
    }

    public function get_name() {
        return 'Elementor Detector';
    }

    public function is_available() {
        return defined( 'ELEMENTOR_VERSION' );
    }
}
