<?php
/**
 * Deep DB Detector
 *
 * Scans all database tables for media usage patterns (Deep Scan only)
 * Performance: Slow (full table scans)
 */
class ORPHANIX_Deep_DB_Detector implements ORPHANIX_Usage_Detector {

    public function detect( $attachment_id, $file_url ) {
        global $wpdb;

        if ( empty( $file_url ) ) {
            return [
                'used' => false,
                'used_by' => [],
                'source' => 'deep_db'
            ];
        }

        $patterns = array_filter([
            $file_url,
            basename($file_url),
        ]);

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $tables = $wpdb->get_col('SHOW TABLES');
        $used_by = [];

        foreach ( $tables as $table ) {
            $validated_table = $this->validate_table_identifier( $table, $wpdb->prefix );
            if ( ! $validated_table ) {
                continue;
            }

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $columns = $wpdb->get_results( "SHOW COLUMNS FROM `{$validated_table}`" );
            if ( empty( $columns ) ) {
                continue;
            }

            foreach ( $columns as $column ) {
                $validated_column = $this->validate_column_identifier( $column->Field );
                if ( ! $validated_column ) {
                    continue;
                }

                $type = strtolower( $column->Type );
                if ( strpos( $type, 'char' ) === false && strpos( $type, 'text' ) === false ) {
                    continue;
                }

                foreach ( $patterns as $pattern ) {
                    // phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
                    $sql = 'SELECT 1 FROM `' . $validated_table . '` WHERE `' . $validated_column . '` LIKE %s LIMIT 1';
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter
                    $found = $wpdb->get_var(
                        $wpdb->prepare(
                            $sql,
                            '%' . $wpdb->esc_like( $pattern ) . '%'
                        )
                    );
                    // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared

                    if ( $found ) {
                        $used_by[] = $validated_table;
                        break 2;
                    }
                }
            }
        }

        $used_by = array_values( array_unique( $used_by ) );

        return [
            'used' => ! empty( $used_by ),
            'used_by' => $used_by,
            'source' => 'deep_db'
        ];
    }

    public function get_name() {
        return 'Deep Database Detector';
    }

    public function is_available() {
        return true;
    }

    /**
     * Allow only prefixed table names with simple SQL identifier characters.
     */
    private function validate_table_identifier( $table_name, $prefix ) {
        $table_name = is_string( $table_name ) ? $table_name : '';
        if ( '' === $table_name ) {
            return '';
        }
        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $table_name ) ) {
            return '';
        }
        if ( 0 !== strpos( $table_name, $prefix ) ) {
            return '';
        }
        return $table_name;
    }

    /**
     * Allow only simple SQL identifier characters for column names.
     */
    private function validate_column_identifier( $column_name ) {
        $column_name = is_string( $column_name ) ? $column_name : '';
        if ( '' === $column_name ) {
            return '';
        }
        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $column_name ) ) {
            return '';
        }
        return $column_name;
    }
}

