<?php
/**
 * Trash Repository
 * 
 * Manages trashed files and provides recovery functionality
 * COMPLIANCE: Operational Rules 9.2 (Move to Trash), 11 (Restore Rules)
 */

class ORPHANIX_Trash_Repository extends ORPHANIX_DB {
    
    /**
     * Add a file to trash
     * 
     * Validates path safety before moving
     * 
     * @param int $scan_item_id Scan item ID
     * @param string $original_path Original file path
     * @param string $trash_path Unused legacy parameter kept for backward compatibility
     * @param string $expires_at Expiration date
     * @return int|WP_Error Insert ID or WP_Error
     */
    public function add_to_trash( $scan_item_id, $original_path, $trash_path, $expires_at ) {
        // RULE: Validate path before any deletion/movement
        $validation = ORPHANIX_Path_Validator::validate_for_deletion( $original_path );
        if ( is_wp_error( $validation ) ) {
            return $validation;
        }

        // Store all plugin-created files under uploads/orphanix-media-cleanup/trash.
        $trash_dir  = ORPHANIX_Paths::ensure_subdir( 'trash' );
        $trash_file = wp_unique_filename( $trash_dir, basename( $original_path ) );
        $trash_path = trailingslashit( $trash_dir ) . $trash_file;

        // Safety check: target must be inside uploads directory.
        $upload_dir    = wp_upload_dir();
        $uploads_base  = trailingslashit( wp_normalize_path( $upload_dir['basedir'] ) );
        $target_path   = wp_normalize_path( $trash_path );
        if ( strpos( $target_path, $uploads_base ) !== 0 ) {
            return new WP_Error( 'orphanix_invalid_path', __( 'Invalid file path.', 'orphanix-media-cleanup' ) );
        }

        $filesystem = $this->get_filesystem();
        // Move file to trash
        if ( ! $filesystem || ! $filesystem->move( $original_path, $trash_path, true ) ) {
            return new WP_Error( 'move_failed', __( 'Failed to move file to trash.', 'orphanix-media-cleanup' ) );
        }

        // Record in database
        $result = $this->wpdb->insert( $this->table( 'trash' ), [
            'scan_item_id' => $scan_item_id,
            'original_path' => $original_path,
            'trash_path'    => $trash_path,
            'deleted_at'    => current_time( 'mysql' ),
            'expires_at'    => $expires_at,
        ]);

        if ( ! $result ) {
            // Restore file if database insert failed
            if ( $filesystem ) {
                $filesystem->move( $trash_path, $original_path, true );
            }
            return new WP_Error( 'db_insert_failed', __( 'Failed to record trash entry.', 'orphanix-media-cleanup' ) );
        }

        return $this->wpdb->insert_id;
    }

    /**
     * Restore a file from trash
     * 
     * RULE: Filesystem restore moves file back to original path + removes record
     * 
     * @param int $id Trash record ID
     * @return bool True if restored, false otherwise
     */
    public function restore( $id ) {
        $table = $this->table( 'trash' );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $item = $this->wpdb->get_row( $this->wpdb->prepare( "SELECT * FROM {$table} WHERE id = %d", $id ), ARRAY_A );

        if ( ! $item ) {
            return false;
        }

        // Validate trash path is safe
        $validation = ORPHANIX_Path_Validator::validate_upload_path( $item['trash_path'] );
        if ( is_wp_error( $validation ) ) {
            return false;
        }

        // Validate original path is safe
        $validation = ORPHANIX_Path_Validator::validate_upload_path( $item['original_path'] );
        if ( is_wp_error( $validation ) ) {
            return false;
        }

        // Move file back to original location
        if ( ! file_exists( $item['trash_path'] ) ) {
            // File already deleted, just remove record
            $this->wpdb->delete( $this->table( 'trash' ), [ 'id' => $id ] );
            return false;
        }

        $filesystem = $this->get_filesystem();
        if ( ! $filesystem || ! $filesystem->move( $item['trash_path'], $item['original_path'], true ) ) {
            return false;
        }

        // Remove trash record
        $this->wpdb->delete( $this->table( 'trash' ), [ 'id' => $id ] );
        return true;
    }

    /**
     * Get expired trash items
     * 
     * @return array Expired items
     */
    public function expired_items() {
        $table = $this->table( 'trash' );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $this->wpdb->get_results( $this->wpdb->prepare( "SELECT * FROM {$table} WHERE expires_at < %s", current_time( 'mysql' ) ), ARRAY_A );
    }

    /**
     * Purge expired items from trash
     * 
     * RULE: Permanent deletion only for truly expired items
     * 
     * @return int Number of items purged
     */
    public function purge_expired() {
        $items = $this->expired_items();
        $purged = 0;

        foreach ( $items as $item ) {
            // Validate path before deletion
            $validation = ORPHANIX_Path_Validator::validate_upload_path( $item['trash_path'] );
            if ( is_wp_error( $validation ) ) {
                // Path is invalid, just remove record
                $this->wpdb->delete( $this->table( 'trash' ), [ 'id' => $item['id'] ] );
                $purged++;
                continue;
            }

            // Delete file
            if ( file_exists( $item['trash_path'] ) && ! is_link( $item['trash_path'] ) ) {
                wp_delete_file( $item['trash_path'] );
            }

            // Remove record
            $this->wpdb->delete( $this->table( 'trash' ), [ 'id' => $item['id'] ] );
            $purged++;
        }

        return $purged;
    }

    /**
     * Get trash items for a specific scan
     * 
     * @param int $scan_id Scan ID
     * @return array Trash items
     */
    public function get_by_scan( $scan_id ) {
        $trash_table = $this->table( 'trash' );
        $items_table = $this->table( 'scan_items' );
        // phpcs:disable WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.PreparedSQL.InterpolatedNotPrepared,PluginCheck.Security.DirectDB.UnescapedDBParameter
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        return $this->wpdb->get_results(
            $this->wpdb->prepare(
                "SELECT t.* FROM {$trash_table} t
             INNER JOIN {$items_table} si ON t.scan_item_id = si.id
             WHERE si.scan_id = %d",
                $scan_id
            ),
            ARRAY_A
        );
        // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.PreparedSQL.InterpolatedNotPrepared,PluginCheck.Security.DirectDB.UnescapedDBParameter
    }

    /**
     * Initialize WP_Filesystem and return it.
     */
    private function get_filesystem() {
        global $wp_filesystem;
        if ( empty( $wp_filesystem ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }
        return $wp_filesystem;
    }
}
