<?php
/**
 * Scan Items Repository
 * 
 * Manages scan result items in the database
 * COMPLIANCE: Operational Rules 3 (Scan Result Statuses) & 9.1 (Delete Action)
 */
class ORPHANIX_Scan_Items_Repository extends ORPHANIX_DB {
    
    /**
     * Get all items for a scan
     */
    public function all_items($scan_id) {
        $table = $this->table('scan_items');
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $this->wpdb->get_results( $this->wpdb->prepare( "SELECT * FROM {$table} WHERE scan_id = %d", $scan_id ), ARRAY_A );
    }

    /**
     * Filter items by status
     */
    public function filter_items($scan_id, $status) {
        $table = $this->table('scan_items');
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $this->wpdb->get_results( $this->wpdb->prepare( "SELECT * FROM {$table} WHERE scan_id = %d AND status = %s", $scan_id, $status ), ARRAY_A );
    }

    /**
     * Get single scan item by ID
     */
    public function get_item($id) {
        $table = $this->table('scan_items');
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        return $this->wpdb->get_row( $this->wpdb->prepare( "SELECT * FROM {$table} WHERE id = %d", $id ), ARRAY_A );
    }

    /**
     * Delete scan result only (scan results table only)
     * 
     * @deprecated Use delete_item_with_attachment() instead
     */
    public function delete_item($id) {
        return $this->wpdb->delete($this->table('scan_items'), ['id' => $id]);
    }

    /**
     * Delete scan item AND its attachment from Media Library and filesystem
     * 
     * RULE 9.1: Delete action removes from Media Library AND filesystem
     * 
     * Deletes:
     * 1. Scan result from database
     * 2. Attachment from WordPress (wp_delete_attachment)
     * 3. File from filesystem
     * 
     * @param int $item_id Scan item ID
     * @return true|WP_Error True on success, WP_Error on failure
     */
    public function delete_item_with_attachment($item_id) {
        // Get scan item details
        $item = $this->get_item($item_id);
        if ( ! $item ) {
            return new WP_Error('item_not_found', __('Scan item not found.', 'orphanix-media-cleanup'));
        }

        $attachment_id = intval($item['attachment_id'] ?? 0);
        $file_path = isset($item['file_path']) ? trim($item['file_path']) : '';

            // Delete attachment from WordPress Media Library FIRST (if attachment ID exists)
            // This will delete the original file + ALL image sizes automatically
        if ( $attachment_id > 0 ) {
                // wp_delete_attachment with force_delete=true deletes:
                // - Attachment post from database
                // - Original file from filesystem
                // - ALL generated image sizes (thumbnail, medium, large, etc.)
                // - Attachment metadata
            $result = wp_delete_attachment($attachment_id, true);
                if ( $result === false ) {
                    return new WP_Error('attachment_delete_failed',
                        __('Failed to delete attachment from Media Library.', 'orphanix-media-cleanup'));
            }
            } else {
                // No attachment - manually delete orphan file and its variants
                if ( ! empty($file_path) ) {
                    // Validate path before deletion
                    if ( class_exists('ORPHANIX_Path_Validator') ) {
                        $validation = ORPHANIX_Path_Validator::validate_for_deletion($file_path);
                        if ( is_wp_error($validation) ) {
                            return $validation;
                        }
                    }
                
                    // Delete original file
                    if ( file_exists( $file_path ) ) {
                        wp_delete_file( $file_path );
                    }
                
                    // Try to delete common image size variants for orphan files
                    $this->delete_orphan_image_sizes($file_path);
                }
        }

        // Delete scan result from database (always delete if we got this far)
        $db_result = $this->wpdb->delete($this->table('scan_items'), ['id' => $item_id]);
        if ( ! $db_result ) {
            return new WP_Error('scan_result_delete_failed',
                __('Failed to remove scan result from database.', 'orphanix-media-cleanup'));
        }

        // Success - at least the scan result was deleted
        return true;
    }

    /**
     * Delete common image size variants for orphan files
     * 
     * For orphan files (not in Media Library), manually delete common size variants
     * 
     * @param string $original_file Path to original file
     */
    private function delete_orphan_image_sizes($original_file) {
        if ( ! file_exists($original_file) ) {
            return;
        }

        $path_info = pathinfo($original_file);
        $dir = $path_info['dirname'];
        $filename = $path_info['filename'];
        $extension = isset($path_info['extension']) ? $path_info['extension'] : '';

        // Common WordPress image sizes
        $sizes = [
            '-150x150',    // thumbnail
            '-300x300',    // medium
            '-1024x1024',  // large
            '-scaled',     // scaled version
        ];

        // Try to delete each size variant
        foreach ($sizes as $size) {
            $size_file = $dir . '/' . $filename . $size . '.' . $extension;
            if ( file_exists( $size_file ) ) {
                wp_delete_file( $size_file );
            }
        }

        // Also try to find any other variants with pattern filename-[digits]x[digits].ext
        $pattern = $dir . '/' . $filename . '-*.' . $extension;
        $variants = glob($pattern);
        if ( is_array($variants) ) {
            foreach ($variants as $variant) {
                if ( file_exists( $variant ) ) {
                    wp_delete_file( $variant );
                }
            }
        }
    }
}
