<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Audit Logger
 * 
 * Logs all critical operations for compliance and debugging
 * COMPLIANCE: Operational Rules 12 (Safety & Security) - Audit logging recommended
 */

class ORPHANIX_Audit_Logger {
    
    /**
     * Log levels
     */
    const LOG_DELETE = 'delete';
    const LOG_TRASH = 'trash';
    const LOG_RESTORE = 'restore';
    const LOG_SCAN_START = 'scan_start';
    const LOG_SCAN_COMPLETE = 'scan_complete';
    const LOG_SECURITY_VIOLATION = 'security_violation';

    /**
     * Log an audit event
     * 
     * @param string $action Action type (use class constants)
     * @param array $data Event data
     * @param string $severity Log severity
     */
    public static function log( $action, $data = [], $severity = 'INFO' ) {
        $user = wp_get_current_user();
        
        $log_entry = [
            'timestamp' => current_time( 'mysql' ),
            'action' => $action,
            'user_id' => $user->ID ?? 0,
            'user_login' => $user->user_login ?? 'unknown',
            'user_ip' => self::get_client_ip(),
            'severity' => $severity,
            'data' => $data,
        ];

        // Log to file
        self::write_log_file( $log_entry );

        // Optionally log to database (for future audit table)
        if ( apply_filters( 'orphanix_log_to_database', false ) ) {
            self::write_to_database( $log_entry );
        }

        // Log security violations to error log
        if ( 'security_violation' === $action && class_exists( 'ORPHANIX_Logger' ) ) {
            ORPHANIX_Logger::log( 'Security Violation', $log_entry );
        }
    }

    /**
     * Log a file deletion
     * 
     * @param int $attachment_id Attachment ID (if applicable)
     * @param string $file_path File path
     * @param bool $permanent Whether it's permanent deletion
     */
    public static function log_delete( $attachment_id, $file_path, $permanent = false ) {
        self::log(
            $permanent ? self::LOG_DELETE : self::LOG_TRASH,
            [
                'attachment_id' => $attachment_id,
                'file_path' => $file_path,
                'permanent' => $permanent,
                'action_type' => $permanent ? 'permanent_delete' : 'move_to_trash',
            ]
        );
    }

    /**
     * Log a file restoration
     * 
     * @param int $trash_id Trash record ID
     * @param string $file_path File path
     */
    public static function log_restore( $trash_id, $file_path ) {
        self::log(
            self::LOG_RESTORE,
            [
                'trash_id' => $trash_id,
                'file_path' => $file_path,
            ]
        );
    }

    /**
     * Log scan start
     * 
     * @param int $scan_id Scan ID
     * @param string $scan_type Scan type (regular/deep)
     * @param array $settings Scan settings
     */
    public static function log_scan_start( $scan_id, $scan_type, $settings = [] ) {
        self::log(
            self::LOG_SCAN_START,
            [
                'scan_id' => $scan_id,
                'scan_type' => $scan_type,
                'settings' => $settings,
            ]
        );
    }

    /**
     * Log scan completion
     * 
     * @param int $scan_id Scan ID
     * @param array $stats Scan statistics
     */
    public static function log_scan_complete( $scan_id, $stats = [] ) {
        self::log(
            self::LOG_SCAN_COMPLETE,
            [
                'scan_id' => $scan_id,
                'stats' => $stats,
            ]
        );
    }

    /**
     * Log security violation
     * 
     * @param string $violation_type Type of violation
     * @param array $details Violation details
     */
    public static function log_security_violation( $violation_type, $details = [] ) {
        self::log(
            self::LOG_SECURITY_VIOLATION,
            array_merge(
                [ 'violation_type' => $violation_type ],
                $details
            ),
            'ERROR'
        );
    }

    /**
     * Write log entry to file
     * 
     * @param array $log_entry Log entry array
     */
    private static function write_log_file( $log_entry ) {
        $filesystem = self::get_filesystem();
        if ( ! $filesystem ) {
            return;
        }

        $log_dir = ORPHANIX_Paths::ensure_subdir( 'logs' );

        // Only log if directory exists and is writable
        if ( ! $filesystem->is_dir( $log_dir ) || ! $filesystem->is_writable( $log_dir ) ) {
            // Directory creation/write failed, try WordPress error log
            if ( defined('WP_DEBUG_LOG') && WP_DEBUG_LOG && class_exists( 'ORPHANIX_Logger' ) ) {
                ORPHANIX_Logger::log( 'Audit Log Fallback', $log_entry );
            }
            return;
        }

        $log_file = $log_dir . 'audit-' . gmdate( 'Y-m-d' ) . '.log';
        $log_message = wp_json_encode( $log_entry ) . "\n";

        // Write to file
        $filesystem->put_contents( $log_file, $log_message, FS_CHMOD_FILE );
    }

    /**
     * Write log entry to database
     * 
     * @param array $log_entry Log entry array
     */
    private static function write_to_database( $log_entry ) {
        global $wpdb;
        
        // Only write if audit table exists
        $table = $wpdb->prefix . 'orphanix_audit_log';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) ) ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->insert( $table, [
                'timestamp' => $log_entry['timestamp'],
                'action' => $log_entry['action'],
                'user_id' => $log_entry['user_id'],
                'user_login' => $log_entry['user_login'],
                'user_ip' => $log_entry['user_ip'],
                'severity' => $log_entry['severity'],
                'data' => wp_json_encode( $log_entry['data'] ),
            ]);
        }
    }

    /**
     * Get client IP address
     * 
     * @return string Client IP
     */
    private static function get_client_ip() {
        if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) {
            $ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_CLIENT_IP'] ) );
        } elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
            $forwarded = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
            $parts = array_map( 'trim', explode( ',', $forwarded ) );
            $ip = $parts[0] ?? '';
        } else {
            $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : 'unknown';
        }

        // Sanitize IP
        return filter_var( $ip, FILTER_VALIDATE_IP ) ? $ip : 'invalid';
    }

    /**
     * Get recent audit logs
     * 
     * @param int $limit Number of logs to retrieve
     * @return array Recent logs
     */
    public static function get_recent_logs( $limit = 50 ) {
        $log_dir = ORPHANIX_Paths::ensure_subdir( 'logs' );
        if ( ! is_dir( $log_dir ) ) {
            return [];
        }

        $logs = [];
        $files = scandir( $log_dir, SCANDIR_SORT_DESCENDING );

        foreach ( $files as $file ) {
            if ( ! preg_match( '/^audit-.*\.log$/', $file ) ) {
                continue;
            }

            $log_file = $log_dir . $file;
            $lines = file( $log_file );

            foreach ( array_reverse( $lines ) as $line ) {
                $logs[] = json_decode( trim( $line ), true );
                if ( count( $logs ) >= $limit ) {
                    break 2;
                }
            }
        }

        return $logs;
    }

    /**
     * Clear old audit logs
     * 
     * Keeps logs for 30 days by default
     * 
     * @param int $days Days to retain
     */
    public static function cleanup_old_logs( $days = 30 ) {
        $log_dir = ORPHANIX_Paths::ensure_subdir( 'logs' );
        if ( ! is_dir( $log_dir ) ) {
            return 0;
        }

        $deleted = 0;
        $cutoff_time = time() - ( $days * DAY_IN_SECONDS );
        $files = scandir( $log_dir );

        foreach ( $files as $file ) {
            if ( ! preg_match( '/^audit-.*\.log$/', $file ) ) {
                continue;
            }

            $log_file = $log_dir . $file;
            if ( filemtime( $log_file ) < $cutoff_time ) {
                wp_delete_file( $log_file );
                $deleted++;
            }
        }

        return $deleted;
    }

    /**
     * Initialize WP_Filesystem and return it.
     */
    private static function get_filesystem() {
        global $wp_filesystem;
        if ( empty( $wp_filesystem ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }
        return $wp_filesystem;
    }
}

