<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( defined('WP_CLI') && WP_CLI ) {
    class ORPHANIX_WP_CLI {
        public function __construct() {
            WP_CLI::add_command('media-cleaner', [$this,'handle_command']);
        }

        public function handle_command($args, $assoc_args) {
            $subcommand = $args[0] ?? null;

            switch ($subcommand) {
                case 'scan':
                    $this->scan($assoc_args);
                    break;
                case 'broken-scan':
                    $this->broken_scan($assoc_args);
                    break;
                case 'results':
                    $this->results($assoc_args);
                    break;
                case 'delete':
                    $this->delete($assoc_args);
                    break;
                case 'restore':
                    $this->restore($assoc_args);
                    break;
                case 'analytics':
                    $this->analytics();
                    break;
                case 'status':
                    $this->status($assoc_args);
                    break;
                case 'help':
                default:
                    $this->help();
                    break;
            }
        }

        private function scan($assoc_args) {
            $type = $assoc_args['type'] ?? 'regular';
            $settings = $assoc_args;

            if ($type === 'regular') {
                $scan = new ORPHANIX_Regular_Scan();
            } else {
                $scan = new ORPHANIX_Deep_Scan();
            }

            $scan_id = $scan->run($settings);
            WP_CLI::success("Scan started. ID: $scan_id");
        }

        private function broken_scan($assoc_args) {
            $type = $assoc_args['type'] ?? 'regular';
            $scan = new ORPHANIX_Broken_Scan();
            $scan_id = $scan->run(['type'=>$type]);
            WP_CLI::success("Broken scan started. ID: $scan_id");
        }

        private function results($assoc_args) {
            $scan_id = $assoc_args['scan_id'] ?? null;
            if (!$scan_id) {
                WP_CLI::error("Please provide --scan_id=<id>");
            }
            $repo = new ORPHANIX_Scan_Items_Repository();
            $items = $repo->all_items($scan_id);
            foreach ($items as $item) {
                WP_CLI::line("{$item['id']} | {$item['file_path']} | {$item['status']}");
            }
        }

        private function delete($assoc_args) {
            $status = $assoc_args['status'] ?? 'orphan';
            $mode   = $assoc_args['mode'] ?? 'trash';

            $repo = new ORPHANIX_Scan_Items_Repository();
            $items = $repo->filter_items($assoc_args['scan_id'],$status);

            foreach ($items as $item) {
                if ($mode === 'trash') {
                    // Move to trash
                    $trash_repo = new ORPHANIX_Trash_Repository();
                    $trash_repo->add_to_trash(
                        $item['id'],
                        $item['file_path'],
                        '',
                        gmdate( 'Y-m-d H:i:s', strtotime( '+30 days' ) )
                    );
                } elseif (!empty($assoc_args['force'])) {
                    wp_delete_file( $item['file_path'] );
                    $repo->delete_item($item['id']);
                }
            }
            WP_CLI::success("Deleted items with status=$status mode=$mode");
        }

        private function restore($assoc_args) {
            $repo = new ORPHANIX_Trash_Repository();
            $items = $repo->expired_items(); // Actually fetch all trash items
            foreach ($items as $item) {
                $repo->restore($item['id']);
            }
            WP_CLI::success("Restored all items from trash.");
        }

        private function analytics() {
            $repo = new ORPHANIX_Scan_Repository();
            $scans = $repo->all_scans();
            foreach ($scans as $scan) {
                WP_CLI::line("Scan {$scan['id']} | Mode: {$scan['scan_mode']} | Used: {$scan['used_files']} | Orphan: {$scan['orphan_files']}");
            }
        }

        private function status($assoc_args) {
            $scan_id = $assoc_args['scan_id'] ?? null;
            if (!$scan_id) {
                WP_CLI::error("Please provide --scan_id=<id>");
            }
            $repo = new ORPHANIX_Scan_Repository();
            $scan = $repo->find($scan_id);
            WP_CLI::line("Scan {$scan['id']} | Status: {$scan['status']} | Processed: {$scan['processed_files']}");
        }

        private function help() {
            WP_CLI::line("Orphanix Media Cleanup CLI Commands:");
            WP_CLI::line("  wp media-cleaner scan --type=regular|deep");
            WP_CLI::line("  wp media-cleaner broken-scan --type=regular|deep");
            WP_CLI::line("  wp media-cleaner results list --scan_id=<id>");
            WP_CLI::line("  wp media-cleaner delete --scan_id=<id> --status=orphan --mode=trash|delete [--force]");
            WP_CLI::line("  wp media-cleaner restore");
            WP_CLI::line("  wp media-cleaner analytics");
            WP_CLI::line("  wp media-cleaner status --scan_id=<id>");
        }
    }

    new ORPHANIX_WP_CLI();
}
