<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class ORPHANIX_Wizard_Ajax {
    public function __construct() {
        add_action('wp_ajax_orphanix_initiate_scan', [$this, 'initiate_scan']);
        add_action('wp_ajax_orphanix_get_scan_progress', [$this, 'get_scan_progress']);
        add_action('wp_ajax_orphanix_cancel_scan', [$this, 'cancel_scan']);
        add_action('wp_ajax_orphanix_delete_scan', [$this, 'delete_scan']);
        add_action('wp_ajax_orphanix_delete_item', [$this, 'delete_item']);
        add_action('wp_ajax_orphanix_process_scan', [$this, 'process_scan']);
        add_action('orphanix_process_scan_event', [$this, 'process_scan_event'], 10, 1);
    }

    public function initiate_scan() {
        check_ajax_referer('orphanix_scan_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Unauthorized', 'orphanix-media-cleanup'));
        }

        $scan_type = isset( $_POST['scan_type'] ) ? sanitize_text_field( wp_unslash( $_POST['scan_type'] ) ) : 'regular';
        $mode = isset( $_POST['mode'] ) ? sanitize_text_field( wp_unslash( $_POST['mode'] ) ) : 'media';
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $raw_settings = isset( $_POST['settings'] ) ? wp_unslash( $_POST['settings'] ) : '';
        $settings = $raw_settings ? json_decode( $raw_settings, true ) : [];

        // Validate scan_type is either 'regular' or 'deep'
        if (!in_array($scan_type, ['regular', 'deep'])) {
            $scan_type = 'regular';
        }

        // Validate mode is either 'media' or 'broken'
        if (!in_array($mode, ['media', 'broken'])) {
            $mode = 'media';
        }
        
        // Ensure settings is an array and sanitize
        if ( ! is_array( $settings ) ) {
            $settings = [];
        }
        $settings = array_map( function( $value ) {
            if ( is_array( $value ) ) {
                return array_map( 'sanitize_text_field', $value );
            }
            return sanitize_text_field( $value );
        }, $settings );

        $manager = new ORPHANIX_Scan_Manager();
        $scan_id = $manager->start_scan($scan_type, $mode, $settings);

        // Start the scan in background
        if ($scan_type === 'regular') {
            $scan = new ORPHANIX_Regular_Scan();
        } else {
            $scan = new ORPHANIX_Deep_Scan();
        }


        // Store scan progress placeholder
        set_transient('orphanix_scan_' . $scan_id . '_progress', [
            'scan_id' => $scan_id,
            'status' => 'running',
            'total' => 0,
            'processed' => 0,
            'percentage' => 0,
            'current_file' => '',
            'started_at' => current_time('timestamp'),
        ], 3600);

        // Persist scan settings and type so the background processor can pick them up
        set_transient('orphanix_scan_' . $scan_id . '_settings', [
            'settings' => $settings,
            'scan_type' => $scan_type,
            'mode'     => $mode,
        ], 3600);

        // Trigger a non-blocking background request to process the scan so the AJAX that
        // initiated the scan can return immediately and the UI can start polling progress.
        $admin_ajax = admin_url('admin-ajax.php');
        $body = [
            'action' => 'orphanix_process_scan',
            'nonce'  => wp_create_nonce('orphanix_scan_nonce'),
            'scan_id'=> $scan_id,
        ];

        // Non-blocking internal request; do not forward raw superglobals.
        $args = [
            'timeout'  => 0.01,
            'blocking' => false,
            'body'     => $body,
        ];

        $loopback_response = wp_remote_post( $admin_ajax, $args );
        if ( is_wp_error( $loopback_response ) ) {
            ORPHANIX_Logger::log(
                'Loopback request failed',
                [
                    'scan_id' => $scan_id,
                    'error'   => $loopback_response->get_error_message(),
                ]
            );
            wp_send_json_error( __( 'Unable to start scan background request.', 'orphanix-media-cleanup' ) );
        }

        // Fallback: schedule a cron event in case loopback requests are blocked.
        if ( ! wp_next_scheduled('orphanix_process_scan_event', [$scan_id]) ) {
            wp_schedule_single_event(time() + 5, 'orphanix_process_scan_event', [$scan_id]);
        }

        wp_send_json_success(['scan_id' => $scan_id]);
    }

    public function process_scan() {
        check_ajax_referer('orphanix_scan_nonce', 'nonce');

        if (! current_user_can('manage_options') ) {
            wp_send_json_error(__('Unauthorized', 'orphanix-media-cleanup'));
        }

        $scan_id = isset( $_POST['scan_id'] ) ? absint( wp_unslash( $_POST['scan_id'] ) ) : 0;
        if (! $scan_id) {
            wp_send_json_error(__('Invalid scan ID', 'orphanix-media-cleanup'));
        }

        if ( ! $this->acquire_scan_lock($scan_id) ) {
            wp_send_json_success(['processed' => false, 'locked' => true]);
        }

        $data = get_transient('orphanix_scan_' . $scan_id . '_settings');
        if ( ! $data || ! is_array($data) ) {
            $this->release_scan_lock($scan_id);
            wp_send_json_error(__('Scan data not found', 'orphanix-media-cleanup'));
        }

        $settings = $data['settings'] ?? [];
        $scan_type = $data['scan_type'] ?? 'regular';

        if ($scan_type === 'regular') {
            $scan = new ORPHANIX_Regular_Scan();
        } else {
            $scan = new ORPHANIX_Deep_Scan();
        }

        // Run the async processor (this runs inside the background request)
        $scan->run_async($scan_id, $settings);

        // cleanup transient settings (progress transient kept for results)
        delete_transient('orphanix_scan_' . $scan_id . '_settings');

        $this->release_scan_lock($scan_id);

        wp_send_json_success(['processed' => true]);
    }

    public function process_scan_event($scan_id) {
        if ( class_exists( 'ORPHANIX_Logger' ) ) {
            ORPHANIX_Logger::log( 'Cron Event Triggered', [ 'scan_id' => $scan_id ] );
        }
        
        $scan_id = absint($scan_id);
        if (! $scan_id) {
            if ( class_exists( 'ORPHANIX_Logger' ) ) {
                ORPHANIX_Logger::log( 'Cron Event - Invalid scan_id' );
            }
            return;
        }

        if ( ! $this->acquire_scan_lock($scan_id) ) {
            if ( class_exists( 'ORPHANIX_Logger' ) ) {
                ORPHANIX_Logger::log( 'Cron Event - Could not acquire lock (already processing)', [ 'scan_id' => $scan_id ] );
            }
            return;
        }

        $data = get_transient('orphanix_scan_' . $scan_id . '_settings');
        if ( ! $data || ! is_array($data) ) {
            if ( class_exists( 'ORPHANIX_Logger' ) ) {
                ORPHANIX_Logger::log( 'Cron Event - No settings transient found', [ 'scan_id' => $scan_id ] );
            }
            $this->release_scan_lock($scan_id);
            return;
        }

        if ( class_exists( 'ORPHANIX_Logger' ) ) {
            ORPHANIX_Logger::log( 'Cron Event - Starting scan', [ 'scan_type' => $data['scan_type'] ?? 'regular' ] );
        }

        $settings = $data['settings'] ?? [];
        $scan_type = $data['scan_type'] ?? 'regular';

        if ($scan_type === 'regular') {
            $scan = new ORPHANIX_Regular_Scan();
        } else {
            $scan = new ORPHANIX_Deep_Scan();
        }

        $scan->run_async($scan_id, $settings);
        delete_transient('orphanix_scan_' . $scan_id . '_settings');

        $this->release_scan_lock($scan_id);
    }

    private function acquire_scan_lock($scan_id) {
        $lock_key = 'orphanix_scan_' . $scan_id . '_processing';
        if (get_transient($lock_key)) {
            return false;
        }
        set_transient($lock_key, 1, 10 * MINUTE_IN_SECONDS);
        return true;
    }

    private function release_scan_lock($scan_id) {
        delete_transient('orphanix_scan_' . $scan_id . '_processing');
    }

    public function get_scan_progress() {
        check_ajax_referer('orphanix_scan_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Unauthorized', 'orphanix-media-cleanup'));
        }

        $scan_id = isset( $_POST['scan_id'] ) ? absint( wp_unslash( $_POST['scan_id'] ) ) : 0;
        if ( $scan_id <= 0 ) {
            wp_send_json_error( __( 'Invalid scan ID', 'orphanix-media-cleanup' ) );
        }
        
        global $wpdb;
        $scan_cache_key = 'orphanix_scan_' . $scan_id;
        $scan = wp_cache_get( $scan_cache_key, 'orphanix' );
        if ( false === $scan ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $scan = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}orphanix_scans WHERE id = %d", $scan_id));
            wp_cache_set( $scan_cache_key, $scan, 'orphanix', 60 );
        }

        if (!$scan) {
            wp_send_json_error(__('Scan not found', 'orphanix-media-cleanup'));
        }

        $progress = get_transient('orphanix_scan_' . $scan_id . '_progress');
        
        if (!$progress) {
            // Recalculate from database
            $items_cache_key = 'orphanix_scan_items_count_' . $scan_id;
            $items = wp_cache_get( $items_cache_key, 'orphanix' );
            if ( false === $items ) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $items = $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d",
                    $scan_id
                ));
                wp_cache_set( $items_cache_key, $items, 'orphanix', 60 );
            }

            $percentage = $scan->total_files > 0 ? round(($scan->processed_files / $scan->total_files) * 100) : 0;

            $progress = [
                'scan_id' => $scan_id,
                'status' => $scan->status,
                'total' => intval($scan->total_files),
                'processed' => intval($scan->processed_files),
                'percentage' => $percentage,
                'current_file' => '',
                'started_at' => strtotime($scan->started_at),
                'used' => intval($scan->used_files),
                'orphan' => intval($scan->orphan_files),
                'in_library' => intval($scan->total_files) - intval($scan->orphan_files),
            ];
        }

        // Force-start stalled scans if progress never advanced.
        if (
            isset($progress['status'], $progress['total'], $progress['processed']) &&
            'running' === $progress['status'] &&
            intval($progress['total']) === 0 &&
            intval($progress['processed']) === 0
        ) {
            $data = get_transient('orphanix_scan_' . $scan_id . '_settings');
            if ($data && is_array($data)) {
                if ($this->acquire_scan_lock($scan_id)) {
                    $settings = $data['settings'] ?? [];
                    $scan_type = $data['scan_type'] ?? 'regular';
                    if ($scan_type === 'regular') {
                        $scan_runner = new ORPHANIX_Regular_Scan();
                    } else {
                        $scan_runner = new ORPHANIX_Deep_Scan();
                    }
                    if ( class_exists( 'ORPHANIX_Logger' ) ) {
                        ORPHANIX_Logger::log( 'Forcing scan start from progress poll', [
                            'scan_id' => $scan_id,
                            'scan_type' => $scan_type,
                            'mode' => $data['mode'] ?? 'media',
                        ] );
                    }
                    $scan_runner->run_async($scan_id, $settings);
                    delete_transient('orphanix_scan_' . $scan_id . '_settings');
                    $this->release_scan_lock($scan_id);

                    $progress = get_transient('orphanix_scan_' . $scan_id . '_progress') ?: $progress;
                }
            }
        }

        // Ensure progress always contains a started_at timestamp
        if (empty($progress['started_at'])) {
            $progress['started_at'] = $scan->started_at ? strtotime($scan->started_at) : current_time('timestamp');
        }

        // Calculate ETA
        $eta = '--';
        if (isset($progress['percentage']) && $progress['percentage'] > 0 && $progress['percentage'] < 100) {
            $elapsed = current_time('timestamp') - $progress['started_at'];
            if ($elapsed > 0) {
                $rate = $progress['percentage'] / $elapsed; // percentage per second
                $remaining_percentage = 100 - $progress['percentage'];
                $remaining_seconds = $remaining_percentage / $rate;
                $eta = gmdate('H:i:s', (int)$remaining_seconds);
            }
        }
        $progress['eta'] = $eta;

        wp_send_json_success($progress);
    }

    public function cancel_scan() {
        check_ajax_referer('orphanix_scan_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Unauthorized', 'orphanix-media-cleanup'));
        }

        $scan_id = isset( $_POST['scan_id'] ) ? absint( wp_unslash( $_POST['scan_id'] ) ) : 0;
        if ( $scan_id <= 0 ) {
            wp_send_json_error( __( 'Invalid scan ID', 'orphanix-media-cleanup' ) );
        }
        
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->update("{$wpdb->prefix}orphanix_scans", [
            'status' => 'cancelled',
        ], ['id' => $scan_id]);

        delete_transient('orphanix_scan_' . $scan_id . '_progress');
        
        wp_send_json_success(__('Scan cancelled', 'orphanix-media-cleanup'));
    }

    public function delete_scan() {
        check_ajax_referer('orphanix_scan_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Unauthorized', 'orphanix-media-cleanup'));
        }

        $scan_id = isset( $_POST['scan_id'] ) ? absint( wp_unslash( $_POST['scan_id'] ) ) : 0;
        if ( $scan_id <= 0 ) {
            wp_send_json_error( __( 'Invalid scan ID', 'orphanix-media-cleanup' ) );
        }
        
        global $wpdb;
        
        // Delete scan items.
        $wpdb->delete("{$wpdb->prefix}orphanix_scan_items", ['scan_id' => $scan_id]); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        
        // Delete scan.
        $wpdb->delete("{$wpdb->prefix}orphanix_scans", ['id' => $scan_id]); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

        delete_transient('orphanix_scan_' . $scan_id . '_progress');
        
        wp_send_json_success(__('Scan deleted', 'orphanix-media-cleanup'));
    }

    public function delete_item() {
        check_ajax_referer('orphanix_scan_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Unauthorized', 'orphanix-media-cleanup'));
        }

        $item_id = isset( $_POST['id'] ) ? absint( wp_unslash( $_POST['id'] ) ) : 0;
        if ( $item_id <= 0 ) {
            wp_send_json_error( __( 'Invalid item ID', 'orphanix-media-cleanup' ) );
        }
        
        global $wpdb;
        $wpdb->delete("{$wpdb->prefix}orphanix_scan_items", ['id' => $item_id]); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

        wp_send_json_success(__('Item deleted', 'orphanix-media-cleanup'));
    }
}
