<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Delete AJAX Handler
 * 
 * Handles deletion of media files and attachments
 * COMPLIANCE: Operational Rules 9.1 (Delete from scan AND Media Library)
 */
class ORPHANIX_Delete_Ajax {
    public function __construct() {
        add_action('wp_ajax_orphanix_delete_item', [$this,'delete_item']);
        add_action('wp_ajax_orphanix_bulk_delete', [$this,'bulk_delete']);
    }

    /**
     * Delete single scan item and its attachment
     */
    public function delete_item() {
        check_ajax_referer('orphanix_delete_item','nonce');
        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error(__('Unauthorized','orphanix-media-cleanup'));
        }

        $id = isset( $_POST['id'] ) ? absint( wp_unslash( $_POST['id'] ) ) : 0;
        if ( $id <= 0 ) {
            wp_send_json_error(__('Invalid item ID','orphanix-media-cleanup'));
        }

        $repo = new ORPHANIX_Scan_Items_Repository();
        $item = $repo->get_item($id);
        
        if ( ! $item ) {
            wp_send_json_error(__('Item not found','orphanix-media-cleanup'));
        }

        $result = $repo->delete_item_with_attachment($id);

        if ( is_wp_error($result) ) {
            wp_send_json_error($result->get_error_message());
            return;
        }

        // Log the deletion
        if ( class_exists('ORPHANIX_Audit_Logger') ) {
            ORPHANIX_Audit_Logger::log_delete(
                intval($item['attachment_id'] ?? 0),
                $item['file_path'] ?? ''
            );
        }

        wp_send_json_success([
            'deleted' => $id,
            'message' => __('File deleted successfully', 'orphanix-media-cleanup')
        ]);
    }

    /**
     * Bulk delete scan items and their attachments
     */
    public function bulk_delete() {
        check_ajax_referer('orphanix_bulk_action','nonce');
        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error(__('Unauthorized','orphanix-media-cleanup'));
        }

        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $raw_ids = isset( $_POST['ids'] ) ? wp_unslash( $_POST['ids'] ) : [];
        $ids = is_array( $raw_ids ) ? array_map( 'absint', $raw_ids ) : [];
        if ( empty($ids) ) {
            wp_send_json_error(__('No items selected','orphanix-media-cleanup'));
        }

        $repo = new ORPHANIX_Scan_Items_Repository();
        $deleted = [];
        $failed = [];

        foreach ( $ids as $id ) {
            if ( $id <= 0 ) continue;
            
            $item = $repo->get_item($id);
            if ( ! $item ) continue;

            $result = $repo->delete_item_with_attachment($id);
            
            if ( is_wp_error($result) ) {
                $failed[$id] = $result->get_error_message();
            } else {
                $deleted[] = $id;
                
                // Log the deletion
                if ( class_exists('ORPHANIX_Audit_Logger') ) {
                    ORPHANIX_Audit_Logger::log_delete(
                        intval($item['attachment_id'] ?? 0),
                        $item['file_path'] ?? ''
                    );
                }
            }
        }

        wp_send_json_success([
            'deleted' => $deleted,
            'failed' => $failed,
            'message' => sprintf(
                /* translators: %d: number of files deleted. */
                __('%d files deleted', 'orphanix-media-cleanup'),
                count($deleted)
            )
        ]);
    }
}
