<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class ORPHANIX_Settings {
	public function __construct() {
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'maybe_override_delete_popup' ) );
	}

	public function register_settings() {
		// Register settings with sanitization.
		register_setting( 'orphanix_settings_group', 'orphanix_scan_compat', array(
			'type'              => 'array',
			'sanitize_callback' => array( $this, 'sanitize_compat_settings' ),
			'default'           => $this->get_default_options(),
		) );

		register_setting( 'orphanix_settings_group', 'orphanix_delete_on_uninstall', array(
			'type'              => 'boolean',
			'sanitize_callback' => function ( $input ) {
				return $input ? 1 : 0;
			},
			'default'           => false,
		) );

        // General Settings Section
        add_settings_section(
            'orphanix_general_section',
            __('General Settings', 'orphanix-media-cleanup'),
            [$this, 'general_section_callback'],
            'orphanix-settings'
        );

        // General Settings Field
        add_settings_field(
            'orphanix_delete_on_uninstall',
            __('Delete Plugin Data', 'orphanix-media-cleanup'),
            [$this, 'delete_on_uninstall_callback'],
            'orphanix-settings',
            'orphanix_general_section'
        );

        // Page Builder & Plugin Compatibility Section
        add_settings_section(
            'orphanix_compat_section',
            __('Page Builder & Plugin Compatibility', 'orphanix-media-cleanup'),
            [$this, 'compatibility_section_callback'],
            'orphanix-settings'
        );

        // Page Builders
        add_settings_field(
            'orphanix_scan_gutenberg',
            __('Gutenberg Blocks', 'orphanix-media-cleanup'),
            [$this, 'checkbox_field_callback'],
            'orphanix-settings',
            'orphanix_compat_section',
            ['name' => 'orphanix_scan_compat[scan_gutenberg]', 'label' => __('Scan Gutenberg blocks for media', 'orphanix-media-cleanup'), 'icon' => '']
        );

        add_settings_field(
            'orphanix_scan_elementor',
            __('Elementor', 'orphanix-media-cleanup'),
            [$this, 'checkbox_field_callback'],
            'orphanix-settings',
            'orphanix_compat_section',
            ['name' => 'orphanix_scan_compat[scan_elementor]', 'label' => __('Scan Elementor pages for media', 'orphanix-media-cleanup'), 'icon' => '']
        );


        // Custom Fields Plugins
        add_settings_field(
            'orphanix_scan_acf',
            __('Advanced Custom Fields (ACF)', 'orphanix-media-cleanup'),
            [$this, 'checkbox_field_callback'],
            'orphanix-settings',
            'orphanix_compat_section',
            ['name' => 'orphanix_scan_compat[scan_acf]', 'label' => __('Scan ACF fields for media', 'orphanix-media-cleanup'), 'icon' => '']
        );


        // eCommerce & Media
        add_settings_field(
            'orphanix_scan_woocommerce',
            __('WooCommerce Product Media', 'orphanix-media-cleanup'),
            [$this, 'checkbox_field_callback'],
            'orphanix-settings',
            'orphanix_compat_section',
            ['name' => 'orphanix_scan_compat[scan_woocommerce]', 'label' => __('Scan WooCommerce product galleries', 'orphanix-media-cleanup'), 'icon' => '']
        );


        // Theme & Core
        add_settings_field(
            'orphanix_scan_widgets',
            __('Widgets', 'orphanix-media-cleanup'),
            [$this, 'checkbox_field_callback'],
            'orphanix-settings',
            'orphanix_compat_section',
            ['name' => 'orphanix_scan_compat[scan_widgets]', 'label' => __('Scan WordPress widgets', 'orphanix-media-cleanup'), 'icon' => '']
        );

        add_settings_field(
            'orphanix_scan_menus',
            __('Navigation Menus', 'orphanix-media-cleanup'),
            [$this, 'checkbox_field_callback'],
            'orphanix-settings',
            'orphanix_compat_section',
            ['name' => 'orphanix_scan_compat[scan_menus]', 'label' => __('Scan menu items', 'orphanix-media-cleanup'), 'icon' => '']
        );

        add_settings_field(
            'orphanix_scan_theme_files',
            __('Theme Files (Deep Scan Only)', 'orphanix-media-cleanup'),
            [$this, 'checkbox_field_callback'],
            'orphanix-settings',
            'orphanix_compat_section',
            ['name' => 'orphanix_scan_compat[scan_theme_files]', 'label' => __('Scan active theme files for media references (slower)', 'orphanix-media-cleanup'), 'icon' => '']
        );
    }

    public function general_section_callback() {
        echo wp_kses_post(
            '<p data-orphanix-style="orphanix-s-0136">' . __('Configure general plugin behavior and preferences.', 'orphanix-media-cleanup') . '</p>'
        );
    }

    public function delete_on_uninstall_callback() {
        $value = get_option('orphanix_delete_on_uninstall', false);
        $is_checked = $value ? true : false;
        ?>
        <div data-orphanix-style="orphanix-s-0054">
            <input type="hidden" name="orphanix_delete_on_uninstall" value="0">
            <input type="checkbox" 
                   name="orphanix_delete_on_uninstall" 
                   id="orphanix_delete_on_uninstall" 
                   value="1" 
                   data-orphanix-style="orphanix-s-0120"
                   <?php checked($is_checked, true); ?>>
            <div data-orphanix-style="orphanix-s-0056">
                <label for="orphanix_delete_on_uninstall" data-orphanix-style="orphanix-s-0057">
                    <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Delete Plugin Data on Uninstall', 'orphanix-media-cleanup'); ?></strong>
                    <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('The database and all the options of the plugin will be removed on uninstall. This also includes the information about the plugin\'s trash.', 'orphanix-media-cleanup'); ?></small>
                </label>
            </div>
        </div>
        <?php
    }

    public function maybe_override_delete_popup( $hook_suffix ) {
        if ( 'plugins.php' !== $hook_suffix ) {
            return;
        }

        $delete_on_uninstall = get_option('orphanix_delete_on_uninstall', false);
        if ( $delete_on_uninstall ) {
            return;
        }

        $plugin = plugin_basename( ORPHANIX_PATH . 'orphanix-media-cleanup.php' );
        $script = "
            jQuery(function($) {
                var pluginRow = 'tr[data-plugin=\"" . esc_js( $plugin ) . "\"]';
                $(document).on('click', pluginRow + ' .delete a', function(e) {
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    window.location.href = this.href;
                });
            });
        ";
        wp_add_inline_script( 'orphanix-admin', $script );
    }

    public function compatibility_section_callback() {
        echo wp_kses_post(
            '<p data-orphanix-style="orphanix-s-0136">' . __('Enable or disable compatibility scans for popular page builders and plugins. Disabling unused integrations can improve scan performance.', 'orphanix-media-cleanup') . '</p>'
        );
        ?>
        <div data-orphanix-style="orphanix-s-0137">
            <div data-orphanix-style="orphanix-s-0138">
                <div>
                    <strong data-orphanix-style="orphanix-s-0139">* <?php esc_html_e('Fast Performance','orphanix-media-cleanup'); ?></strong>
                    <small data-orphanix-style="orphanix-s-0140"><?php esc_html_e('Minimal database queries required','orphanix-media-cleanup'); ?></small>
                </div>
                <div>
                    <strong data-orphanix-style="orphanix-s-0139">* <?php esc_html_e('Medium Performance','orphanix-media-cleanup'); ?></strong>
                    <small data-orphanix-style="orphanix-s-0141"><?php esc_html_e('Serialized data scanning','orphanix-media-cleanup'); ?></small>
                </div>
                <div>
                    <strong data-orphanix-style="orphanix-s-0139"> <?php esc_html_e('Comprehensive','orphanix-media-cleanup'); ?></strong>
                    <small data-orphanix-style="orphanix-s-0142"><?php esc_html_e('Filesystem deep scanning','orphanix-media-cleanup'); ?></small>
                </div>
            </div>
        </div>
        <?php
    }

    public function checkbox_field_callback($args) {
        $options = get_option('orphanix_scan_compat', $this->get_default_options());
        $name = $args['name'];
        $field_key = str_replace('orphanix_scan_compat[', '', str_replace(']', '', $name));
        $value = isset($options[$field_key]) ? intval($options[$field_key]) : 0;
        $is_checked = (int) $value === 1;
        $icon = isset($args['icon']) ? $args['icon'] : '&#10003;';
        
        // Determine performance level
        $perf_map = [
            'scan_gutenberg' => 'fast',
            'scan_elementor' => 'medium',
            'scan_acf' => 'medium',
            'scan_woocommerce' => 'fast',
            'scan_widgets' => 'fast',
            'scan_menus' => 'fast',
            'scan_theme_files' => 'slow',
        ];
        
        $perf = $perf_map[$field_key] ?? 'medium';
        $perf_colors = [
            'fast' => ['bg' => '#f0fdf4', 'text' => '#15803d', 'icon' => '*', 'label' => __( 'Fast', 'orphanix-media-cleanup' )],
            'medium' => ['bg' => '#fffbf0', 'text' => '#c17e17', 'icon' => '*', 'label' => __( 'Medium', 'orphanix-media-cleanup' )],
            'slow' => ['bg' => '#fef2f2', 'text' => '#b91c1c', 'icon' => '', 'label' => __( 'Slow', 'orphanix-media-cleanup' )],
        ];
        
        $colors = $perf_colors[$perf];
        $perf_badge_text = trim( $colors['icon'] . ' ' . $colors['label'] . ' ' . __( 'Performance', 'orphanix-media-cleanup' ) );

        ?>
        <div data-orphanix-style="orphanix-s-0143" class="orphanix-hover-card">
            <input type="hidden" name="<?php echo esc_attr($name); ?>" value="0">
            <input type="checkbox" 
                   name="<?php echo esc_attr($name); ?>" 
                   id="<?php echo esc_attr($field_key); ?>" 
                   value="1"
                   data-orphanix-style="orphanix-s-0144"
                   <?php checked($is_checked, true); ?>>
            <div data-orphanix-style="orphanix-s-0056">
                <label for="<?php echo esc_attr($field_key); ?>" data-orphanix-style="orphanix-s-0145">
                    <span data-orphanix-style="orphanix-s-0146"><?php echo esc_html($icon); ?></span>
                    <strong data-orphanix-style="orphanix-s-0147"><?php echo esc_html($args['label']); ?></strong>
                </label>
                <div data-orphanix-style="orphanix-s-0148">
                    <div class="orphanix-perf-badge <?php echo esc_attr( 'orphanix-perf-badge--' . sanitize_html_class( $perf ) ); ?>">
                        <?php echo esc_html( $perf_badge_text ); ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    private function get_default_options() {
        return [
            'scan_gutenberg'   => true,
            'scan_elementor'   => true,
            'scan_acf'         => true,
            'scan_woocommerce' => true,
            'scan_widgets'     => true,
            'scan_menus'       => true,
            'scan_theme_files' => false,
        ];
    }

    public function sanitize_compat_settings($input) {
        if (!is_array($input)) {
            $input = [];
        }

        $defaults = $this->get_default_options();
        $sanitized = [];

        foreach ($defaults as $key => $default) {
            // Convert checkbox value (0 or 1 string) to integer
            $sanitized[$key] = isset($input[$key]) && $input[$key] ? 1 : 0;
        }

        return $sanitized;
    }

    public function render() {
        $this->add_settings_styles();
        $this->display_settings_notices();
        ?>
        <div class="wrap" data-orphanix-style="orphanix-s-0149">
            <h1 data-orphanix-style="orphanix-s-0150"><?php esc_html_e('Orphanix Media Cleanup Settings','orphanix-media-cleanup'); ?></h1>
            <p data-orphanix-style="orphanix-s-0151"><?php esc_html_e('Configure plugin behavior, compatibility settings, and preferences.','orphanix-media-cleanup'); ?></p>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('orphanix_settings_group');
                do_settings_sections('orphanix-settings');
                ?>
                <div data-orphanix-style="orphanix-s-0152">
                    <?php submit_button( __( 'Save Settings', 'orphanix-media-cleanup' ), 'primary orphanix-settings-save', 'submit', false ); ?>
                    <a href="<?php echo esc_url( admin_url('admin.php?page=orphanix-dashboard') ); ?>" class="button" data-orphanix-style="orphanix-s-0153"><?php esc_html_e('<- Back to Dashboard', 'orphanix-media-cleanup'); ?></a>
                </div>
            </form>
        </div>
        <?php
    }

    private function display_settings_notices() {
        // Check if settings were saved
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $settings_updated = isset( $_GET['settings-updated'] ) ? sanitize_text_field( wp_unslash( $_GET['settings-updated'] ) ) : '';
        if ( 'true' === $settings_updated ) {
            add_settings_error(
                'orphanix_messages',
                'orphanix_message',
                __('Settings saved successfully!', 'orphanix-media-cleanup'),
                'updated'
            );
        }

        // Display all settings errors/messages
        settings_errors('orphanix_messages');
    }

    private function add_settings_styles() {
        $css = '
            /* Settings Page Wrapper */
            .wrap { margin-top: 20px; }
            /* Notice Styles */
            .notice.notice-updated { border-left-color: #28a745; background-color: #f0fdf4; border: 1px solid #c6f6d5; border-left: 4px solid #28a745; padding: 12px 16px; border-radius: 6px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); }
            .notice.notice-updated strong { color: #15803d; }
            .notice.notice-error { border-left-color: #e74c3c; background-color: #fef2f2; border: 1px solid #fed7d7; border-left: 4px solid #e74c3c; padding: 12px 16px; border-radius: 6px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); }
            .notice.notice-error strong { color: #b91c1c; }
            /* Form Sections */
            .form-table { border: none; background: transparent; margin-top: 24px; border-top: 1px solid #e5e7eb; padding-top: 24px; width: 100%; }
            /* Left Side Titles - Main Settings */
            .form-table th { padding: 16px 20px !important; text-align: left; font-weight: 600 !important; width: auto !important; font-size: 14px !important; color: #1f2937 !important; background: linear-gradient(135deg, #f9fafb 0%, #f3f4f6 100%) !important; border: 1px solid #e5e7eb !important; border-radius: 6px !important; margin-bottom: 0 !important; margin-right: 16px !important; display: table-cell !important; min-width: 160px !important; border-left: 4px solid #0073aa !important; box-shadow: 0 1px 2px rgba(0,0,0,0.04) !important; transition: all 0.2s ease; vertical-align: middle; height: auto; }
            .form-table th:hover { background: linear-gradient(135deg, #f3f4f6 0%, #e5e7eb 100%) !important; box-shadow: 0 2px 4px rgba(0,0,0,0.08) !important; }
            .form-table td { padding: 8px 0 !important; vertical-align: middle; display: table-cell !important; height: auto; }
            .form-table tr { display: table-row !important; margin-bottom: 0 !important; border: none; height: auto; }
            .form-table tr:not(:last-child) { border-bottom: 0; }
            /* Section Headers */
            .form-table h3 { font-size: 18px; font-weight: 700; color: #1f2937; margin: 0 0 12px 0; padding: 0; }
            /* Submit Button */
            .submit button { background: linear-gradient(135deg, #0073aa 0%, #005a87 100%) !important; border: none !important; color: white !important; padding: 10px 24px !important; font-weight: 600 !important; border-radius: 6px !important; cursor: pointer; transition: all 0.2s ease; box-shadow: 0 2px 4px rgba(0,115,170,0.2); }
            .submit button:hover { background: linear-gradient(135deg, #005a87 0%, #003d5c 100%) !important; box-shadow: 0 4px 8px rgba(0,115,170,0.3) !important; }
            /* Remove default WordPress table styling */
            .form-table tr:first-child > th, .form-table tr:first-child > td { border: none; padding: 0; }
            /* Responsive */
            @media (max-width: 900px) { .form-table th { min-width: 140px !important; font-size: 12px !important; padding: 8px 12px !important; } }
        ';
        wp_add_inline_style( 'orphanix-admin', $css );
    }
}
