<?php
// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class ORPHANIX_Media_Scan {
    public function render() {
        $action  = isset( $_GET['action'] ) ? sanitize_text_field( wp_unslash( $_GET['action'] ) ) : '';
        $scan_id = isset( $_GET['scan_id'] ) ? absint( wp_unslash( $_GET['scan_id'] ) ) : 0;
        $nonce   = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';

        if ( 'wizard' === $action ) {
            if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'orphanix_media_wizard' ) ) {
                wp_die( esc_html__( 'Invalid request.', 'orphanix-media-cleanup' ) );
            }
        }

        if ( 'wizard' === $action ) {
            $this->render_wizard();
        } elseif ( 'results' === $action && $scan_id ) {
            $this->render_results( $scan_id );
        } else {
            $this->render_main();
        }
    }

    private function render_main() {
        global $wpdb;
        $status_filter_override = null;

        // Handle bulk actions and status filter submit.
        if ( isset( $_SERVER['REQUEST_METHOD'] ) && 'POST' === $_SERVER['REQUEST_METHOD'] ) {
            $posted_bulk_nonce = isset( $_POST['orphanix_bulk_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['orphanix_bulk_nonce'] ) ) : '';
            $filter_status = isset( $_POST['filter_status'] ) ? sanitize_text_field( wp_unslash( $_POST['filter_status'] ) ) : '';
            $allowed_status_filters = array( 'published', 'running', 'completed', 'failed', 'trashed' );
            if ( ! in_array( $filter_status, $allowed_status_filters, true ) ) {
                $filter_status = '';
            }
            $status_filter_override = $filter_status;

            if ( current_user_can( 'manage_options' ) && ! empty( $posted_bulk_nonce ) && wp_verify_nonce( $posted_bulk_nonce, 'orphanix_bulk_action' ) ) {
                $bulk_action = isset( $_POST['orphanix_bulk_action'] ) ? sanitize_text_field( wp_unslash( $_POST['orphanix_bulk_action'] ) ) : '';
                $scan_ids = isset( $_POST['scan_ids'] ) && is_array( $_POST['scan_ids'] ) ? array_map( 'absint', wp_unslash( $_POST['scan_ids'] ) ) : array();

                if ( 'trash' === $bulk_action && ! empty( $scan_ids ) ) {
                    foreach ( $scan_ids as $sid ) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->update( "{$wpdb->prefix}orphanix_scans", array( 'status' => 'trashed' ), array( 'id' => $sid ) );
                    }
                } elseif ( 'delete' === $bulk_action && ! empty( $scan_ids ) ) {
                    foreach ( $scan_ids as $sid ) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->delete( "{$wpdb->prefix}orphanix_scan_items", array( 'scan_id' => $sid ) );
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->delete( "{$wpdb->prefix}orphanix_scans", array( 'id' => $sid ) );
                        delete_transient( 'orphanix_scan_' . $sid . '_progress' );
                    }
                } elseif ( 'restore' === $bulk_action && ! empty( $scan_ids ) ) {
                    foreach ( $scan_ids as $sid ) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->update( "{$wpdb->prefix}orphanix_scans", array( 'status' => 'completed' ), array( 'id' => $sid ) );
                    }
                }
            }

        }

        // Sorting and pagination
        $allowed_orderby = array( 'scan_mode' => 'scan_mode', 'started_at' => 'started_at', 'status' => 'status', 'total_files' => 'total_files' );
        $orderby_raw = isset( $_GET['orderby'] ) ? sanitize_key( wp_unslash( $_GET['orderby'] ) ) : '';
        $order_raw = isset( $_GET['order'] ) ? strtoupper( sanitize_key( wp_unslash( $_GET['order'] ) ) ) : 'DESC';
        $orderby = isset( $allowed_orderby[ $orderby_raw ] ) ? $allowed_orderby[ $orderby_raw ] : 'started_at';
        $order = ( 'ASC' === $order_raw ) ? 'ASC' : 'DESC';
        $current_orderby = $orderby;
        $current_order = $order;

        $per_page = isset( $_GET['per_page'] ) ? max( 1, absint( wp_unslash( $_GET['per_page'] ) ) ) : 10;
        $paged = isset( $_GET['paged'] ) ? max( 1, absint( wp_unslash( $_GET['paged'] ) ) ) : 1;
        $offset = ( $paged - 1 ) * $per_page;

        // Filter by status
        $status_filter = ( null !== $status_filter_override ) ? $status_filter_override : ( isset( $_GET['filter_status'] ) ? sanitize_text_field( wp_unslash( $_GET['filter_status'] ) ) : '' );
        $allowed_status_filters = [ 'published', 'running', 'completed', 'failed', 'trashed' ];
        if ( ! in_array( $status_filter, $allowed_status_filters, true ) ) {
            $status_filter = '';
        }
        
        // Build query with proper prepared statements
        $order_by_sql = sprintf( '%s %s', $orderby, $order );

        if ( '' === $status_filter || 'published' === $status_filter ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $total_scans = intval( $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status != %s",
                'media',
                'trashed'
            ) ) );
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $scans = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status != %s ORDER BY {$order_by_sql} LIMIT %d OFFSET %d",
                    'media',
                    'trashed',
                    $per_page,
                    $offset
                )
            );
            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        } elseif ( $status_filter ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $total_scans = intval( $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status = %s",
                'media',
                $status_filter
            ) ) );
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $scans = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status = %s ORDER BY {$order_by_sql} LIMIT %d OFFSET %d",
                    'media',
                    $status_filter,
                    $per_page,
                    $offset
                )
            );
            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        }
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $published_count = intval( $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status != %s",
            'media',
            'trashed'
        ) ) );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $trash_count = intval( $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status = %s",
            'media',
            'trashed'
        ) ) );

        $safe_page_arg = isset( $_GET['page'] ) ? sanitize_key( wp_unslash( $_GET['page'] ) ) : 'orphanix-scan';
        if ( 'orphanix-scan' !== $safe_page_arg ) {
            $safe_page_arg = 'orphanix-scan';
        }
        $sort_base_query_args = [
            'page' => 'orphanix-scan',
        ];
        if ( '' !== $status_filter ) {
            $sort_base_query_args['filter_status'] = $status_filter;
        }
        if ( $per_page > 0 ) {
            $sort_base_query_args['per_page'] = $per_page;
        }
        $sort_base_url = add_query_arg( $sort_base_query_args, admin_url( 'admin.php' ) );
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Media Scan','orphanix-media-cleanup'); ?></h1>
            <div data-orphanix-style="orphanix-s-0017">
                <a href="<?php echo esc_url( wp_nonce_url( add_query_arg('action', 'wizard', admin_url('admin.php?page=orphanix-scan')), 'orphanix_media_wizard' ) ); ?>" class="button button-primary button-large">
                    <?php esc_html_e('Start New Media Scan','orphanix-media-cleanup'); ?>
                </a>
            </div>
            <h2><?php esc_html_e('Recent Scans','orphanix-media-cleanup'); ?></h2>
            <ul class="subsubsub">
                <li>
                    <a href="<?php echo esc_url( add_query_arg( array( 'page' => 'orphanix-scan', 'filter_status' => 'published' ), admin_url( 'admin.php' ) ) ); ?>" class="<?php echo ( '' === $status_filter || 'published' === $status_filter ) ? 'current' : ''; ?>">
                        <?php echo esc_html__( 'Publish', 'orphanix-media-cleanup' ); ?> (<?php echo esc_html( $published_count ); ?>)
                    </a> |
                </li>
                <li>
                    <a href="<?php echo esc_url( add_query_arg( array( 'page' => 'orphanix-scan', 'filter_status' => 'trashed' ), admin_url( 'admin.php' ) ) ); ?>" class="<?php echo ( 'trashed' === $status_filter ) ? 'current' : ''; ?>">
                        <?php echo esc_html__( 'Trash', 'orphanix-media-cleanup' ); ?> (<?php echo esc_html( $trash_count ); ?>)
                    </a>
                </li>
            </ul>
            <br class="clear" />

            <form method="post" action="<?php echo esc_url( admin_url( 'admin.php?page=orphanix-scan' ) ); ?>">
                <?php wp_nonce_field('orphanix_bulk_action', 'orphanix_bulk_nonce'); ?>
                <input type="hidden" name="page" value="<?php echo esc_attr( $safe_page_arg ); ?>">
                <div data-orphanix-style="orphanix-s-0018">
                    <select name="orphanix_bulk_action">
                        <option value="">-- <?php esc_html_e('Bulk Actions','orphanix-media-cleanup'); ?> --</option>
                        <?php if ( $status_filter !== 'trashed' ) : ?>
                            <option value="trash"><?php esc_html_e('Move to Trash','orphanix-media-cleanup'); ?></option>
                        <?php else : ?>
                            <option value="restore"><?php esc_html_e('Restore','orphanix-media-cleanup'); ?></option>
                            <option value="delete"><?php esc_html_e('Delete Permanently','orphanix-media-cleanup'); ?></option>
                        <?php endif; ?>
                    </select>
                    <button class="button" type="submit"><?php esc_html_e('Apply','orphanix-media-cleanup'); ?></button>
                    <span data-orphanix-style="orphanix-s-0019">
                        <label><?php esc_html_e('Status:', 'orphanix-media-cleanup'); ?>
                            <select name="filter_status">
                                <option value="">-- <?php esc_html_e('All','orphanix-media-cleanup'); ?> --</option>
                                <option value="published" <?php selected( $status_filter, 'published' ); ?>><?php esc_html_e('Publish','orphanix-media-cleanup'); ?></option>
                                <option value="running" <?php selected( $status_filter, 'running' ); ?>><?php esc_html_e('Running','orphanix-media-cleanup'); ?></option>
                                <option value="completed" <?php selected( $status_filter, 'completed' ); ?>><?php esc_html_e('Completed','orphanix-media-cleanup'); ?></option>
                                <option value="failed" <?php selected( $status_filter, 'failed' ); ?>><?php esc_html_e('Failed','orphanix-media-cleanup'); ?></option>
                                <option value="trashed" <?php selected( $status_filter, 'trashed' ); ?>><?php esc_html_e('Trashed','orphanix-media-cleanup'); ?></option>
                            </select>
                        </label>
                        <button class="button" type="submit" name="filter_action" value="filter"><?php esc_html_e('Filter','orphanix-media-cleanup'); ?></button>
                    </span>
                </div>

            <table class="widefat striped">
                    <thead>
                        <tr>
                            <th data-orphanix-style="orphanix-s-0020"><input type="checkbox" class="orphanix-select-all"></th>
                            <th><a href="<?php echo esc_url( add_query_arg( ['orderby' => 'scan_mode', 'order' => ($current_orderby === 'scan_mode' && 'ASC' === $current_order ? 'DESC' : 'ASC'), 'paged' => 1], $sort_base_url ) ); ?>"><?php esc_html_e('Scan Type','orphanix-media-cleanup'); ?></a></th>
                            <th><a href="<?php echo esc_url( add_query_arg( ['orderby' => 'started_at', 'order' => ($current_orderby === 'started_at' && 'ASC' === $current_order ? 'DESC' : 'ASC'), 'paged' => 1], $sort_base_url ) ); ?>"><?php esc_html_e('Date','orphanix-media-cleanup'); ?></a></th>
                            <th><a href="<?php echo esc_url( add_query_arg( ['orderby' => 'status', 'order' => ($current_orderby === 'status' && 'ASC' === $current_order ? 'DESC' : 'ASC'), 'paged' => 1], $sort_base_url ) ); ?>"><?php esc_html_e('Status','orphanix-media-cleanup'); ?></a></th>
                            <th><a href="<?php echo esc_url( add_query_arg( ['orderby' => 'total_files', 'order' => ($current_orderby === 'total_files' && 'ASC' === $current_order ? 'DESC' : 'ASC'), 'paged' => 1], $sort_base_url ) ); ?>"><?php esc_html_e('Results','orphanix-media-cleanup'); ?></a></th>
                            <th><?php esc_html_e('Actions','orphanix-media-cleanup'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ( $scans ) : foreach ( $scans as $scan ) : ?>
                            <tr>
                                <td><input type="checkbox" name="scan_ids[]" value="<?php echo esc_attr($scan->id); ?>" /></td>
                                <td><?php echo esc_html(ucfirst($scan->scan_mode)); ?></td>
                                <td><?php echo esc_html(date_i18n('Y-m-d H:i', strtotime($scan->started_at))); ?></td>
                                <td><span class="orphanix-status-pill <?php echo esc_attr( 'orphanix-status-pill--' . sanitize_html_class( $scan->status ) ); ?>"><?php echo esc_html(ucfirst($scan->status)); ?></span></td>
                                <td><?php echo esc_html($scan->total_files); ?></td>
                                <td>
                                    <?php if ( 'trashed' !== $scan->status ) : ?>
                                        <a href="<?php echo esc_url( add_query_arg(['action' => 'results', 'scan_id' => $scan->id], admin_url('admin.php?page=orphanix-scan')) ); ?>"><?php esc_html_e('View','orphanix-media-cleanup'); ?></a>
                                    <?php else : ?>
                                        -
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; else : ?>
                            <tr><td colspan="6" data-orphanix-style="orphanix-s-0021"><?php esc_html_e('No scans found.','orphanix-media-cleanup'); ?></td></tr>
                        <?php endif; ?>
                    </tbody>
            </table>

            <!-- Pagination -->
            <?php if ( $total_scans > 0 ) : ?>
                <div class="tablenav-pages" data-orphanix-style="orphanix-s-0022">
                    <span class="displaying-num"><?php echo esc_html( intval( $total_scans ) ); ?> <?php esc_html_e( 'items', 'orphanix-media-cleanup' ); ?></span>
                    <span class="pagination-links" data-orphanix-style="orphanix-s-0023">
                        <?php
                        $total_pages = ceil( $total_scans / $per_page );
                        $base_query_args = [
                            'page' => 'orphanix-scan',
                        ];
                        if ( '' !== $status_filter ) {
                            $base_query_args['filter_status'] = $status_filter;
                        }
                        if ( $per_page > 0 ) {
                            $base_query_args['per_page'] = $per_page;
                        }
                        if ( '' !== $current_orderby ) {
                            $base_query_args['orderby'] = $current_orderby;
                        }
                        if ( '' !== $current_order ) {
                            $base_query_args['order'] = $current_order;
                        }
                        $current_url_base = add_query_arg( $base_query_args, admin_url( 'admin.php' ) );
                        
                        // First page button
                        if ( $paged <= 1 ) {
                            echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&laquo;</span>';
                        } else {
                            echo '<a class="first-page button" href="' . esc_url( add_query_arg( 'paged', 1, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'First page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&laquo;</span></a>';
                        }
                        
                        // Previous page button
                        if ( $paged <= 1 ) {
                            echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&lsaquo;</span>';
                        } else {
                            echo '<a class="prev-page button" href="' . esc_url( add_query_arg( 'paged', $paged - 1, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'Previous page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&lsaquo;</span></a>';
                        }
                        ?>
                        <span class="screen-reader-text"><?php esc_html_e( 'Current Page', 'orphanix-media-cleanup' ); ?></span>
                        <span class="paging-input">
                            <span class="tablenav-paging-text"><?php echo esc_html( intval( $paged ) ); ?> <?php esc_html_e( 'of', 'orphanix-media-cleanup' ); ?> <span class="total-pages"><?php echo esc_html( intval( $total_pages ) ); ?></span></span>
                        </span>
                        <?php
                        // Next page button
                        if ( $paged >= $total_pages ) {
                            echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&rsaquo;</span>';
                        } else {
                            echo '<a class="next-page button" href="' . esc_url( add_query_arg( 'paged', $paged + 1, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'Next page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&rsaquo;</span></a>';
                        }
                        
                        // Last page button
                        if ( $paged >= $total_pages ) {
                            echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&raquo;</span>';
                        } else {
                            echo '<a class="last-page button" href="' . esc_url( add_query_arg( 'paged', $total_pages, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'Last page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&raquo;</span></a>';
                        }
                        ?>
                    </span>
                </div>
            <?php endif; ?>
            </form>
        </div>

        <?php
        ob_start();
        ?>
        (function($){
            $(function(){
                $('.orphanix-select-all').on('change', function(){
                    var checked = $(this).is(':checked');
                    $(this).closest('table').find('tbody input[type=checkbox]').prop('checked', checked);
                });
            });
        })(jQuery);
        <?php
        $inline_script = trim( (string) ob_get_clean() );
        wp_add_inline_script( 'orphanix-admin', $inline_script );
        ?>
        <?php
    }

    private function render_wizard() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Media Scan Wizard','orphanix-media-cleanup'); ?></h1>
            
            <div id="orphanix-wizard" data-orphanix-style="orphanix-s-0024">
                <!-- Step 1: Choose Scan Type -->
                <div class="orphanix-step" data-step="1" data-orphanix-style="orphanix-s-0025">
                    <h2 data-orphanix-style="orphanix-s-0026"><?php esc_html_e('Step 1: Choose Scan Type','orphanix-media-cleanup'); ?></h2>
                    <p data-orphanix-style="orphanix-s-0027"><?php esc_html_e('Select the type of scan that best matches your needs:','orphanix-media-cleanup'); ?></p>
                    
                    <div data-orphanix-style="orphanix-s-0028">
                        <!-- Regular Scan -->
                        <div class="orphanix-scan-option orphanix-scan-regular" onclick="orphanix_select_media_scan_type('regular')" data-orphanix-style="orphanix-s-0109">
                            <div data-orphanix-style="orphanix-s-0030">
                                <span data-orphanix-style="orphanix-s-0031"></span>
                                <h3 data-orphanix-style="orphanix-s-0032"><?php esc_html_e('Regular Scan','orphanix-media-cleanup'); ?></h3>
                            </div>
                            <p data-orphanix-style="orphanix-s-0033"><?php esc_html_e('Scan your media library against posts, pages, and common plugin references to identify unused media files.','orphanix-media-cleanup'); ?></p>
                            <ul data-orphanix-style="orphanix-s-0034">
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0110">&#10003;</span><?php esc_html_e('Analyzes media usage in standard post types','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0110">&#10003;</span><?php esc_html_e('Checks featured images and post content','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0110">&#10003;</span><?php esc_html_e('Detects usage in popular page builders','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0110">&#10003;</span><?php esc_html_e('Scans custom fields and meta data','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0110">&#10003;</span><?php esc_html_e('Provides actionable cleanup suggestions','orphanix-media-cleanup'); ?></li>
                            </ul>
                            <div data-orphanix-style="orphanix-s-0111">
                                <p data-orphanix-style="orphanix-s-0112"><strong><?php esc_html_e('Best for:','orphanix-media-cleanup'); ?></strong><br><?php esc_html_e('Routine cleanup and optimization of your media library.','orphanix-media-cleanup'); ?></p>
                            </div>
                        </div>

                        <!-- Deep Scan -->
                        <div class="orphanix-scan-option orphanix-scan-deep" onclick="orphanix_select_media_scan_type('deep')" data-orphanix-style="orphanix-s-0113">
                            <div data-orphanix-style="orphanix-s-0030">
                                <span data-orphanix-style="orphanix-s-0031"></span>
                                <h3 data-orphanix-style="orphanix-s-0032"><?php esc_html_e('Deep Scan','orphanix-media-cleanup'); ?></h3>
                            </div>
                            <p data-orphanix-style="orphanix-s-0033"><?php esc_html_e('Perform an extensive scan of all files in the uploads directory and across your entire WordPress database.','orphanix-media-cleanup'); ?></p>
                            <ul data-orphanix-style="orphanix-s-0034">
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0114">&#10003;</span><?php esc_html_e('Scans all files in uploads directory recursively','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0114">&#10003;</span><?php esc_html_e('Checks all custom post types and taxonomy attachments','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0114">&#10003;</span><?php esc_html_e('Analyzes theme and plugin references','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0114">&#10003;</span><?php esc_html_e('Reviews serialized and JSON data in database','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0114">&#10003;</span><?php esc_html_e('Generates comprehensive usage reports','orphanix-media-cleanup'); ?></li>
                            </ul>
                            <div data-orphanix-style="orphanix-s-0115">
                                <p data-orphanix-style="orphanix-s-0116"><strong><?php esc_html_e('Best for:','orphanix-media-cleanup'); ?></strong><br><?php esc_html_e('Large sites with complex setups requiring comprehensive analysis.','orphanix-media-cleanup'); ?></p>
                            </div>
                        </div>
                    </div>

                    <input type="hidden" id="orphanix-media-scan-type" value="">
                    
                    <div data-orphanix-style="orphanix-s-0044">
                        <a href="<?php echo esc_url( admin_url('admin.php?page=orphanix-scan') ); ?>" class="button">
                            <?php esc_html_e('Cancel','orphanix-media-cleanup'); ?>
                        </a>
                        <button class="button button-primary" onclick="orphanix_next_media_step(1)" disabled>
                            <?php esc_html_e('Next Step &rarr;','orphanix-media-cleanup'); ?>
                        </button>
                    </div>
                </div>

                <!-- Step 2: Settings -->
                <div class="orphanix-step" data-step="2" data-orphanix-style="orphanix-s-0039">
                    <h2><?php esc_html_e('Step 2: Scan Settings','orphanix-media-cleanup'); ?></h2>
                    <div id="orphanix-media-settings-container"></div>
                    
                    <div data-orphanix-style="orphanix-s-0044">
                        <button class="button" onclick="orphanix_prev_media_step(2)">
                            <?php esc_html_e('&larr; Back','orphanix-media-cleanup'); ?>
                        </button>
                        <button class="button button-primary" onclick="orphanix_next_media_step(2)">
                            <?php esc_html_e('Next Step &rarr;','orphanix-media-cleanup'); ?>
                        </button>
                    </div>
                </div>

                <!-- Step 3: Confirm -->
                <div class="orphanix-step" data-step="3" data-orphanix-style="orphanix-s-0039">
                    <h2><?php esc_html_e('Step 3: Confirm Scan Settings','orphanix-media-cleanup'); ?></h2>
                    <div id="orphanix-media-confirm-container"></div>
                    
                    <div data-orphanix-style="orphanix-s-0044">
                        <button class="button" onclick="orphanix_prev_media_step(3)">
                            <?php esc_html_e('&larr; Back','orphanix-media-cleanup'); ?>
                        </button>
                        <button class="button button-primary" onclick="orphanix_start_media_scan()">
                            <?php esc_html_e('Start Scan &rarr;','orphanix-media-cleanup'); ?>
                        </button>
                    </div>
                </div>

                <!-- Step 4: Scanning -->
                <div class="orphanix-step" data-step="4" data-orphanix-style="orphanix-s-0039">
                    <h2><?php esc_html_e('Step 4: Scanning','orphanix-media-cleanup'); ?></h2>
                    <h3><?php esc_html_e('Analyzing Media Library','orphanix-media-cleanup'); ?></h3>
                    <p><?php esc_html_e('Discovering unused and orphaned files...','orphanix-media-cleanup'); ?></p>
                    
                    <div data-orphanix-style="orphanix-s-0017">
                        <div data-orphanix-style="orphanix-s-0045">
                            <div data-orphanix-style="orphanix-s-0046">
                                <strong><?php esc_html_e('Progress:','orphanix-media-cleanup'); ?></strong> <span id="orphanix-media-progress-percent">0</span>%
                            </div>
                            <div data-orphanix-style="orphanix-s-0047">
                                <div id="orphanix-media-progress-bar" data-orphanix-style="orphanix-s-0117"></div>
                            </div>
                        </div>

                        <div id="orphanix-media-scan-stats" data-orphanix-style="orphanix-s-0049">
                            <div data-orphanix-style="orphanix-s-0050">
                                <strong><?php esc_html_e('Total Files to Scan:','orphanix-media-cleanup'); ?></strong> <span id="orphanix-media-total-files">0</span>
                            </div>
                            <div data-orphanix-style="orphanix-s-0050">
                                <strong><?php esc_html_e('Files Scanned:','orphanix-media-cleanup'); ?></strong> <span id="orphanix-media-analyzed-files">0</span>
                            </div>
                            <div data-orphanix-style="orphanix-s-0050">
                                <strong><?php esc_html_e('Files Remaining:','orphanix-media-cleanup'); ?></strong> <span id="orphanix-media-remaining-files">0</span>
                            </div>
                            <div data-orphanix-style="orphanix-s-0050">
                                <strong><?php esc_html_e('Estimated Time:','orphanix-media-cleanup'); ?></strong> <span id="orphanix-media-eta">--</span>
                            </div>
                        </div>

                        <div id="orphanix-media-current-file" data-orphanix-style="orphanix-s-0118">
                            <small><?php esc_html_e('Current:','orphanix-media-cleanup'); ?> <span id="orphanix-media-current-file-name">--</span></small>
                        </div>
                    </div>

                    <button class="button button-secondary" onclick="orphanix_cancel_media_scan()" id="orphanix-media-cancel-btn">
                        <?php esc_html_e('Cancel Scan','orphanix-media-cleanup'); ?>
                    </button>
                </div>

                <!-- Step 5: Results -->
                <div class="orphanix-step" data-step="5" data-orphanix-style="orphanix-s-0039">
                    <h2><?php esc_html_e('Step 5: Results','orphanix-media-cleanup'); ?></h2>
                    <h3 data-orphanix-style="orphanix-s-0119">&#10003; <?php esc_html_e('Scan Completed Successfully!','orphanix-media-cleanup'); ?></h3>
                    
                    <div id="orphanix-media-results-summary"></div>

                    <div data-orphanix-style="orphanix-s-0044">
                        <button class="button" onclick="orphanix_start_new_media_scan()">
                            <?php esc_html_e('Start New Media Scan','orphanix-media-cleanup'); ?>
                        </button>
                        <a href="#" class="button button-primary" id="orphanix-media-view-results-btn">
                            <?php esc_html_e('View Results &rarr;','orphanix-media-cleanup'); ?>
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <?php
        ob_start();
        ?>
        var orphanix_media_current_scan_id = null;
        var orphanix_media_scan_type = '';

        function orphanix_select_media_scan_type(type) {
            orphanix_media_scan_type = type;
            jQuery('#orphanix-media-scan-type').val(type);
            jQuery('.orphanix-scan-option').removeClass('selected');
            jQuery('.orphanix-scan-' + type).addClass('selected');

            jQuery('button[onclick="orphanix_next_media_step(1)"]').prop('disabled', false);
        }

        function orphanix_next_media_step(current) {
            if (current === 1) {
                orphanix_load_media_settings();
                jQuery('.orphanix-step').hide();
                jQuery('[data-step="2"]').show();
            } else if (current === 2) {
                orphanix_load_media_confirm();
                jQuery('.orphanix-step').hide();
                jQuery('[data-step="3"]').show();
            }
        }

        function orphanix_prev_media_step(current) {
            jQuery('.orphanix-step').hide();
            jQuery('[data-step="' + (current - 1) + '"]').show();
        }

        function orphanix_load_media_settings() {
            var settings = orphanix_media_scan_type === 'regular' ? orphanix_get_media_regular_settings() : orphanix_get_media_deep_settings();
            jQuery('#orphanix-media-settings-container').html(settings);
        }

        function orphanix_get_media_regular_settings() {
            return `
                <div data-orphanix-style="orphanix-s-0053">
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_featured_images" value="1" checked data-orphanix-style="orphanix-s-0120">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Featured Images','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Check featured images in posts and pages.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_post_content" value="1" checked data-orphanix-style="orphanix-s-0120">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Post Content','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Search for media references in post content.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_custom_fields" value="1" data-orphanix-style="orphanix-s-0120">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Custom Fields','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Check custom field media references.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                </div>
            `;
        }

        function orphanix_get_media_deep_settings() {
            return `
                <div data-orphanix-style="orphanix-s-0053">
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_featured_images" value="1" checked data-orphanix-style="orphanix-s-0120">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Featured Images','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Check featured images across all post types.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_post_content" value="1" checked data-orphanix-style="orphanix-s-0120">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Post Content','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Search for media references in all content.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_custom_fields" value="1" checked data-orphanix-style="orphanix-s-0120">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Custom Fields','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Check all custom field media references.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_theme_files" value="1" data-orphanix-style="orphanix-s-0120">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Theme Files','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Check theme directory for media references.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                </div>
            `;
        }

        function orphanix_load_media_confirm() {
            var mode = orphanix_media_scan_type;
            var featured = jQuery('input[name="scan_featured_images"]').is(':checked') ? 1 : 0;
            var content = jQuery('input[name="scan_post_content"]').is(':checked') ? 1 : 0;
            var custom = jQuery('input[name="scan_custom_fields"]').is(':checked') ? 1 : 0;
            var theme = jQuery('input[name="scan_theme_files"]').is(':checked') ? 1 : 0;

            // Build scope settings with icons
            var scopeItems = [];
            if (featured) scopeItems.push('<span data-orphanix-style="orphanix-s-0066">&#10003;</span> <?php echo esc_js( __( 'Featured Images', 'orphanix-media-cleanup' ) ); ?>');
            else scopeItems.push('<span data-orphanix-style="orphanix-s-0067">&#10007;</span> <?php echo esc_js( __( 'Featured Images', 'orphanix-media-cleanup' ) ); ?>');
            
            if (content) scopeItems.push('<span data-orphanix-style="orphanix-s-0066">&#10003;</span> <?php echo esc_js( __( 'Post Content', 'orphanix-media-cleanup' ) ); ?>');
            else scopeItems.push('<span data-orphanix-style="orphanix-s-0067">&#10007;</span> <?php echo esc_js( __( 'Post Content', 'orphanix-media-cleanup' ) ); ?>');
            
            if (custom) scopeItems.push('<span data-orphanix-style="orphanix-s-0066">&#10003;</span> <?php echo esc_js( __( 'Custom Fields', 'orphanix-media-cleanup' ) ); ?>');
            else scopeItems.push('<span data-orphanix-style="orphanix-s-0067">&#10007;</span> <?php echo esc_js( __( 'Custom Fields', 'orphanix-media-cleanup' ) ); ?>');
            
            if (mode === 'deep') {
                if (theme) scopeItems.push('<span data-orphanix-style="orphanix-s-0066">&#10003;</span> <?php echo esc_js( __( 'Theme Files', 'orphanix-media-cleanup' ) ); ?>');
                else scopeItems.push('<span data-orphanix-style="orphanix-s-0067">&#10007;</span> <?php echo esc_js( __( 'Theme Files', 'orphanix-media-cleanup' ) ); ?>');
            }

            var scopeList = scopeItems.map(item => '<div data-orphanix-style="orphanix-s-0068">' + item + '</div>').join('');

            // Build detector compatibility list with icons
            var gutenberg = featured || content || custom ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>';
            var elementor = featured || content || custom ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>';
            var divi = featured || content || custom ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>';
            var wpbakery = featured || content || custom ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>';
            var acf = custom ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>';
            var metabox = custom ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>';
            var pods = custom ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>';
            var woo = content ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>';
            var sliders = featured || content ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>';
            var widgets = featured || content ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>';
            var menus = featured || content ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>';
            var themeFiles = (theme && mode === 'deep') ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>';

            var detectorItems = [
                gutenberg + ' <?php echo esc_js( __( 'Gutenberg Blocks', 'orphanix-media-cleanup' ) ); ?>',
                elementor + ' <?php echo esc_js( __( 'Elementor', 'orphanix-media-cleanup' ) ); ?>',
                divi + ' <?php echo esc_js( __( 'Divi Builder', 'orphanix-media-cleanup' ) ); ?>',
                wpbakery + ' <?php echo esc_js( __( 'WPBakery Page Builder', 'orphanix-media-cleanup' ) ); ?>',
                acf + ' <?php echo esc_js( __( 'Advanced Custom Fields (ACF)', 'orphanix-media-cleanup' ) ); ?>',
                metabox + ' <?php echo esc_js( __( 'Meta Box', 'orphanix-media-cleanup' ) ); ?>',
                pods + ' <?php echo esc_js( __( 'Pods', 'orphanix-media-cleanup' ) ); ?>',
                woo + ' <?php echo esc_js( __( 'WooCommerce Product Media', 'orphanix-media-cleanup' ) ); ?>',
                sliders + ' <?php echo esc_js( __( 'Sliders (Revolution, Smart Slider)', 'orphanix-media-cleanup' ) ); ?>',
                widgets + ' <?php echo esc_js( __( 'Widgets', 'orphanix-media-cleanup' ) ); ?>',
                menus + ' <?php echo esc_js( __( 'Navigation Menus', 'orphanix-media-cleanup' ) ); ?>',
                themeFiles + ' <?php echo esc_js( __( 'Theme Files (Deep Scan Only)', 'orphanix-media-cleanup' ) ); ?>'
            ];

            var detectorList = detectorItems.map((item, index) => {
                return '<div data-orphanix-style="orphanix-s-0069">' + item + '</div>';
            }).join('');

            var html = `
                <div data-orphanix-style="orphanix-s-0070">
                    <!-- Scan Type Card -->
                    <div data-orphanix-style="orphanix-s-0071">
                        <h3 data-orphanix-style="orphanix-s-0072"><?php echo esc_js( __( 'Scan Configuration', 'orphanix-media-cleanup' ) ); ?></h3>
                        <p data-orphanix-style="orphanix-s-0073">${mode === 'regular' ? '<?php echo esc_js( __( 'Regular Media Library Scan', 'orphanix-media-cleanup' ) ); ?>' : '<?php echo esc_js( __( 'Deep Filesystem Scan', 'orphanix-media-cleanup' ) ); ?>'}</p>
                    </div>

                    <!-- Scan Scope Section -->
                    <div data-orphanix-style="orphanix-s-0074">
                        <h4 data-orphanix-style="orphanix-s-0075"><?php echo esc_js( __( 'Scan Scope:', 'orphanix-media-cleanup' ) ); ?></h4>
                        <div data-orphanix-style="orphanix-s-0076">
                            ${scopeList}
                        </div>
                    </div>

                    <!-- Scan Types Section -->
                    <div data-orphanix-style="orphanix-s-0077">
                        <h4 data-orphanix-style="orphanix-s-0075"><?php echo esc_js( __( 'Scan Types:', 'orphanix-media-cleanup' ) ); ?></h4>
                        <div data-orphanix-style="orphanix-s-0076">
                            ${detectorList}
                        </div>
                    </div>

                    <!-- Safety Note -->
                    <div data-orphanix-style="orphanix-s-0078">
                        <strong><?php esc_html_e('Safety Note:','orphanix-media-cleanup'); ?></strong> <?php esc_html_e('This scan only analyzes files and does not delete anything. You will review all results before any cleanup actions.','orphanix-media-cleanup'); ?>
                    </div>
                </div>
            `;
            jQuery('#orphanix-media-confirm-container').html(html);
        }

        function orphanix_start_media_scan() {
            var featured = jQuery('input[name="scan_featured_images"]').is(':checked') ? 1 : 0;
            var content = jQuery('input[name="scan_post_content"]').is(':checked') ? 1 : 0;
            var custom = jQuery('input[name="scan_custom_fields"]').is(':checked') ? 1 : 0;
            var theme = jQuery('input[name="scan_theme_files"]').is(':checked') ? 1 : 0;

            jQuery.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'orphanix_initiate_scan',
                    scan_type: orphanix_media_scan_type,
                    mode: 'media',
                    settings: JSON.stringify({
                        featured: featured,
                        content: content,
                        custom: custom,
                        theme: theme
                    }),
                    nonce: '<?php echo esc_js( wp_create_nonce('orphanix_scan_nonce') ); ?>'
                },
                success: function(response) {
                    if (response.success && response.data.scan_id) {
                        orphanix_media_current_scan_id = response.data.scan_id;
                        jQuery('.orphanix-step').hide();
                        jQuery('[data-step="4"]').show();
                        orphanix_poll_media_progress();
                    } else {
                        alert('<?php echo esc_js( __( 'Error starting scan.', 'orphanix-media-cleanup' ) ); ?>');
                    }
                },
                error: function() {
                    alert('<?php echo esc_js( __( 'Failed to initiate scan.', 'orphanix-media-cleanup' ) ); ?>');
                }
            });
        }

        function orphanix_poll_media_progress() {
            var interval = setInterval(function(){
                jQuery.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'orphanix_get_scan_progress',
                        scan_id: orphanix_media_current_scan_id,
                        nonce: '<?php echo esc_js( wp_create_nonce('orphanix_scan_nonce') ); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            var data = response.data;
                            var percent = data.percentage || 0;
                            var total = data.total || 0;
                            var processed = data.processed || 0;
                            var status = data.status || '';
                            jQuery('#orphanix-media-progress-percent').text(Math.round(percent));
                            jQuery('#orphanix-media-progress-bar').css('width', percent + '%');
                            jQuery('#orphanix-media-total-files').text(total);
                            jQuery('#orphanix-media-analyzed-files').text(processed);
                            jQuery('#orphanix-media-remaining-files').text(total - processed);
                            jQuery('#orphanix-media-current-file-name').text(data.current_file || '--');
                            jQuery('#orphanix-media-eta').text(data.eta || '--');

                            // Fallback: if scan is stuck at 0 for several seconds, retrigger processing
                            if (status === 'running' && total === 0 && processed === 0) {
                                if (!window.orphanix_scan_retry_count) {
                                    window.orphanix_scan_retry_count = 0;
                                }
                                if (window.orphanix_scan_retry_count < 3) {
                                    window.orphanix_scan_retry_count++;
                                    jQuery.ajax({
                                        url: ajaxurl,
                                        type: 'POST',
                                        data: {
                                            action: 'orphanix_process_scan',
                                            scan_id: orphanix_media_current_scan_id,
                                            nonce: '<?php echo esc_js( wp_create_nonce('orphanix_scan_nonce') ); ?>'
                                        }
                                    });
                                }
                            }

                            if (data.status === 'complete' || data.status === 'completed') {
                                clearInterval(interval);
                                var summary = `
                                    <div data-orphanix-style="orphanix-s-0079">
                                        <div data-orphanix-style="orphanix-s-0080">
                                            <h4 data-orphanix-style="orphanix-s-0006">` + '<?php echo esc_js( __( 'Total Files', 'orphanix-media-cleanup' ) ); ?>' + `</h4>
                                            <p data-orphanix-style="orphanix-s-0081">${data.total || 0}</p>
                                        </div>
                                        <div data-orphanix-style="orphanix-s-0080">
                                            <h4 data-orphanix-style="orphanix-s-0006">` + '<?php echo esc_js( __( 'In Media Library', 'orphanix-media-cleanup' ) ); ?>' + `</h4>
                                            <p data-orphanix-style="orphanix-s-0081">${data.in_library || 0}</p>
                                        </div>
                                        <div data-orphanix-style="orphanix-s-0080">
                                            <h4 data-orphanix-style="orphanix-s-0006">` + '<?php echo esc_js( __( 'Used (Not in Media)', 'orphanix-media-cleanup' ) ); ?>' + `</h4>
                                            <p data-orphanix-style="orphanix-s-0081">${data.used_not_in_library || 0}</p>
                                        </div>
                                        <div data-orphanix-style="orphanix-s-0080">
                                            <h4 data-orphanix-style="orphanix-s-0006">` + '<?php echo esc_js( __( 'Unused Files', 'orphanix-media-cleanup' ) ); ?>' + `</h4>
                                            <p data-orphanix-style="orphanix-s-0081">${data.orphan || 0}</p>
                                        </div>
                                    </div>
                                `;
                                jQuery('#orphanix-media-results-summary').html(summary);
            jQuery('#orphanix-media-view-results-btn').attr('href', '<?php echo esc_url( add_query_arg('action', 'results', admin_url('admin.php?page=orphanix-scan')) ); ?>' + '&scan_id=' + orphanix_media_current_scan_id);
                                jQuery('.orphanix-step').hide();
                                jQuery('[data-step="5"]').show();
                            }
                        }
                    }
                });
            }, 1000);
        }

        function orphanix_cancel_media_scan() {
            jQuery.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'orphanix_cancel_scan',
                    scan_id: orphanix_media_current_scan_id,
                    nonce: '<?php echo esc_js( wp_create_nonce('orphanix_scan_nonce') ); ?>'
                },
                success: function() {
                    window.location.href = '<?php echo esc_url( admin_url('admin.php?page=orphanix-scan') ); ?>';
                }
            });
        }

        function orphanix_start_new_media_scan() {
            window.location.href = '<?php echo esc_url( admin_url('admin.php?page=orphanix-scan&action=wizard') ); ?>';
        }
        <?php
        $inline_script = trim( (string) ob_get_clean() );
        wp_add_inline_script( 'orphanix-admin', $inline_script );
        ?>
        <?php
    }

    private function render_results( $scan_id ) {
        global $wpdb;
        $results_orderby_override = null;
        $results_order_override = null;
        $status_filter_override = null;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $scan = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}orphanix_scans WHERE id = %d", $scan_id ) );
        if ( ! $scan ) {
            echo '<div class="wrap"><p>' . esc_html__( 'Scan not found', 'orphanix-media-cleanup' ) . '</p></div>';
            return;
        }
        nocache_headers();

        if ( isset( $_SERVER['REQUEST_METHOD'] ) && 'POST' === $_SERVER['REQUEST_METHOD'] ) {
            $posted_bulk_nonce = isset( $_POST['orphanix_bulk_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['orphanix_bulk_nonce'] ) ) : '';
            $allowed_order_tokens = array( 'id', 'attachment_id', 'file_path', 'directory_type', 'file_size', 'status' );
            $posted_orderby = isset( $_POST['orderby'] ) ? sanitize_key( wp_unslash( $_POST['orderby'] ) ) : '';
            $posted_order = isset( $_POST['order'] ) ? strtoupper( sanitize_text_field( wp_unslash( $_POST['order'] ) ) ) : 'DESC';
            $posted_orderby = in_array( $posted_orderby, $allowed_order_tokens, true ) ? $posted_orderby : 'id';
            $posted_order = ( 'ASC' === $posted_order ) ? 'ASC' : 'DESC';
            $results_orderby_override = $posted_orderby;
            $results_order_override = $posted_order;

            $filter_status = isset( $_POST['filter_status'] ) ? sanitize_text_field( wp_unslash( $_POST['filter_status'] ) ) : '';
            $allowed_result_filters = array( 'published', 'alt_tag_missing', 'in_media_library', 'not_in_media_library', 'used', 'not_used', 'orphan', 'trashed' );
            if ( ! in_array( $filter_status, $allowed_result_filters, true ) ) {
                $filter_status = '';
            }
            $status_filter_override = $filter_status;
            $did_process_bulk_action = false;

            if ( current_user_can( 'manage_options' ) && ! empty( $posted_bulk_nonce ) && wp_verify_nonce( $posted_bulk_nonce, 'orphanix_bulk_action' ) ) {
                $bulk_action = isset( $_POST['orphanix_bulk_action'] ) ? sanitize_text_field( wp_unslash( $_POST['orphanix_bulk_action'] ) ) : '';
                $item_ids = isset( $_POST['item_ids'] ) && is_array( $_POST['item_ids'] ) ? array_map( 'absint', wp_unslash( $_POST['item_ids'] ) ) : array();

                if ( 'trash' === $bulk_action && ! empty( $item_ids ) ) {
                    foreach ( $item_ids as $iid ) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->update( "{$wpdb->prefix}orphanix_scan_items", array( 'status' => 'trashed' ), array( 'id' => $iid ) );
                    }
                    $did_process_bulk_action = true;
                } elseif ( 'delete' === $bulk_action && ! empty( $item_ids ) ) {
                    $repo = new ORPHANIX_Scan_Items_Repository();
                    foreach ( $item_ids as $iid ) {
                        $repo->delete_item_with_attachment( $iid );
                    }
                    $did_process_bulk_action = true;
                } elseif ( 'restore' === $bulk_action && ! empty( $item_ids ) ) {
                    foreach ( $item_ids as $iid ) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->update( "{$wpdb->prefix}orphanix_scan_items", array( 'status' => '' ), array( 'id' => $iid ) );
                    }
                    $did_process_bulk_action = true;
                }
            }

            $did_request_filter = isset( $_POST['filter_action'] ) && 'filter' === sanitize_key( wp_unslash( $_POST['filter_action'] ) );
            if ( $did_process_bulk_action || $did_request_filter ) {
                $redirect_args = array(
                    'page'    => 'orphanix-scan',
                    'action'  => 'results',
                    'scan_id' => $scan_id,
                    'orderby' => $posted_orderby,
                    'order'   => $posted_order,
                );
                if ( '' !== $filter_status ) {
                    $redirect_args['filter_status'] = $filter_status;
                }
                $redirect_args['orefresh'] = time();
                $redirect_url = add_query_arg( $redirect_args, admin_url( 'admin.php' ) );
                if ( ! headers_sent() ) {
                    wp_safe_redirect( $redirect_url );
                    exit;
                }
            }

        }

        // Pagination and filtering
        $per_page = 10;
        $paged = isset( $_GET['paged'] ) ? max( 1, absint( wp_unslash( $_GET['paged'] ) ) ) : 1;
        $offset = ( $paged - 1 ) * $per_page;
        $allowed_results_orderby = array(
            'id'             => 'id',
            'attachment_id'  => 'attachment_id',
            'file_path'      => 'file_path',
            'directory_type' => 'directory_type',
            'file_size'      => 'file_size',
            'status'         => 'status',
        );
        $results_orderby_raw = isset( $_GET['orderby'] ) ? sanitize_key( wp_unslash( $_GET['orderby'] ) ) : '';
        $results_order_raw   = isset( $_GET['order'] ) ? strtoupper( sanitize_text_field( wp_unslash( $_GET['order'] ) ) ) : 'DESC';
        $results_orderby     = ( null !== $results_orderby_override ) ? $results_orderby_override : ( isset( $allowed_results_orderby[ $results_orderby_raw ] ) ? $allowed_results_orderby[ $results_orderby_raw ] : 'id' );
        $results_order       = ( null !== $results_order_override ) ? $results_order_override : ( ( 'ASC' === $results_order_raw ) ? 'ASC' : 'DESC' );
        // Filter by status
        $status_filter = ( null !== $status_filter_override ) ? $status_filter_override : ( isset( $_GET['filter_status'] ) ? sanitize_text_field( wp_unslash( $_GET['filter_status'] ) ) : '' );
        $allowed_result_filters = [ 'published', 'alt_tag_missing', 'in_media_library', 'not_in_media_library', 'used', 'not_used', 'orphan', 'trashed' ];
        if ( ! in_array( $status_filter, $allowed_result_filters, true ) ) {
            $status_filter = '';
        }
        if ( '' === $status_filter ) {
            $status_filter = 'published';
        }
        $where_parts = array( 'scan_id = %d' );
        $where_args  = array( $scan_id );

        if ( 'published' === $status_filter ) {
            $where_parts[] = 'status != %s';
            $where_args[]  = 'trashed';
        } elseif ( 'alt_tag_missing' === $status_filter ) {
            $where_parts[] = "(alt_text IS NULL OR alt_text = '')";
        } elseif ( 'in_media_library' === $status_filter ) {
            $where_parts[] = 'directory_type = %s';
            $where_args[]  = 'media';
        } elseif ( 'not_in_media_library' === $status_filter ) {
            $where_parts[] = 'directory_type != %s';
            $where_args[]  = 'media';
        } elseif ( 'used' === $status_filter ) {
            $where_parts[] = 'status = %s';
            $where_args[]  = 'used';
        } elseif ( 'not_used' === $status_filter ) {
            $where_parts[] = 'status = %s';
            $where_args[]  = 'not_used';
        } elseif ( 'orphan' === $status_filter ) {
            $where_parts[] = 'status = %s';
            $where_args[]  = 'orphan';
        } elseif ( 'trashed' === $status_filter ) {
            $where_parts[] = 'status = %s';
            $where_args[]  = 'trashed';
        }

        $where_sql      = 'WHERE ' . implode( ' AND ', $where_parts );

        // Strict allowlist for ORDER BY
        $allowed_order_columns = array(
            'id', 'attachment_id', 'file_path', 'directory_type', 'file_size', 'status'
        );
        $order_by_col = in_array( $results_orderby, $allowed_order_columns, true ) ? $results_orderby : 'id';
        $order_dir = ( 'ASC' === $results_order ) ? 'ASC' : 'DESC';
        $order_by_sql = $order_by_col . ' ' . $order_dir;

        // Build SQL with only placeholders in WHERE, and strict ORDER BY
        $where_joined = implode( ' AND ', $where_parts );
        // phpcs:disable WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber, PluginCheck.Security.DirectDB.UnescapedDBParameter
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $total_items = intval( $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE {$where_joined}", $where_args ) ) );
        $total_pages = ceil( $total_items / $per_page );
        $items_args  = array_merge( $where_args, array( $per_page, $offset ) );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $items = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}orphanix_scan_items WHERE {$where_joined} ORDER BY {$order_by_sql} LIMIT %d OFFSET %d", $items_args ) );
        // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber, PluginCheck.Security.DirectDB.UnescapedDBParameter

        // Publish/Trash counters for this result table.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $published_count = intval( $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d AND status != %s",
            $scan_id,
            'trashed'
        ) ) );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $trash_count = intval( $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d AND status = %s",
            $scan_id,
            'trashed'
        ) ) );
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__( 'Scan Results', 'orphanix-media-cleanup' ); ?></h1>
            <p>
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=orphanix-scan' ) ); ?>" class="button"><?php echo esc_html__( 'Back', 'orphanix-media-cleanup' ); ?></a>
                <button type="button" class="button button-primary" id="orphanix-run-live-scan" data-scan-id="<?php echo esc_attr( $scan_id ); ?>" data-nonce="<?php echo esc_attr( wp_create_nonce( 'orphanix_scan_nonce' ) ); ?>"><?php echo esc_html__( 'Run Live Site Scan', 'orphanix-media-cleanup' ); ?></button>
            </p>
            <?php
            $results_base_args = array(
                'page'   => 'orphanix-scan',
                'action' => 'results',
                'scan_id'=> $scan_id,
                'orderby'=> $results_orderby,
                'order'  => $results_order,
            );
            if ( '' !== $status_filter ) {
                $results_base_args['filter_status'] = $status_filter;
            }
            $results_base_url = add_query_arg( $results_base_args, admin_url( 'admin.php' ) );
            ?>
            <ul class="subsubsub">
                <li>
                    <a href="<?php echo esc_url( add_query_arg( array( 'filter_status' => 'published' ), $results_base_url ) ); ?>" class="<?php echo ( '' === $status_filter || 'published' === $status_filter ) ? 'current' : ''; ?>">
                        <?php echo esc_html__( 'Publish', 'orphanix-media-cleanup' ); ?> (<?php echo esc_html( $published_count ); ?>)
                    </a> |
                </li>
                <li>
                    <a href="<?php echo esc_url( add_query_arg( array( 'filter_status' => 'trashed' ), $results_base_url ) ); ?>" class="<?php echo ( 'trashed' === $status_filter ) ? 'current' : ''; ?>">
                        <?php echo esc_html__( 'Trash', 'orphanix-media-cleanup' ); ?> (<?php echo esc_html( $trash_count ); ?>)
                    </a>
                </li>
            </ul>
            <br class="clear" />

            <form method="post" action="<?php echo esc_url( $results_base_url ); ?>">
                <?php wp_nonce_field( 'orphanix_bulk_action', 'orphanix_bulk_nonce' ); ?>
                <input type="hidden" name="orderby" value="<?php echo esc_attr( $results_orderby ); ?>">
                <input type="hidden" name="order" value="<?php echo esc_attr( $results_order ); ?>">
                <div data-orphanix-style="orphanix-s-0018">
                    <select name="orphanix_bulk_action">
                        <option value="">-- <?php esc_html_e('Bulk Actions','orphanix-media-cleanup'); ?> --</option>
                        <?php if ( $status_filter !== 'trashed' ) : ?>
                            <option value="trash"><?php esc_html_e('Move to Trash','orphanix-media-cleanup'); ?></option>
                        <?php else : ?>
                            <option value="restore"><?php esc_html_e('Restore','orphanix-media-cleanup'); ?></option>
                            <option value="delete"><?php esc_html_e('Delete Permanently','orphanix-media-cleanup'); ?></option>
                        <?php endif; ?>
                    </select>
                    <button class="button" type="submit"><?php esc_html_e('Apply','orphanix-media-cleanup'); ?></button>
                    <span data-orphanix-style="orphanix-s-0019">
                        <label><?php esc_html_e('Filter:', 'orphanix-media-cleanup'); ?>
                            <select name="filter_status">
                                <option value="">-- <?php esc_html_e('All','orphanix-media-cleanup'); ?> --</option>
                                <option value="published" <?php selected( $status_filter, 'published' ); ?>><?php esc_html_e('Publish','orphanix-media-cleanup'); ?></option>
                                <option value="alt_tag_missing" <?php selected( $status_filter, 'alt_tag_missing' ); ?>><?php esc_html_e('Alt Tag Missing','orphanix-media-cleanup'); ?></option>
                                <option value="in_media_library" <?php selected( $status_filter, 'in_media_library' ); ?>><?php esc_html_e('In Media Library','orphanix-media-cleanup'); ?></option>
                                <option value="not_in_media_library" <?php selected( $status_filter, 'not_in_media_library' ); ?>><?php esc_html_e('Not in Media Library','orphanix-media-cleanup'); ?></option>
                                <option value="used" <?php selected( $status_filter, 'used' ); ?>><?php esc_html_e('Used','orphanix-media-cleanup'); ?></option>
                                <option value="not_used" <?php selected( $status_filter, 'not_used' ); ?>><?php esc_html_e('Not Used','orphanix-media-cleanup'); ?></option>
                                <option value="orphan" <?php selected( $status_filter, 'orphan' ); ?>><?php esc_html_e('Orphan','orphanix-media-cleanup'); ?></option>
                                <option value="trashed" <?php selected( $status_filter, 'trashed' ); ?>><?php esc_html_e('Trashed','orphanix-media-cleanup'); ?></option>
                            </select>
                        </label>
                        <button class="button" type="submit" name="filter_action" value="filter"><?php esc_html_e('Filter','orphanix-media-cleanup'); ?></button>
                    </span>
                </div>

                <table class="widefat striped" data-orphanix-style="orphanix-s-0082">
                    <thead>
                        <tr>
                            <th data-orphanix-style="orphanix-s-0020"><input type="checkbox" class="orphanix-select-all"></th>
                            <th data-orphanix-style="orphanix-s-0083"><?php esc_html_e('Thumb', 'orphanix-media-cleanup'); ?></th>
                            <th><a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'id', 'order' => ( 'id' === $results_orderby && 'ASC' === $results_order ) ? 'DESC' : 'ASC', 'paged' => 1 ), $results_base_url ) ); ?>"><?php esc_html_e('ID', 'orphanix-media-cleanup'); ?></a></th>
                            <th><?php esc_html_e('Title', 'orphanix-media-cleanup'); ?></th>
                            <th><a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'file_path', 'order' => ( 'file_path' === $results_orderby && 'ASC' === $results_order ) ? 'DESC' : 'ASC', 'paged' => 1 ), $results_base_url ) ); ?>"><?php esc_html_e('Path', 'orphanix-media-cleanup'); ?></a></th>
                            <th><a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'directory_type', 'order' => ( 'directory_type' === $results_orderby && 'ASC' === $results_order ) ? 'DESC' : 'ASC', 'paged' => 1 ), $results_base_url ) ); ?>"><?php esc_html_e('Directory', 'orphanix-media-cleanup'); ?></a></th>
                            <th><?php esc_html_e('Alt Tag', 'orphanix-media-cleanup'); ?></th>
                            <th><a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'file_size', 'order' => ( 'file_size' === $results_orderby && 'ASC' === $results_order ) ? 'DESC' : 'ASC', 'paged' => 1 ), $results_base_url ) ); ?>"><?php esc_html_e('Size', 'orphanix-media-cleanup'); ?></a></th>
                            <th><?php esc_html_e('Belongs to', 'orphanix-media-cleanup'); ?></th>
                            <th><a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'status', 'order' => ( 'status' === $results_orderby && 'ASC' === $results_order ) ? 'DESC' : 'ASC', 'paged' => 1 ), $results_base_url ) ); ?>"><?php esc_html_e('Status', 'orphanix-media-cleanup'); ?></a></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ( $items ) : 
                            foreach ( $items as $item ) : 
                                $thumbnail = '';
                                if ( ! empty( $item->file_url ) && preg_match( '/\.(jpg|jpeg|png|gif|webp)$/i', $item->file_url ) ) {
                                    $thumbnail = '<img src="' . esc_url( $item->file_url ) . '" data-orphanix-style="orphanix-s-0084" />';
                                } else {
                                    $thumbnail = '&ndash;';
                                }
                        ?>
                            <tr>
                                <td><input type="checkbox" name="item_ids[]" value="<?php echo esc_attr( $item->id ); ?>" /></td>
                                <td><?php echo wp_kses_post( $thumbnail ); ?></td>
                                <td><?php echo esc_html( (string) $item->id ); ?></td>
                                <td><?php echo esc_html( basename( $item->file_path ) ); ?></td>
                                <td><a href="<?php echo esc_url( $item->file_url ); ?>" target="_blank" title="<?php echo esc_attr( $item->file_url ); ?>" data-orphanix-style="orphanix-s-0085"><?php echo esc_html( wp_parse_url( $item->file_url, PHP_URL_PATH ) ); ?></a></td>
                                <td><?php echo esc_html( 'media' === $item->directory_type ? esc_html__( 'In Media Library', 'orphanix-media-cleanup' ) : esc_html__( 'Not in Media Library', 'orphanix-media-cleanup' ) ); ?></td>
                                <td><?php echo ! empty( $item->alt_text ) ? esc_html( $item->alt_text ) : wp_kses_post( '<span data-orphanix-style="orphanix-s-0086">' . esc_html__( 'Missing', 'orphanix-media-cleanup' ) . '</span>' ); ?></td>
                                <td><?php echo esc_html( size_format( $item->file_size ) ); ?></td>
                                <td><?php echo wp_kses_post( $this->render_usage_context( $item->usage_context, isset( $item->attachment_id ) ? absint( $item->attachment_id ) : 0 ) ); ?></td>
                                <td>
                                    <span class="orphanix-status-pill <?php echo esc_attr( 'orphanix-status-pill--' . sanitize_html_class( $item->status ) ); ?>">
                                        <?php echo esc_html( ucfirst( str_replace( '_', ' ', $item->status ) ) ); ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; 
                        else : ?>
                            <tr><td colspan="10"><?php echo esc_html__( 'No results found.', 'orphanix-media-cleanup' ); ?></td></tr>
                        <?php endif; ?>
                    </tbody>
                </table>

                <!-- Pagination -->
                <?php if ( $total_items > 0 ) : ?>
                    <div class="tablenav-pages" data-orphanix-style="orphanix-s-0022">
                        <span class="displaying-num"><?php echo esc_html( intval( $total_items ) ); ?> <?php esc_html_e( 'items', 'orphanix-media-cleanup' ); ?></span>
                        <span class="pagination-links" data-orphanix-style="orphanix-s-0023">
                            <?php
                            $current_url_base = $results_base_url;
                            $current_url_base = add_query_arg(
                                array(
                                    'orderby' => $results_orderby,
                                    'order'   => $results_order,
                                ),
                                $current_url_base
                            );
                            
                            // First page button
                            if ( $paged <= 1 ) {
                                echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&laquo;</span>';
                            } else {
                                echo '<a class="first-page button" href="' . esc_url( add_query_arg( 'paged', 1, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'First page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&laquo;</span></a>';
                            }
                            
                            // Previous page button
                            if ( $paged <= 1 ) {
                                echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&lsaquo;</span>';
                            } else {
                                echo '<a class="prev-page button" href="' . esc_url( add_query_arg( 'paged', $paged - 1, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'Previous page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&lsaquo;</span></a>';
                            }
                            ?>
                            <span class="screen-reader-text"><?php esc_html_e( 'Current Page', 'orphanix-media-cleanup' ); ?></span>
                            <span class="paging-input">
                                <span class="tablenav-paging-text"><?php echo esc_html( intval( $paged ) ); ?> <?php esc_html_e( 'of', 'orphanix-media-cleanup' ); ?> <span class="total-pages"><?php echo esc_html( intval( $total_pages ) ); ?></span></span>
                            </span>
                            <?php
                            // Next page button
                            if ( $paged >= $total_pages ) {
                                echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&rsaquo;</span>';
                            } else {
                                echo '<a class="next-page button" href="' . esc_url( add_query_arg( 'paged', $paged + 1, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'Next page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&rsaquo;</span></a>';
                            }
                            
                            // Last page button
                            if ( $paged >= $total_pages ) {
                                echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&raquo;</span>';
                            } else {
                                echo '<a class="last-page button" href="' . esc_url( add_query_arg( 'paged', $total_pages, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'Last page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&raquo;</span></a>';
                            }
                            ?>
                        </span>
                    </div>
            <?php endif; ?>
            </form>
        </div>

        <?php
        ob_start();
        ?>
        (function($) {
            $(document).ready(function() {
                $('.orphanix-select-all').on('change', function() {
                    $('input[name="item_ids[]"]').prop('checked', this.checked);
                });

                // Live Site Scan button handler
                $('#orphanix-run-live-scan').on('click', function(e) {
                    e.preventDefault();
                    var $btn = $(this);
                    var scanId = $btn.data('scan-id');
                    var i18n = {
                        scanIdNotFound: '<?php echo esc_js( __( 'Scan ID not found', 'orphanix-media-cleanup' ) ); ?>',
                        startingLiveScan: '<?php echo esc_js( __( 'Starting Live Scan...', 'orphanix-media-cleanup' ) ); ?>',
                        liveSiteScan: '<?php echo esc_js( __( 'Live Site Scan', 'orphanix-media-cleanup' ) ); ?>',
                        pages: '<?php echo esc_js( __( 'Pages', 'orphanix-media-cleanup' ) ); ?>',
                        progress: '<?php echo esc_js( __( 'Progress', 'orphanix-media-cleanup' ) ); ?>',
                        calculating: '<?php echo esc_js( __( 'Calculating...', 'orphanix-media-cleanup' ) ); ?>',
                        analyzing: '<?php echo esc_js( __( 'Analyzing...', 'orphanix-media-cleanup' ) ); ?>',
                        estimatedTime: '<?php echo esc_js( __( 'Estimated time:', 'orphanix-media-cleanup' ) ); ?>',
                        remaining: '<?php echo esc_js( __( 'remaining', 'orphanix-media-cleanup' ) ); ?>',
                        current: '<?php echo esc_js( __( 'Current:', 'orphanix-media-cleanup' ) ); ?>',
                        scanComplete: '<?php echo esc_js( __( 'Scan Complete!', 'orphanix-media-cleanup' ) ); ?>',
                        liveScanFinished: '<?php echo esc_js( __( 'Live Site Scan has finished successfully. View the results to see which pages are using your media files.', 'orphanix-media-cleanup' ) ); ?>',
                        startNewLiveScan: '<?php echo esc_js( __( 'Start New Live Site Scan', 'orphanix-media-cleanup' ) ); ?>',
                        viewResults: '<?php echo esc_js( __( 'View Results', 'orphanix-media-cleanup' ) ); ?>',
                        close: '<?php echo esc_js( __( 'Close', 'orphanix-media-cleanup' ) ); ?>',
                        error: '<?php echo esc_js( __( 'Error:', 'orphanix-media-cleanup' ) ); ?>'
                    };

                    if (!scanId) {
                        alert(i18n.scanIdNotFound);
                        return;
                    }

                    // Disable button and show loading state
                    $btn.prop('disabled', true);
                    var originalText = $btn.text();
                    $btn.text(i18n.startingLiveScan);

                    // Create progress modal
                    var progressModal = $('<div id="orphanix-live-scan-modal" data-orphanix-style="orphanix-s-0121">')
                        .append(
                            $('<div data-orphanix-style="orphanix-s-0122">')
                                .append('<h3 data-orphanix-style="orphanix-s-0123">' + i18n.liveSiteScan + '</h3>')
                                
                                .append('<div data-orphanix-style="orphanix-s-0124"><div data-orphanix-style="orphanix-s-0125"><span data-orphanix-style="orphanix-s-0126">' + i18n.pages + '</span><span id="orphanix-pages-count" data-orphanix-style="orphanix-s-0127">0/0</span></div><div data-orphanix-style="orphanix-s-0128"><div id="orphanix-pages-progress-fill" data-orphanix-style="orphanix-s-0129"></div></div></div>')
                                
                                .append('<div data-orphanix-style="orphanix-s-0124"><div data-orphanix-style="orphanix-s-0125"><span data-orphanix-style="orphanix-s-0126">' + i18n.progress + '</span><span id="orphanix-scan-percentage" data-orphanix-style="orphanix-s-0127">0%</span></div><div data-orphanix-style="orphanix-s-0128"><div id="orphanix-progress-fill" data-orphanix-style="orphanix-s-0130"></div></div></div>')
                                
                                .append('<p id="orphanix-eta" data-orphanix-style="orphanix-s-0131">' + i18n.calculating + '</p>')
                                .append('<p id="orphanix-scan-url" data-orphanix-style="orphanix-s-0132">' + i18n.analyzing + '</p>')
                        );
                    $('body').append(progressModal);
                    progressModal.fadeIn();

                    // Track start time for ETA calculation
                    var startTime = Date.now();
                    var lastProcessed = 0;
                    var completionHandled = false;

                    // Start polling for progress immediately
                    var progressInterval = setInterval(function() {
                        $.ajax({
                            url: ajaxurl,
                            type: 'POST',
                            data: {
                                action: 'orphanix_get_live_scan_progress',
                                scan_id: scanId,
                                nonce: $btn.data('nonce') || ''
                            },
                            success: function(progressResponse) {
                                if (progressResponse.success && progressResponse.data) {
                                    var progress = progressResponse.data;
                                    var percentage = progress.percentage || 0;
                                    var processed = progress.processed || 0;
                                    var total = progress.total || 0;
                                    var status = progress.status || 'Running';
                                    var currentUrl = progress.current_url || '';

                                    // Update pages progress
                                    var pagesPercentage = total > 0 ? (processed / total * 100) : 0;
                                    $('#orphanix-pages-progress-fill').css('width', pagesPercentage + '%');
                                    $('#orphanix-pages-count').text(processed + ' / ' + total + ' pages');

                                    // Update overall progress
                                    $('#orphanix-progress-fill').css('width', percentage + '%');
                                    $('#orphanix-scan-percentage').text(percentage + '%');
                                    
                                    // Calculate ETA
                                    if (processed > lastProcessed && total > 0) {
                                        var elapsed = (Date.now() - startTime) / 1000; // seconds
                                        var rate = processed / elapsed; // pages per second
                                        var remaining = total - processed;
                                        var eta = remaining / rate; // seconds
                                        
                                        if (eta > 60) {
                                            var minutes = Math.floor(eta / 60);
                                            var seconds = Math.floor(eta % 60);
                                            $('#orphanix-eta').text(i18n.estimatedTime + ' ' + minutes + 'm ' + seconds + 's ' + i18n.remaining);
                                        } else {
                                            $('#orphanix-eta').text(i18n.estimatedTime + ' ' + Math.floor(eta) + 's ' + i18n.remaining);
                                        }
                                        
                                        lastProcessed = processed;
                                    }
                                    
                                    if (currentUrl) {
                                        $('#orphanix-scan-url').text(i18n.current + ' ' + currentUrl);
                                    }

                                    // Check if complete (only once)
                                    if (!completionHandled && (percentage >= 100 || status === 'completed')) {
                                        completionHandled = true;
                                        clearInterval(progressInterval);
                                        
                                        // Clear all existing content and update with completion message
                                        var modalContent = progressModal.find('div').first();
                                        modalContent.html(
                                            '<h2 data-orphanix-style="orphanix-s-0133">&#10003; ' + i18n.scanComplete + '</h2>' +
                                            '<p data-orphanix-style="orphanix-s-0134">' + i18n.liveScanFinished + '</p>' +
                                            '<div data-orphanix-style="orphanix-s-0135">' +
                                            '<button id="orphanix-new-scan" class="button button-secondary">' + i18n.startNewLiveScan + '</button>' +
                                            '<button id="orphanix-view-results" class="button button-primary">' + i18n.viewResults + '</button>' +
                                            '<button id="orphanix-close-modal" class="button">' + i18n.close + '</button>' +
                                            '</div>'
                                        );
                                        
                                        // Handle button clicks
                                        $('#orphanix-new-scan').on('click', function() {
                                            progressModal.fadeOut(function() {
                                                $(this).remove();
                                            });
                                            $btn.trigger('click');
                                        });
                                        
                                        $('#orphanix-view-results').on('click', function() {
                                            window.location.href = '<?php echo esc_url( admin_url('admin.php?page=orphanix-live') ); ?>';
                                        });
                                        
                                        $('#orphanix-close-modal').on('click', function() {
                                            progressModal.fadeOut(function() {
                                                $(this).remove();
                                            });
                                            $btn.prop('disabled', false);
                                            $btn.text(originalText);
                                        });
                                    }
                                }
                            }
                        });
                    }, 500);

                    // Start live scan via AJAX (this will take time to complete)
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        timeout: 0, // No timeout - can take long
                        data: {
                            action: 'orphanix_start_live_scan',
                            scan_id: scanId,
                            nonce: $btn.data('nonce') || ''
                        },
                        error: function(xhr, status, error) {
                            clearInterval(progressInterval);
                            alert(i18n.error + ' ' + error);
                            $btn.prop('disabled', false);
                            $btn.text(originalText);
                            progressModal.fadeOut(function() {
                                $(this).remove();
                            });
                        }
                    });
                });
            });
        })(jQuery);
        <?php
        $inline_script = trim( (string) ob_get_clean() );
        wp_add_inline_script( 'orphanix-admin', $inline_script );
        ?>
        <?php
    }

    /**
     * Render "Belongs To" column with formatted usage context
     */
    private function render_usage_context( $usage_context, $attachment_id = 0 ) {
        if ( empty( $usage_context ) ) {
            return '&ndash;';
        }

        // Decode JSON
        $context = json_decode( $usage_context, true );
        if ( ! is_array( $context ) ) {
            return esc_html( $usage_context );
        }

        $post_ids = $context['post_ids'] ?? [];
        $sources = $context['sources'] ?? [];

        if ( empty( $post_ids ) && empty( $sources ) ) {
            return '<span data-orphanix-style="orphanix-s-0087">Orphan</span>';
        }

        // Source label map
        $source_map = [
            'featured_image' => ['Featured Image', '#e74c3c'],
            'post_content' => ['Post Content', '#3498db'],
            'custom_field' => ['Custom Field', '#9b59b6'],
            'gutenberg' => ['Gutenberg Block', '#f39c12'],
            'elementor' => ['Elementor', '#4ecdc4'],
            'divi' => ['Divi', '#2ecc71'],
            'wpbakery' => ['WPBakery', '#e91e63'],
            'acf' => ['ACF', '#ff6b6b'],
            'metabox' => ['MetaBox', '#4a90e2'],
            'pods' => ['Pods', '#50c878'],
            'woocommerce' => ['WooCommerce', '#9c27b0'],
            'sliders' => ['Sliders', '#ff9800'],
            'widget' => ['Widget', '#795548'],
            'menu' => ['Menu', '#607d8b'],
            'theme' => ['Theme', '#455a64'],
            'deep_db' => ['Database', '#2c3e50'],
        ];

        $html = '<div data-orphanix-style="orphanix-s-0088">';

        // Separate numeric post IDs from table names
        $numeric_post_ids = [];
        $table_names = [];

        foreach ( $post_ids as $pid ) {
            if ( is_numeric( $pid ) ) {
                $numeric_post_ids[] = intval( $pid );
            } else {
                $table_names[] = $pid;
            }
        }

        if ( empty( $numeric_post_ids ) && $attachment_id > 0 ) {
            $numeric_post_ids[] = $attachment_id;
        }

        if ( empty( $table_names ) && ! empty( $sources ) ) {
            $inferred_tables = array();
            foreach ( $sources as $source_key ) {
                if ( in_array( $source_key, array( 'post_content', 'featured_image', 'gutenberg', 'menu', 'widget' ), true ) ) {
                    $inferred_tables[] = 'wp_posts';
                }
                if ( in_array( $source_key, array( 'custom_field', 'acf', 'metabox', 'pods', 'woocommerce' ), true ) ) {
                    $inferred_tables[] = 'wp_postmeta';
                }
            }
            $table_names = array_values( array_unique( $inferred_tables ) );
        }

        // Display post titles with links
        if ( ! empty( $numeric_post_ids ) ) {
            $html .= '<div data-orphanix-style="orphanix-s-0089">';
            
            foreach ( array_unique( $numeric_post_ids ) as $post_id ) {
                $post = get_post( $post_id );
                if ( $post ) {
                    $post_type_obj = get_post_type_object( $post->post_type );
                    $post_type_label = $post_type_obj ? $post_type_obj->labels->singular_name : ucfirst( $post->post_type );
                    $edit_url = get_edit_post_link( $post_id );
                    $post_title = ! empty( $post->post_title ) ? $post->post_title : sprintf( '(ID: %d)', $post_id );

                    $html .= '<div data-orphanix-style="orphanix-s-0090">';
                    $html .= '<a href="' . esc_url( $edit_url ) . '" data-orphanix-style="orphanix-s-0091">' . esc_html( $post_title ) . '</a>';
                    $html .= '<span data-orphanix-style="orphanix-s-0092">' . esc_html( $post_type_label ) . '</span>';
                    $html .= '</div>';
                }
            }
            
            $html .= '</div>';
        }

        // Display sources as badges
        if ( ! empty( $sources ) ) {
            $html .= '<div data-orphanix-style="orphanix-s-0089">';
            $html .= '<span data-orphanix-style="orphanix-s-0093" title="The source badges showing where the image is used">Source: i</span>';
            
            foreach ( $sources as $source ) {
                $source_info = $source_map[ $source ] ?? [ $source, '#95a5a6' ];
                list( $label, $color ) = $source_info;

                $html .= '<span data-orphanix-style="orphanix-s-0094">' . esc_html( $label ) . '</span>';
            }
            
            $html .= '</div>';
        }

        // Display database table names
        if ( ! empty( $table_names ) ) {
            $html .= '<div data-orphanix-style="orphanix-s-0095">';
            $html .= '<strong data-orphanix-style="orphanix-s-0096" title="Database tables where the image was found">Tables: i</strong> ' . esc_html( implode( ', ', $table_names ) );
            $html .= '</div>';
        }

        $html .= '</div>';

        // Allow necessary HTML tags with style attributes
        $allowed_html = [
            'div' => ['style' => true, 'title' => true, 'data-orphanix-style' => true],
            'span' => ['style' => true, 'title' => true, 'data-orphanix-style' => true],
            'a' => ['href' => true, 'style' => true, 'title' => true, 'data-orphanix-style' => true],
            'strong' => ['style' => true, 'title' => true, 'data-orphanix-style' => true],
        ];

        return wp_kses( $html, $allowed_html );
    }
}
