<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class ORPHANIX_Live_Scan_Page {
    public function render() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'orphanix_scan_items';
        $details_table = $wpdb->prefix . 'orphanix_live_scan_details';
        
        // Pagination
        $per_page = 10;
        $page_nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
        if ( $page_nonce && ! wp_verify_nonce( $page_nonce, 'orphanix_live_scan_page' ) ) {
            $page_nonce = '';
        }
        $current_page = max( 1, isset( $_GET['lscan_paged'] ) ? absint( wp_unslash( $_GET['lscan_paged'] ) ) : 1 );
        $offset = ($current_page - 1) * $per_page;
        
        // Get items found during live scans (live_used = 1)
        $live_used_count = $this->get_cached_var(
            'live_used_count',
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE live_used = %d",
            [ 1 ]
        );
        
        // Get total URLs where items were found
        $total_live_urls = $this->get_cached_var(
            'live_total_urls',
            "SELECT SUM(live_url_count) FROM {$wpdb->prefix}orphanix_scan_items WHERE live_used = %d",
            [ 1 ]
        );
        $total_live_urls = $total_live_urls ?: 0;
        
        // Get total items for pagination
        $total_items = intval($live_used_count);
        $total_pages = ceil($total_items / $per_page);
        
        // Get items with their URLs (all pages that used them)
        $live_items = $this->get_cached_results(
            'live_items_' . $current_page . '_' . $per_page,
            "SELECT si.id, si.attachment_id, si.file_path, si.file_url, si.file_size, si.live_url_count,
                    ld.page_url, ld.page_title
             FROM {$wpdb->prefix}orphanix_scan_items si
             LEFT JOIN {$wpdb->prefix}orphanix_live_scan_details ld ON si.id = ld.scan_item_id
             WHERE si.live_used = 1 
             ORDER BY si.live_url_count DESC, si.id DESC
             LIMIT %d OFFSET %d",
            [ $per_page, $offset ]
        );
        $settings = [
            [
                'label' => __('PHP max_execution_time', 'orphanix-media-cleanup'),
                'current' => ini_get('max_execution_time') . 's',
                'required' => '300s',
                'ok' => intval(ini_get('max_execution_time')) >= 300 || intval(ini_get('max_execution_time')) === 0,
            ],
            [
                'label' => __('PHP memory_limit', 'orphanix-media-cleanup'),
                'current' => ini_get('memory_limit'),
                'required' => '256M',
                'ok' => $this->memory_limit_to_bytes(ini_get('memory_limit')) >= $this->memory_limit_to_bytes('256M'),
            ],
            [
                'label' => __('libxml / DOMDocument', 'orphanix-media-cleanup'),
                'current' => class_exists('DOMDocument') ? __('Enabled', 'orphanix-media-cleanup') : __('Missing', 'orphanix-media-cleanup'),
                'required' => __('Enabled', 'orphanix-media-cleanup'),
                'ok' => class_exists('DOMDocument'),
            ],
            [
                'label' => __('WP HTTP API (wp_remote_get)', 'orphanix-media-cleanup'),
                'current' => function_exists('wp_remote_get') ? __('Available', 'orphanix-media-cleanup') : __('Missing', 'orphanix-media-cleanup'),
                'required' => __('Available', 'orphanix-media-cleanup'),
                'ok' => function_exists('wp_remote_get'),
            ],
        ];
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Live Site Scan','orphanix-media-cleanup'); ?></h1>
            <p><?php esc_html_e('View media files detected during Live Site Scans by analyzing your front-end HTML and rendered content.','orphanix-media-cleanup'); ?></p>

            <div data-orphanix-style="orphanix-s-0097">
                <h2><?php esc_html_e('Live Site Scan Requirements','orphanix-media-cleanup'); ?></h2>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Setting', 'orphanix-media-cleanup'); ?></th>
                            <th><?php esc_html_e('Current', 'orphanix-media-cleanup'); ?></th>
                            <th><?php esc_html_e('Required', 'orphanix-media-cleanup'); ?></th>
                            <th><?php esc_html_e('Status', 'orphanix-media-cleanup'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($settings as $setting) : ?>
                            <tr>
                                <td><?php echo esc_html($setting['label']); ?></td>
                                <td><?php echo esc_html($setting['current']); ?></td>
                                <td><?php echo esc_html($setting['required']); ?></td>
                                <td>
                                    <?php if ($setting['ok']) : ?>
                                        <span data-orphanix-style="orphanix-s-0098">&#10003; <?php esc_html_e('OK', 'orphanix-media-cleanup'); ?></span>
                                    <?php else : ?>
                                        <span data-orphanix-style="orphanix-s-0099">&#10007; <?php esc_html_e('Needs update', 'orphanix-media-cleanup'); ?></span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <div data-orphanix-style="orphanix-s-0100">
                <div data-orphanix-style="orphanix-s-0008">
                    <p data-orphanix-style="orphanix-s-0009"><?php esc_html_e('Media Files Found', 'orphanix-media-cleanup'); ?></p>
                    <p data-orphanix-style="orphanix-s-0010"><?php echo intval($live_used_count); ?></p>
                </div>
                <div data-orphanix-style="orphanix-s-0013">
                    <p data-orphanix-style="orphanix-s-0009"><?php esc_html_e('Total Page Appearances', 'orphanix-media-cleanup'); ?></p>
                    <p data-orphanix-style="orphanix-s-0014"><?php echo intval($total_live_urls); ?></p>
                </div>
            </div>

            <div data-orphanix-style="orphanix-s-0101">
                <h2><?php esc_html_e('All Media Files from Live Scans','orphanix-media-cleanup'); ?></h2>
                
                <?php if (empty($live_items)): ?>
                    <div data-orphanix-style="orphanix-s-0102">
                        <p><?php esc_html_e('No live scan data yet. Run a Media Scan and then click "Run Live Site Scan" on the results page to detect dynamically-loaded media.','orphanix-media-cleanup'); ?></p>
                    </div>
                <?php else: ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('File Path','orphanix-media-cleanup'); ?></th>
                                <th><?php esc_html_e('Size','orphanix-media-cleanup'); ?></th>
                                <th><?php esc_html_e('Belongs to','orphanix-media-cleanup'); ?></th>
                                <th><?php esc_html_e('Page Title','orphanix-media-cleanup'); ?></th>
                                <th><?php esc_html_e('Page URL','orphanix-media-cleanup'); ?></th>
                                <th><?php esc_html_e('View','orphanix-media-cleanup'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($live_items as $item): ?>
                                <?php
                                    $belongs_to_html = '-';
                                    if (!empty($item->attachment_id)) {
                                        $parent_id = wp_get_post_parent_id($item->attachment_id);
                                        if ($parent_id) {
                                            $parent_title = get_the_title($parent_id);
                                            $parent_type = get_post_type($parent_id);
                                            $type_obj = $parent_type ? get_post_type_object($parent_type) : null;
                                            $type_label = $type_obj && !empty($type_obj->labels->singular_name) ? $type_obj->labels->singular_name : $parent_type;

                                            $belongs_to_html = sprintf(
                                                '<a href="%s" data-orphanix-style="orphanix-s-0103">%s</a> <span data-orphanix-style="orphanix-s-0104">%s</span>',
                                                esc_url(get_edit_post_link($parent_id)),
                                                esc_html($parent_title ?: __('(No title)','orphanix-media-cleanup')),
                                                esc_html($type_label ?: __('Post','orphanix-media-cleanup'))
                                            );
                                        } else {
                                            /* translators: %d: attachment ID. */
                                            $attachment_label = sprintf( __( 'Attachment #%d', 'orphanix-media-cleanup' ), intval( $item->attachment_id ) );
                                            $belongs_to_html = sprintf(
                                                '<span data-orphanix-style="orphanix-s-0105">%s</span>',
                                                esc_html( $attachment_label )
                                            );
                                        }
                                    }
                                ?>
                                <tr>
                                    <td><code data-orphanix-style="orphanix-s-0106"><?php echo esc_html($item->file_path); ?></code></td>
                                    <td><?php echo esc_html($this->format_bytes(intval($item->file_size))); ?></td>
                                    <td><?php echo wp_kses_post($belongs_to_html); ?></td>
                                    <td><?php echo esc_html( $item->page_title ?: __( 'Unknown', 'orphanix-media-cleanup' ) ); ?></td>
                                    <td><code data-orphanix-style="orphanix-s-0107"><?php echo esc_html(substr($item->page_url, 0, 50)); ?></code></td>
                                    <td><a href="<?php echo esc_url($item->page_url); ?>" target="_blank" class="button button-small"><?php esc_html_e('View Page', 'orphanix-media-cleanup'); ?></a></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>

                    <!-- Pagination -->
                    <div class="tablenav bottom">
                        <div class="tablenav-pages">
                            <?php /* translators: %d: total items count. */ ?>
                            <span class="displaying-num"><?php echo esc_html( sprintf( _n('%d item', '%d items', $total_items, 'orphanix-media-cleanup'), $total_items ) ); ?></span>
                            <span class="pagination-links">
                                <?php if ($current_page > 1): ?>
                                    <a class="first-page button" href="<?php echo esc_url( wp_nonce_url( add_query_arg('lscan_paged', 1), 'orphanix_live_scan_page' ) ); ?>">&laquo;</a>
                                    <a class="prev-page button" href="<?php echo esc_url( wp_nonce_url( add_query_arg('lscan_paged', $current_page - 1), 'orphanix_live_scan_page' ) ); ?>">&lsaquo;</a>
                                <?php else: ?>
                                    <span class="tablenav-pages-navspan button disabled">&laquo;</span>
                                    <span class="tablenav-pages-navspan button disabled">&lsaquo;</span>
                                <?php endif; ?>
                                <span class="screen-reader-text"><?php esc_html_e('Current Page', 'orphanix-media-cleanup'); ?></span>
                                <span class="paging-input">
                                    <?php /* translators: 1: current page, 2: total pages. */ ?>
                                    <span class="tablenav-paging-text"><?php echo esc_html( sprintf( _x('%1$d of %2$d', 'paging', 'orphanix-media-cleanup'), $current_page, max( 1, $total_pages ) ) ); ?></span>
                                </span>
                                <?php if ($current_page < $total_pages): ?>
                                    <a class="next-page button" href="<?php echo esc_url( wp_nonce_url( add_query_arg('lscan_paged', $current_page + 1), 'orphanix_live_scan_page' ) ); ?>">&rsaquo;</a>
                                    <a class="last-page button" href="<?php echo esc_url( wp_nonce_url( add_query_arg('lscan_paged', $total_pages), 'orphanix_live_scan_page' ) ); ?>">&raquo;</a>
                                <?php else: ?>
                                    <span class="tablenav-pages-navspan button disabled">&rsaquo;</span>
                                    <span class="tablenav-pages-navspan button disabled">&raquo;</span>
                                <?php endif; ?>
                            </span>
                        </div>
                    </div>
                <?php endif; ?>
            </div>

            <div data-orphanix-style="orphanix-s-0108">
                <h3><?php esc_html_e('How Live Site Scan Works','orphanix-media-cleanup'); ?></h3>
                <ol>
                    <li><?php esc_html_e('Run a Media Scan from the "Media Scan" menu','orphanix-media-cleanup'); ?></li>
                    <li><?php esc_html_e('Go to the scan results and click "Run Live Site Scan"','orphanix-media-cleanup'); ?></li>
                    <li><?php esc_html_e('The scanner fetches your site\'s front-end pages and analyzes the rendered HTML','orphanix-media-cleanup'); ?></li>
                    <li><?php esc_html_e('Any media files found in img tags, background-images, video/audio sources are marked as "used"','orphanix-media-cleanup'); ?></li>
                    <li><?php esc_html_e('Results appear here and in the Media Scan results table','orphanix-media-cleanup'); ?></li>
                </ol>
            </div>
        </div>
        <?php
    }

    private function memory_limit_to_bytes($value) {
        $value = trim($value);
        if ($value === '' || $value === '-1') {
            return PHP_INT_MAX;
        }
        $unit = strtolower(substr($value, -1));
        $number = (int) $value;

        switch ($unit) {
            case 'g':
                return $number * 1024 * 1024 * 1024;
            case 'm':
                return $number * 1024 * 1024;
            case 'k':
                return $number * 1024;
            default:
                return (int) $value;
        }
    }

    private function get_cached_var( $cache_key, $query, array $args ) {
        $cached = wp_cache_get( $cache_key, 'orphanix' );
        if ( false !== $cached ) {
            return $cached;
        }

        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $value = $wpdb->get_var( $wpdb->prepare( $query, $args ) );
        wp_cache_set( $cache_key, $value, 'orphanix', 300 );
        return $value;
    }

    private function get_cached_results( $cache_key, $query, array $args ) {
        $cached = wp_cache_get( $cache_key, 'orphanix' );
        if ( false !== $cached ) {
            return $cached;
        }

        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $results = $wpdb->get_results( $wpdb->prepare( $query, $args ) );
        wp_cache_set( $cache_key, $results, 'orphanix', 300 );
        return $results;
    }

    private function format_bytes($bytes) {
        $bytes = intval($bytes);
        $units = array('B', 'KB', 'MB', 'GB');
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= (1 << (10 * $pow));
        return round($bytes, 2) . ' ' . $units[$pow];
    }
}

