<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class ORPHANIX_Broken_Scan {
    public function render() {
        $action = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : '';
        $scan_id = isset($_GET['scan_id']) ? absint(wp_unslash($_GET['scan_id'])) : 0;
        $nonce = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : '';

        if ( 'wizard' === $action ) {
            if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'orphanix_broken_wizard' ) ) {
                wp_die( esc_html__( 'Invalid request.', 'orphanix-media-cleanup' ) );
            }
        }

        if ($action === 'wizard') {
            $this->render_wizard();
        } elseif ($action === 'results' && $scan_id) {
            $this->render_results($scan_id);
        } else {
            $this->render_main();
        }
    }

    private function render_main() {
        global $wpdb;
        $status_filter_override = null;

        // Handle bulk actions and status filter submit.
        if ( isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST') {
            $posted_bulk_nonce = isset($_POST['orphanix_bulk_nonce']) ? sanitize_text_field(wp_unslash($_POST['orphanix_bulk_nonce'])) : '';
            $filter_status = isset($_POST['filter_status']) ? sanitize_text_field(wp_unslash($_POST['filter_status'])) : '';
            $allowed_status_filters = array( 'published', 'running', 'completed', 'failed', 'trashed' );
            if ( ! in_array( $filter_status, $allowed_status_filters, true ) ) {
                $filter_status = '';
            }
            $status_filter_override = $filter_status;

            if ( current_user_can('manage_options') && ! empty($posted_bulk_nonce) && wp_verify_nonce($posted_bulk_nonce, 'orphanix_bulk_action') ) {
                $bulk_action = isset($_POST['orphanix_bulk_action']) ? sanitize_text_field(wp_unslash($_POST['orphanix_bulk_action'])) : '';
                $scan_ids = isset($_POST['scan_ids']) && is_array($_POST['scan_ids']) ? array_map('absint', wp_unslash($_POST['scan_ids'])) : [];

                if ('trash' === $bulk_action && ! empty($scan_ids)) {
                    foreach ($scan_ids as $sid) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->update("{$wpdb->prefix}orphanix_scans", ['status' => 'trashed'], ['id' => $sid]);
                    }
                } elseif ('delete' === $bulk_action && ! empty($scan_ids)) {
                    foreach ($scan_ids as $sid) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->delete("{$wpdb->prefix}orphanix_scan_items", ['scan_id' => $sid]);
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->delete("{$wpdb->prefix}orphanix_scans", ['id' => $sid]);
                        delete_transient('orphanix_scan_' . $sid . '_progress');
                    }
                } elseif ('restore' === $bulk_action && ! empty($scan_ids)) {
                    foreach ($scan_ids as $sid) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->update("{$wpdb->prefix}orphanix_scans", ['status' => 'completed'], ['id' => $sid]);
                    }
                }
            }

        }

        // Sorting and pagination
        $allowed_orderby = ['scan_mode' => 'scan_mode', 'started_at' => 'started_at', 'status' => 'status', 'total_files' => 'total_files'];
        $orderby_raw = isset($_GET['orderby']) ? sanitize_key(wp_unslash($_GET['orderby'])) : '';
        $orderby = isset($allowed_orderby[$orderby_raw]) ? $allowed_orderby[$orderby_raw] : 'started_at';
        $order_raw = isset($_GET['order']) ? strtoupper(sanitize_text_field(wp_unslash($_GET['order']))) : '';
        $order = ( $order_raw === 'ASC' ) ? 'ASC' : 'DESC';
        $current_orderby = $orderby;
        $current_order = $order;
        $per_page = isset($_GET['per_page']) ? max(1, absint(wp_unslash($_GET['per_page']))) : 10;
        $paged = isset($_GET['paged']) ? max(1, absint(wp_unslash($_GET['paged']))) : 1;
        $offset = ($paged - 1) * $per_page;

        // Filter by status
        $status_filter = ( null !== $status_filter_override ) ? $status_filter_override : ( isset($_GET['filter_status']) ? sanitize_text_field(wp_unslash($_GET['filter_status'])) : '' );
        $allowed_status_filters = [ 'published', 'running', 'completed', 'failed', 'trashed' ];
        if ( ! in_array( $status_filter, $allowed_status_filters, true ) ) {
            $status_filter = '';
        }

        $safe_page_arg = isset( $_GET['page'] ) ? sanitize_key( wp_unslash( $_GET['page'] ) ) : 'orphanix-broken';
        if ( 'orphanix-broken' !== $safe_page_arg ) {
            $safe_page_arg = 'orphanix-broken';
        }
        $sort_base_query_args = [
            'page' => 'orphanix-broken',
        ];
        if ( '' !== $status_filter ) {
            $sort_base_query_args['filter_status'] = $status_filter;
        }
        if ( $per_page > 0 ) {
            $sort_base_query_args['per_page'] = $per_page;
        }
        $sort_base_url = add_query_arg( $sort_base_query_args, admin_url( 'admin.php' ) );
        
        // Build query with proper prepared statements
        $order_by_sql = sprintf( '%s %s', $orderby, $order );

        if ( '' === $status_filter || 'published' === $status_filter ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $total_scans = intval( $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status != %s",
                'broken',
                'trashed'
            ) ) );
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $scans = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status != %s ORDER BY {$order_by_sql} LIMIT %d OFFSET %d",
                    'broken',
                    'trashed',
                    $per_page,
                    $offset
                )
            );
            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        } elseif ( $status_filter ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $total_scans = intval( $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status = %s",
                'broken',
                $status_filter
            ) ) );
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $scans = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status = %s ORDER BY {$order_by_sql} LIMIT %d OFFSET %d",
                    'broken',
                    $status_filter,
                    $per_page,
                    $offset
                )
            );
            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        }
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $published_count = intval( $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status != %s",
            'broken',
            'trashed'
        ) ) );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $trash_count = intval( $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status = %s",
            'broken',
            'trashed'
        ) ) );
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Broken Media Scan','orphanix-media-cleanup'); ?></h1>
            <div data-orphanix-style="orphanix-s-0017">
                <a href="<?php echo esc_url( wp_nonce_url( add_query_arg('action', 'wizard', admin_url('admin.php?page=orphanix-broken')), 'orphanix_broken_wizard' ) ); ?>" class="button button-primary button-large">
                    <?php esc_html_e('Start New Broken Media Scan','orphanix-media-cleanup'); ?>
                </a>
            </div>
            <h2><?php esc_html_e('Recent Scans','orphanix-media-cleanup'); ?></h2>
            <ul class="subsubsub">
                <li>
                    <a href="<?php echo esc_url( add_query_arg( array( 'page' => 'orphanix-broken', 'filter_status' => 'published' ), admin_url( 'admin.php' ) ) ); ?>" class="<?php echo ( '' === $status_filter || 'published' === $status_filter ) ? 'current' : ''; ?>">
                        <?php echo esc_html__( 'Publish', 'orphanix-media-cleanup' ); ?> (<?php echo esc_html( $published_count ); ?>)
                    </a> |
                </li>
                <li>
                    <a href="<?php echo esc_url( add_query_arg( array( 'page' => 'orphanix-broken', 'filter_status' => 'trashed' ), admin_url( 'admin.php' ) ) ); ?>" class="<?php echo ( 'trashed' === $status_filter ) ? 'current' : ''; ?>">
                        <?php echo esc_html__( 'Trash', 'orphanix-media-cleanup' ); ?> (<?php echo esc_html( $trash_count ); ?>)
                    </a>
                </li>
            </ul>
            <br class="clear" />

            <form method="post" action="<?php echo esc_url( admin_url( 'admin.php?page=orphanix-broken' ) ); ?>">
                <?php wp_nonce_field('orphanix_bulk_action', 'orphanix_bulk_nonce'); ?>
                <input type="hidden" name="page" value="<?php echo esc_attr( $safe_page_arg ); ?>">
                <div data-orphanix-style="orphanix-s-0018">
                    <select name="orphanix_bulk_action">
                        <option value="">-- <?php esc_html_e('Bulk Actions', 'orphanix-media-cleanup'); ?> --</option>
                        <?php if ( $status_filter !== 'trashed' ) : ?>
                            <option value="trash"><?php esc_html_e('Move to Trash', 'orphanix-media-cleanup'); ?></option>
                        <?php else : ?>
                            <option value="restore"><?php esc_html_e('Restore', 'orphanix-media-cleanup'); ?></option>
                            <option value="delete"><?php esc_html_e('Delete Permanently', 'orphanix-media-cleanup'); ?></option>
                        <?php endif; ?>
                    </select>
                    <button class="button" type="submit"><?php esc_html_e('Apply', 'orphanix-media-cleanup'); ?></button>
                    <span data-orphanix-style="orphanix-s-0019">
                        <label><?php esc_html_e('Status:', 'orphanix-media-cleanup'); ?>
                            <select name="filter_status">
                                <option value="">-- <?php esc_html_e('All', 'orphanix-media-cleanup'); ?> --</option>
                                <option value="published" <?php selected( $status_filter, 'published' ); ?>><?php esc_html_e('Publish', 'orphanix-media-cleanup'); ?></option>
                                <option value="running" <?php selected( $status_filter, 'running' ); ?>><?php esc_html_e('Running', 'orphanix-media-cleanup'); ?></option>
                                <option value="completed" <?php selected( $status_filter, 'completed' ); ?>><?php esc_html_e('Completed', 'orphanix-media-cleanup'); ?></option>
                                <option value="failed" <?php selected( $status_filter, 'failed' ); ?>><?php esc_html_e('Failed', 'orphanix-media-cleanup'); ?></option>
                                <option value="trashed" <?php selected( $status_filter, 'trashed' ); ?>><?php esc_html_e('Trashed', 'orphanix-media-cleanup'); ?></option>
                            </select>
                        </label>
                        <button class="button" type="submit" name="filter_action" value="filter"><?php esc_html_e('Filter', 'orphanix-media-cleanup'); ?></button>
                    </span>
                </div>

            <table class="widefat striped">
                <thead>
                    <tr>
                        <th data-orphanix-style="orphanix-s-0020"><input type="checkbox" class="orphanix-select-all"></th>
                        <th><a href="<?php echo esc_url( add_query_arg( ['orderby' => 'scan_mode', 'order' => ($current_orderby === 'scan_mode' && 'ASC' === $current_order ? 'DESC' : 'ASC'), 'paged' => 1], $sort_base_url ) ); ?>"><?php esc_html_e('Scan Type','orphanix-media-cleanup'); ?></a></th>
                        <th><a href="<?php echo esc_url( add_query_arg( ['orderby' => 'started_at', 'order' => ($current_orderby === 'started_at' && 'ASC' === $current_order ? 'DESC' : 'ASC'), 'paged' => 1], $sort_base_url ) ); ?>"><?php esc_html_e('Date','orphanix-media-cleanup'); ?></a></th>
                        <th><a href="<?php echo esc_url( add_query_arg( ['orderby' => 'status', 'order' => ($current_orderby === 'status' && 'ASC' === $current_order ? 'DESC' : 'ASC'), 'paged' => 1], $sort_base_url ) ); ?>"><?php esc_html_e('Status','orphanix-media-cleanup'); ?></a></th>
                        <th><a href="<?php echo esc_url( add_query_arg( ['orderby' => 'total_files', 'order' => ($current_orderby === 'total_files' && 'ASC' === $current_order ? 'DESC' : 'ASC'), 'paged' => 1], $sort_base_url ) ); ?>"><?php esc_html_e('Results','orphanix-media-cleanup'); ?></a></th>
                        <th><?php esc_html_e('Actions','orphanix-media-cleanup'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($scans): ?>
                        <?php foreach ($scans as $scan): ?>
                            <tr>
                                <td><input type="checkbox" name="scan_ids[]" value="<?php echo intval($scan->id); ?>"></td>
                                <td><?php echo esc_html( ucfirst($scan->scan_mode) ); ?></td>
                                <td><?php echo esc_html( date_i18n('Y-m-d H:i', strtotime($scan->started_at)) ); ?></td>
                                <td><span class="orphanix-status-pill <?php echo esc_attr( 'orphanix-status-pill--' . sanitize_html_class( $scan->status ) ); ?>"><?php echo esc_html(ucfirst($scan->status)); ?></span></td>
                                <td><?php echo intval($scan->total_files); ?></td>
                                <td>
                                    <?php if ( 'trashed' !== $scan->status ) : ?>
                                        <a href="<?php echo esc_url( add_query_arg(['action' => 'results', 'scan_id' => $scan->id], admin_url('admin.php?page=orphanix-broken')) ); ?>"><?php esc_html_e('View', 'orphanix-media-cleanup'); ?></a>
                                    <?php else : ?>
                                        -
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="6" data-orphanix-style="orphanix-s-0021"><?php esc_html_e('No scans found.','orphanix-media-cleanup'); ?></td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>

            <!-- Pagination -->
            <?php if ( $total_scans > 0 ) : ?>
                <div class="tablenav-pages" data-orphanix-style="orphanix-s-0022">
                    <span class="displaying-num"><?php echo esc_html( intval( $total_scans ) ); ?> <?php esc_html_e( 'items', 'orphanix-media-cleanup' ); ?></span>
                    <span class="pagination-links" data-orphanix-style="orphanix-s-0023">
                        <?php
                        $total_pages = ceil( $total_scans / $per_page );
                        $base_query_args = [
                            'page' => 'orphanix-broken',
                        ];
                        if ( '' !== $status_filter ) {
                            $base_query_args['filter_status'] = $status_filter;
                        }
                        if ( $per_page > 0 ) {
                            $base_query_args['per_page'] = $per_page;
                        }
                        if ( '' !== $current_orderby ) {
                            $base_query_args['orderby'] = $current_orderby;
                        }
                        if ( '' !== $current_order ) {
                            $base_query_args['order'] = $current_order;
                        }
                        $current_url_base = add_query_arg( $base_query_args, admin_url( 'admin.php' ) );
                        
                        // First page button
                        if ( $paged <= 1 ) {
                            echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&laquo;</span>';
                        } else {
                            echo '<a class="first-page button" href="' . esc_url( add_query_arg( 'paged', 1, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'First page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&laquo;</span></a>';
                        }
                        
                        // Previous page button
                        if ( $paged <= 1 ) {
                            echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&lsaquo;</span>';
                        } else {
                            echo '<a class="prev-page button" href="' . esc_url( add_query_arg( 'paged', $paged - 1, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'Previous page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&lsaquo;</span></a>';
                        }
                        ?>
                        <span class="screen-reader-text"><?php esc_html_e( 'Current Page', 'orphanix-media-cleanup' ); ?></span>
                        <span class="paging-input">
                            <span class="tablenav-paging-text"><?php echo esc_html( intval( $paged ) ); ?> <?php esc_html_e( 'of', 'orphanix-media-cleanup' ); ?> <span class="total-pages"><?php echo esc_html( intval( $total_pages ) ); ?></span></span>
                        </span>
                        <?php
                        // Next page button
                        if ( $paged >= $total_pages ) {
                            echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&rsaquo;</span>';
                        } else {
                            echo '<a class="next-page button" href="' . esc_url( add_query_arg( 'paged', $paged + 1, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'Next page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&rsaquo;</span></a>';
                        }
                        
                        // Last page button
                        if ( $paged >= $total_pages ) {
                            echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&raquo;</span>';
                        } else {
                            echo '<a class="last-page button" href="' . esc_url( add_query_arg( 'paged', $total_pages, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'Last page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&raquo;</span></a>';
                        }
                        ?>
                    </span>
                </div>
            <?php endif; ?>
            </form>
        </div>

        <?php
        ob_start();
        ?>
        (function($){
            $(function(){
                $('.orphanix-select-all').on('change', function(){
                    var checked = $(this).is(':checked');
                    $(this).closest('table').find('tbody input[type=checkbox]').prop('checked', checked);
                });
            });
        })(jQuery);
        <?php
        $inline_script = trim( (string) ob_get_clean() );
        wp_add_inline_script( 'orphanix-admin', $inline_script );
        ?>
        <?php
    }

    private function render_wizard() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Broken Media Scan Wizard','orphanix-media-cleanup'); ?></h1>
            
            <div id="orphanix-wizard" data-orphanix-style="orphanix-s-0024">
                <!-- Step 1: Choose Scan Type -->
                <div class="orphanix-step" data-step="1" data-orphanix-style="orphanix-s-0025">
                    <h2 data-orphanix-style="orphanix-s-0026"><?php esc_html_e('Step 1: Choose Scan Type','orphanix-media-cleanup'); ?></h2>
                    <p data-orphanix-style="orphanix-s-0027"><?php esc_html_e('Select the type of scan that best matches your needs:','orphanix-media-cleanup'); ?></p>
                    
                    <div data-orphanix-style="orphanix-s-0028">
                        <!-- Regular Scan -->
                        <div class="orphanix-scan-option orphanix-scan-regular" onclick="orphanix_select_broken_scan_type('regular')" data-orphanix-style="orphanix-s-0029">
                            <div data-orphanix-style="orphanix-s-0030">
                                <span data-orphanix-style="orphanix-s-0031">&#10003;</span>
                                <h3 data-orphanix-style="orphanix-s-0032"><?php esc_html_e('Regular Scan','orphanix-media-cleanup'); ?></h3>
                            </div>
                            <p data-orphanix-style="orphanix-s-0033"><?php esc_html_e('Scan your WordPress Media Library attachments to detect broken or missing media files.','orphanix-media-cleanup'); ?></p>
                            <ul data-orphanix-style="orphanix-s-0034">
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0036">&#10003;</span><?php esc_html_e('Checks registered media library attachments','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0036">&#10003;</span><?php esc_html_e('Verifies if the original media file exists on the server','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0036">&#10003;</span><?php esc_html_e('Detects missing, inaccessible, or corrupted files','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0036">&#10003;</span><?php esc_html_e('Identifies media with broken references in content','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0036">&#10003;</span><?php esc_html_e('Fast and safe scan with no file changes','orphanix-media-cleanup'); ?></li>
                            </ul>
                            <div data-orphanix-style="orphanix-s-0037">
                                <p data-orphanix-style="orphanix-s-0038"><strong><?php esc_html_e('Best for:','orphanix-media-cleanup'); ?></strong><br><?php esc_html_e('Quickly identifying broken media after migrations, restores, or file cleanup.','orphanix-media-cleanup'); ?></p>
                            </div>
                        </div>

                        <!-- Deep Scan -->
                        <div class="orphanix-scan-option orphanix-scan-deep" onclick="orphanix_select_broken_scan_type('deep')" data-orphanix-style="orphanix-s-0040">
                            <div data-orphanix-style="orphanix-s-0030">
                                <span data-orphanix-style="orphanix-s-0031"></span>
                                <h3 data-orphanix-style="orphanix-s-0032"><?php esc_html_e('Deep Scan','orphanix-media-cleanup'); ?></h3>
                            </div>
                            <p data-orphanix-style="orphanix-s-0033"><?php esc_html_e('Perform a comprehensive scan of your uploads directory and content references to detect broken or invalid media usage.','orphanix-media-cleanup'); ?></p>
                            <ul data-orphanix-style="orphanix-s-0034">
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0041">&#10003;</span><?php esc_html_e('Scans all files in the uploads directory','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0041">&#10003;</span><?php esc_html_e('Detects media references pointing to missing files','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0041">&#10003;</span><?php esc_html_e('Checks content, custom fields, and supported plugins','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0041">&#10003;</span><?php esc_html_e('Identifies broken URLs, missing files, and orphaned references','orphanix-media-cleanup'); ?></li>
                                <li data-orphanix-style="orphanix-s-0035"><span data-orphanix-style="orphanix-s-0041">&#10003;</span><?php esc_html_e('More thorough scan and may take longer on large sites','orphanix-media-cleanup'); ?></li>
                            </ul>
                            <div data-orphanix-style="orphanix-s-0042">
                                <p data-orphanix-style="orphanix-s-0043"><strong><?php esc_html_e('Best for:','orphanix-media-cleanup'); ?></strong><br><?php esc_html_e('Sites with migrations, CDN changes, manual file edits, or complex setups.','orphanix-media-cleanup'); ?></p>
                            </div>
                        </div>
                    </div>

                    <input type="hidden" id="orphanix-broken-scan-type" value="">
                    
                    <div data-orphanix-style="orphanix-s-0044">
                        <a href="<?php echo esc_url( admin_url('admin.php?page=orphanix-broken') ); ?>" class="button">
                            <?php esc_html_e('Cancel','orphanix-media-cleanup'); ?>
                        </a>
                        <button class="button button-primary" onclick="orphanix_next_broken_step(1)" disabled>
                            <?php esc_html_e('Next Step &rarr;','orphanix-media-cleanup'); ?>
                        </button>
                    </div>
                </div>

                <!-- Step 2: Settings -->
                <div class="orphanix-step" data-step="2" data-orphanix-style="orphanix-s-0039">
                    <h2><?php esc_html_e('Step 2: Scan Settings','orphanix-media-cleanup'); ?></h2>
                    <div id="orphanix-broken-settings-container"></div>
                    
                    <div data-orphanix-style="orphanix-s-0044">
                        <button class="button" onclick="orphanix_prev_broken_step(2)">
                            <?php esc_html_e('&larr; Back','orphanix-media-cleanup'); ?>
                        </button>
                        <button class="button button-primary" onclick="orphanix_next_broken_step(2)">
                            <?php esc_html_e('Next Step &rarr;','orphanix-media-cleanup'); ?>
                        </button>
                    </div>
                </div>

                <!-- Step 3: Confirm -->
                <div class="orphanix-step" data-step="3" data-orphanix-style="orphanix-s-0039">
                    <h2><?php esc_html_e('Step 3: Confirm Scan Settings','orphanix-media-cleanup'); ?></h2>
                    <div id="orphanix-broken-confirm-container"></div>
                    
                    <div data-orphanix-style="orphanix-s-0044">
                        <button class="button" onclick="orphanix_prev_broken_step(3)">
                            <?php esc_html_e('&larr; Back','orphanix-media-cleanup'); ?>
                        </button>
                        <button class="button button-primary" onclick="orphanix_start_broken_scan()">
                            <?php esc_html_e('Start Scan &rarr;','orphanix-media-cleanup'); ?>
                        </button>
                    </div>
                </div>

                <!-- Step 4: Scanning -->
                <div class="orphanix-step" data-step="4" data-orphanix-style="orphanix-s-0039">
                    <h2><?php esc_html_e('Step 4: Scanning','orphanix-media-cleanup'); ?></h2>
                    <h3><?php esc_html_e('Scanning File System','orphanix-media-cleanup'); ?></h3>
                    <p><?php esc_html_e('Discovering broken and missing files...','orphanix-media-cleanup'); ?></p>
                    
                    <div data-orphanix-style="orphanix-s-0017">
                        <div data-orphanix-style="orphanix-s-0045">
                            <div data-orphanix-style="orphanix-s-0046">
                                <strong><?php esc_html_e('Progress:','orphanix-media-cleanup'); ?></strong> <span id="orphanix-broken-progress-percent">0</span>%
                            </div>
                            <div data-orphanix-style="orphanix-s-0047">
                                <div id="orphanix-broken-progress-bar" data-orphanix-style="orphanix-s-0048"></div>
                            </div>
                        </div>

                        <div id="orphanix-broken-scan-stats" data-orphanix-style="orphanix-s-0049">
                            <div data-orphanix-style="orphanix-s-0050">
                                <strong><?php esc_html_e('Total Files to Scan:','orphanix-media-cleanup'); ?></strong> <span id="orphanix-broken-total-files">0</span>
                            </div>
                            <div data-orphanix-style="orphanix-s-0050">
                                <strong><?php esc_html_e('Files Scanned:','orphanix-media-cleanup'); ?></strong> <span id="orphanix-broken-scanned-files">0</span>
                            </div>
                            <div data-orphanix-style="orphanix-s-0050">
                                <strong><?php esc_html_e('Files Remaining:','orphanix-media-cleanup'); ?></strong> <span id="orphanix-broken-remaining-files">0</span>
                            </div>
                            <div data-orphanix-style="orphanix-s-0050">
                                <strong><?php esc_html_e('Estimated Time:','orphanix-media-cleanup'); ?></strong> <span id="orphanix-broken-eta">--</span>
                            </div>
                        </div>

                        <div id="orphanix-broken-current-file" data-orphanix-style="orphanix-s-0051">
                            <small><?php esc_html_e('Current:','orphanix-media-cleanup'); ?> <span id="orphanix-broken-current-file-name">--</span></small>
                        </div>
                    </div>

                    <button class="button button-secondary" onclick="orphanix_cancel_broken_scan()" id="orphanix-broken-cancel-btn">
                        <?php esc_html_e('Cancel Scan','orphanix-media-cleanup'); ?>
                    </button>
                </div>

                <!-- Step 5: Results -->
                <div class="orphanix-step" data-step="5" data-orphanix-style="orphanix-s-0039">
                    <h2><?php esc_html_e('Step 5: Results','orphanix-media-cleanup'); ?></h2>
                    <h3 data-orphanix-style="orphanix-s-0052">&check; <?php esc_html_e('Scan Completed Successfully!','orphanix-media-cleanup'); ?></h3>
                    
                    <div id="orphanix-broken-results-summary"></div>

                    <div data-orphanix-style="orphanix-s-0044">
                        <button class="button" onclick="orphanix_start_new_broken_scan()">
                            <?php esc_html_e('Start New Broken Media Scan','orphanix-media-cleanup'); ?>
                        </button>
                        <a href="#" class="button button-primary" id="orphanix-broken-view-results-btn">
                            <?php esc_html_e('View Results &rarr;','orphanix-media-cleanup'); ?>
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <?php
        ob_start();
        ?>
        var orphanix_broken_current_scan_id = null;
        var orphanix_broken_scan_settings = {};
        var orphanix_broken_scan_type = '';

        function orphanix_select_broken_scan_type(type) {
            orphanix_broken_scan_type = type;
            jQuery('#orphanix-broken-scan-type').val(type);

            // Update UI
            jQuery('.orphanix-scan-option').removeClass('selected');
            jQuery('.orphanix-scan-' + type).addClass('selected');

            // Enable next button
            jQuery('button[onclick="orphanix_next_broken_step(1)"]').prop('disabled', false);
        }

        function orphanix_next_broken_step(current) {
            if (current === 1) {
                orphanix_load_broken_settings();
                jQuery('.orphanix-step').hide();
                jQuery('[data-step="2"]').show();
            } else if (current === 2) {
                orphanix_load_broken_confirm();
                jQuery('.orphanix-step').hide();
                jQuery('[data-step="3"]').show();
            }
        }

        function orphanix_prev_broken_step(current) {
            jQuery('.orphanix-step').hide();
            jQuery('[data-step="' + (current - 1) + '"]').show();
        }

        function orphanix_load_broken_settings() {
            var settings = orphanix_broken_scan_type === 'regular' ? orphanix_get_broken_regular_settings() : orphanix_get_broken_deep_settings();
            jQuery('#orphanix-broken-settings-container').html(settings);
        }

        function orphanix_get_broken_regular_settings() {
            return `
                <div data-orphanix-style="orphanix-s-0053">
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_featured" value="1" checked data-orphanix-style="orphanix-s-0055">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Featured Images','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Check featured image attachment integrity.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_content" value="1" checked data-orphanix-style="orphanix-s-0055">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Post Content','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Scan post content for broken media references.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_custom_fields" value="1" checked data-orphanix-style="orphanix-s-0055">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Custom Fields','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Scan custom fields for broken media references.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_image_sizes" value="1" checked data-orphanix-style="orphanix-s-0055">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Generated Image Sizes','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Check all registered image size files.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0060">
                        <input type="checkbox" name="scan_page_builders" value="1" disabled data-orphanix-style="orphanix-s-0061">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0062">
                                <strong data-orphanix-style="orphanix-s-0063"><?php esc_html_e('Page Builder Content','orphanix-media-cleanup'); ?> </strong>
                                <small data-orphanix-style="orphanix-s-0064"><?php esc_html_e('Available in Deep Scan only','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="http_validation" value="1" checked data-orphanix-style="orphanix-s-0055">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('HTTP URL Validation','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Verify media URLs return valid HTTP status.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0060">
                        <input type="checkbox" name="scan_theme_files" value="1" disabled data-orphanix-style="orphanix-s-0061">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0062">
                                <strong data-orphanix-style="orphanix-s-0063"><?php esc_html_e('Scan Theme Files','orphanix-media-cleanup'); ?> </strong>
                                <small data-orphanix-style="orphanix-s-0064"><?php esc_html_e('Available in Deep Scan only','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                </div>
            `;
        }

        function orphanix_get_broken_deep_settings() {
            return `
                <div data-orphanix-style="orphanix-s-0053">
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_featured" value="1" checked data-orphanix-style="orphanix-s-0055">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Featured Images','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Check featured image attachment integrity.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_content" value="1" checked data-orphanix-style="orphanix-s-0055">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Post Content','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Scan post content for broken media references (includes Pods, WooCommerce, Sliders, Widgets, Menus).','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_custom_fields" value="1" checked data-orphanix-style="orphanix-s-0055">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Custom Fields','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Scan custom fields for broken media references (includes ACF Fields, Meta Box).','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_image_sizes" value="1" checked data-orphanix-style="orphanix-s-0055">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Generated Image Sizes','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Check all registered image size files.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="scan_page_builders" value="1" checked data-orphanix-style="orphanix-s-0055">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Page Builder Content','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Scan page builder content (includes Gutenberg Blocks, Elementor, Divi Builder, WPBakery).','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0054">
                        <input type="checkbox" name="http_validation" value="1" checked data-orphanix-style="orphanix-s-0055">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('HTTP URL Validation','orphanix-media-cleanup'); ?></strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Verify media URLs return valid HTTP status.','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                    <div data-orphanix-style="orphanix-s-0065">
                        <input type="checkbox" name="scan_theme_files" value="1" data-orphanix-style="orphanix-s-0055">
                        <div data-orphanix-style="orphanix-s-0056">
                            <label data-orphanix-style="orphanix-s-0057">
                                <strong data-orphanix-style="orphanix-s-0058"><?php esc_html_e('Scan Theme Files','orphanix-media-cleanup'); ?> !</strong>
                                <small data-orphanix-style="orphanix-s-0059"><?php esc_html_e('Advanced: scan active theme files for media references (read-only, may be slow).','orphanix-media-cleanup'); ?></small>
                            </label>
                        </div>
                    </div>
                </div>
            `;
        }

        function orphanix_collect_broken_settings() {
            var settings = {};
            jQuery('#orphanix-broken-settings-container').find('input[type="checkbox"]').each(function() {
                settings[jQuery(this).attr('name')] = jQuery(this).is(':checked') ? 1 : 0;
            });
            return settings;
        }

        function orphanix_load_broken_confirm() {
            var settings = orphanix_collect_broken_settings();
            orphanix_broken_scan_settings = settings;

            var mode = orphanix_broken_scan_type;
            var featured = settings.scan_featured ? 1 : 0;
            var content = settings.scan_content ? 1 : 0;
            var custom = settings.scan_custom_fields ? 1 : 0;
            var sizes = settings.scan_image_sizes ? 1 : 0;
            var pageBuilders = settings.scan_page_builders ? 1 : 0;
            var http = settings.http_validation ? 1 : 0;
            var theme = settings.scan_theme_files ? 1 : 0;
            
            // Page Builder & Plugin Compatibility (Deep only)
            // These are controlled by the grouped checkboxes:
            // - Page Builder Content: Gutenberg, Elementor, Divi, WPBakery
            // - Scan Custom Fields: ACF Fields, Meta Box
            // - Scan Post Content: Pods, WooCommerce, Sliders, Widgets, Menus

            var scopeItems = [];
            scopeItems.push((featured ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Featured Images', 'orphanix-media-cleanup' ) ); ?>');
            scopeItems.push((content ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Post Content', 'orphanix-media-cleanup' ) ); ?>');
            scopeItems.push((custom ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Custom Fields', 'orphanix-media-cleanup' ) ); ?>');
            scopeItems.push((sizes ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Generated Image Sizes', 'orphanix-media-cleanup' ) ); ?>');
            if (mode === 'deep') {
                scopeItems.push((pageBuilders ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Page Builder Content', 'orphanix-media-cleanup' ) ); ?>');
            }
            scopeItems.push((http ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'HTTP URL Validation', 'orphanix-media-cleanup' ) ); ?>');
            if (mode === 'deep') {
                scopeItems.push((theme ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Theme Files', 'orphanix-media-cleanup' ) ); ?>');
            }

            var scopeList = scopeItems.map(item => '<div data-orphanix-style="orphanix-s-0068">' + item + '</div>').join('');

            // Page Builder & Plugin Compatibility List
            var compatItems = [];
            if (mode === 'deep') {
                // Page Builder Content group
                compatItems.push((pageBuilders ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Gutenberg Blocks', 'orphanix-media-cleanup' ) ); ?>');
                compatItems.push((pageBuilders ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Elementor', 'orphanix-media-cleanup' ) ); ?>');
                compatItems.push((pageBuilders ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Divi Builder', 'orphanix-media-cleanup' ) ); ?>');
                compatItems.push((pageBuilders ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'WPBakery', 'orphanix-media-cleanup' ) ); ?>');
                
                // Scan Custom Fields group
                compatItems.push((custom ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'ACF Fields', 'orphanix-media-cleanup' ) ); ?>');
                compatItems.push((custom ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Meta Box', 'orphanix-media-cleanup' ) ); ?>');
                
                // Scan Post Content group
                compatItems.push((content ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Pods', 'orphanix-media-cleanup' ) ); ?>');
                compatItems.push((content ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'WooCommerce', 'orphanix-media-cleanup' ) ); ?>');
                compatItems.push((content ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Sliders', 'orphanix-media-cleanup' ) ); ?>');
                compatItems.push((content ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Widgets', 'orphanix-media-cleanup' ) ); ?>');
                compatItems.push((content ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Navigation Menus', 'orphanix-media-cleanup' ) ); ?>');
            }
            var compatList = compatItems.map((item, index) => {
                return '<div data-orphanix-style="orphanix-s-0069">' + item + '</div>';
            }).join('');

            var checkItems = [
                (featured ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Attachment exists', 'orphanix-media-cleanup' ) ); ?>',
                (featured ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Original file exists', 'orphanix-media-cleanup' ) ); ?>',
                (sizes ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Generated sizes exist', 'orphanix-media-cleanup' ) ); ?>',
                (content ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Content references checked', 'orphanix-media-cleanup' ) ); ?>',
                (custom ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Custom fields checked', 'orphanix-media-cleanup' ) ); ?>',
                (http ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'HTTP status validated', 'orphanix-media-cleanup' ) ); ?>',
                (mode === 'deep' ? '<span data-orphanix-style="orphanix-s-0066">&#10003;</span>' : '<span data-orphanix-style="orphanix-s-0067">&#10007;</span>') + ' <?php echo esc_js( __( 'Filesystem integrity checks', 'orphanix-media-cleanup' ) ); ?>'
            ];

            var scanTypesItems = checkItems.slice();
            if (mode === 'deep' && compatItems.length) {
                scanTypesItems = scanTypesItems.concat(compatItems);
            }

            var scanTypesList = scanTypesItems.map((item, index) => {
                return '<div data-orphanix-style="orphanix-s-0069">' + item + '</div>';
            }).join('');

            var html = `
                <div data-orphanix-style="orphanix-s-0070">
                    <!-- Scan Type Card -->
                    <div data-orphanix-style="orphanix-s-0071">
                        <h3 data-orphanix-style="orphanix-s-0072"><?php echo esc_js( __( 'Scan Configuration', 'orphanix-media-cleanup' ) ); ?></h3>
                        <p data-orphanix-style="orphanix-s-0073">${mode === 'regular' ? '<?php echo esc_js( __( 'Regular Broken Media Scan', 'orphanix-media-cleanup' ) ); ?>' : '<?php echo esc_js( __( 'Deep Broken Media Scan', 'orphanix-media-cleanup' ) ); ?>'}</p>
                    </div>

                    <!-- Scan Scope Section -->
                    <div data-orphanix-style="orphanix-s-0074">
                        <h4 data-orphanix-style="orphanix-s-0075"><?php echo esc_js( __( 'Scan Scope:', 'orphanix-media-cleanup' ) ); ?></h4>
                        <div data-orphanix-style="orphanix-s-0076">
                            ${scopeList}
                        </div>
                    </div>

                    <!-- Scan Types Section -->
                    <div data-orphanix-style="orphanix-s-0077">
                        <h4 data-orphanix-style="orphanix-s-0075"><?php echo esc_js( __( 'Scan Types:', 'orphanix-media-cleanup' ) ); ?></h4>
                        <div data-orphanix-style="orphanix-s-0076">
                            ${scanTypesList}
                        </div>
                    </div>

                    <!-- Safety Note -->
                    <div data-orphanix-style="orphanix-s-0078">
                        <strong><?php esc_html_e('Safety Note:','orphanix-media-cleanup'); ?></strong> <?php esc_html_e('This scan only analyzes files and does not delete anything. You will review all results before any cleanup actions.','orphanix-media-cleanup'); ?>
                    </div>
                </div>
            `;

            jQuery('#orphanix-broken-confirm-container').html(html);
        }

        function orphanix_start_broken_scan() {
            jQuery('.orphanix-step').hide();
            jQuery('[data-step="4"]').show();

            jQuery.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'orphanix_initiate_scan',
                    nonce: '<?php echo esc_js( wp_create_nonce('orphanix_scan_nonce') ); ?>',
                    scan_type: orphanix_broken_scan_type,
                    mode: 'broken',
                    settings: JSON.stringify(orphanix_broken_scan_settings)
                },
                success: function(response) {
                    if (response.success) {
                        orphanix_broken_current_scan_id = response.data.scan_id;
                        orphanix_poll_broken_scan_progress();
                    }
                },
                error: function() {
                    alert('Error starting scan');
                }
            });
        }

        function orphanix_poll_broken_scan_progress() {
            jQuery.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'orphanix_get_scan_progress',
                    nonce: '<?php echo esc_js( wp_create_nonce('orphanix_scan_nonce') ); ?>',
                    scan_id: orphanix_broken_current_scan_id
                },
                success: function(response) {
                    if (response.success) {
                        var data = response.data;
                        jQuery('#orphanix-broken-progress-percent').text(data.percentage);
                        jQuery('#orphanix-broken-progress-bar').css('width', data.percentage + '%');
                        jQuery('#orphanix-broken-total-files').text(data.total);
                        jQuery('#orphanix-broken-scanned-files').text(data.processed);
                        jQuery('#orphanix-broken-remaining-files').text(data.total - data.processed);
                        jQuery('#orphanix-broken-current-file-name').text(data.current_file || 'Processing...');

                        // Compute ETA if possible
                        if (data.started_at && data.processed && data.total && data.processed > 0 && data.processed < data.total) {
                            var now = Math.floor(Date.now() / 1000);
                            var elapsed = Math.max(1, now - parseInt(data.started_at, 10));
                            var rate = data.processed / elapsed; // items per second
                            if (rate > 0) {
                                var remaining = data.total - data.processed;
                                var estSeconds = Math.round(remaining / rate);
                                jQuery('#orphanix-broken-eta').text(orphanix_format_seconds(estSeconds));
                            } else {
                                jQuery('#orphanix-broken-eta').text('--');
                            }
                        } else if (data.status === 'completed') {
                            jQuery('#orphanix-broken-eta').text('00:00:00');
                        } else {
                            jQuery('#orphanix-broken-eta').text('--');
                        }

                        if (data.status === 'completed') {
                            orphanix_show_broken_results(data);
                        } else if (data.status === 'running') {
                            setTimeout(orphanix_poll_broken_scan_progress, 1000);
                        }
                    }
                }
            });
        }

        function orphanix_show_broken_results(data) {
            jQuery('.orphanix-step').hide();
            jQuery('[data-step="5"]').show();

            var html = `
                <div data-orphanix-style="orphanix-s-0079">
                    <div data-orphanix-style="orphanix-s-0080">
                        <h4 data-orphanix-style="orphanix-s-0006">` + '<?php echo esc_js( __( 'Total Files', 'orphanix-media-cleanup' ) ); ?>' + `</h4>
                        <p data-orphanix-style="orphanix-s-0081">${data.total}</p>
                    </div>
                    <div data-orphanix-style="orphanix-s-0080">
                        <h4 data-orphanix-style="orphanix-s-0006">` + '<?php echo esc_js( __( 'In Media Library', 'orphanix-media-cleanup' ) ); ?>' + `</h4>
                        <p data-orphanix-style="orphanix-s-0081">${data.in_library || 0}</p>
                    </div>
                    <div data-orphanix-style="orphanix-s-0080">
                        <h4 data-orphanix-style="orphanix-s-0006">` + '<?php echo esc_js( __( 'Used (Not in Media)', 'orphanix-media-cleanup' ) ); ?>' + `</h4>
                        <p data-orphanix-style="orphanix-s-0081">${data.used_not_in_library || 0}</p>
                    </div>
                    <div data-orphanix-style="orphanix-s-0080">
                        <h4 data-orphanix-style="orphanix-s-0006">` + '<?php echo esc_js( __( 'Broken/Missing', 'orphanix-media-cleanup' ) ); ?>' + `</h4>
                        <p data-orphanix-style="orphanix-s-0081">${data.orphan || 0}</p>
                    </div>
                </div>
            `;

            jQuery('#orphanix-broken-results-summary').html(html);
            jQuery('#orphanix-broken-view-results-btn').attr('href', '<?php echo esc_url( add_query_arg(['action' => 'results', 'scan_id' => ''], admin_url('admin.php?page=orphanix-broken')) ); ?>' + orphanix_broken_current_scan_id);
        }

        function orphanix_cancel_broken_scan() {
            jQuery.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'orphanix_cancel_scan',
                    nonce: '<?php echo esc_js( wp_create_nonce('orphanix_scan_nonce') ); ?>',
                    scan_id: orphanix_broken_current_scan_id
                },
                success: function() {
                    window.location.href = '<?php echo esc_url( admin_url('admin.php?page=orphanix-broken') ); ?>';
                }
            });
        }

        function orphanix_format_seconds(sec) {
            sec = parseInt(sec, 10);
            if (isNaN(sec) || sec < 0) return '--';
            var h = Math.floor(sec / 3600);
            var m = Math.floor((sec % 3600) / 60);
            var s = sec % 60;
            return ('0' + h).slice(-2) + ':' + ('0' + m).slice(-2) + ':' + ('0' + s).slice(-2);
        }

        function orphanix_start_new_broken_scan() {
            window.location.href = '<?php echo esc_url( wp_nonce_url( add_query_arg('action', 'wizard', admin_url('admin.php?page=orphanix-broken')), 'orphanix_broken_wizard' ) ); ?>';
        }

        // Delete scan
        function orphanix_delete_broken_scan(scan_id) {
            jQuery.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'orphanix_delete_scan',
                    nonce: '<?php echo esc_js( wp_create_nonce('orphanix_scan_nonce') ); ?>',
                    scan_id: scan_id
                },
                success: function() {
                    location.reload();
                }
            });
        }
        <?php
        $inline_script = trim( (string) ob_get_clean() );
        wp_add_inline_script( 'orphanix-admin', $inline_script );
        ?>
        <?php
    }

    private function render_results($scan_id) {
        global $wpdb;
        $results_orderby_override = null;
        $results_order_override = null;
        $status_filter_override = null;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $scan = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}orphanix_scans WHERE id = %d", $scan_id));
        if (!$scan) {
            echo '<div class="wrap"><p>' . esc_html__( 'Scan not found', 'orphanix-media-cleanup' ) . '</p></div>';
            return;
        }
        nocache_headers();

        if ( isset( $_SERVER['REQUEST_METHOD'] ) && 'POST' === $_SERVER['REQUEST_METHOD'] ) {
            $posted_bulk_nonce = isset( $_POST['orphanix_bulk_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['orphanix_bulk_nonce'] ) ) : '';
            $allowed_order_tokens = array( 'id', 'attachment_id', 'file_path', 'directory_type', 'file_size', 'status' );
            $posted_orderby = isset( $_POST['orderby'] ) ? sanitize_key( wp_unslash( $_POST['orderby'] ) ) : '';
            $posted_order = isset( $_POST['order'] ) ? strtoupper( sanitize_text_field( wp_unslash( $_POST['order'] ) ) ) : 'DESC';
            $posted_orderby = in_array( $posted_orderby, $allowed_order_tokens, true ) ? $posted_orderby : 'id';
            $posted_order = ( 'ASC' === $posted_order ) ? 'ASC' : 'DESC';
            $results_orderby_override = $posted_orderby;
            $results_order_override = $posted_order;

            $filter_status = isset( $_POST['filter_status'] ) ? sanitize_text_field( wp_unslash( $_POST['filter_status'] ) ) : '';
            $allowed_result_filters = array( 'published', 'alt_tag_missing', 'in_media_library', 'not_in_media_library', 'used', 'not_used', 'orphan', 'trashed' );
            if ( ! in_array( $filter_status, $allowed_result_filters, true ) ) {
                $filter_status = '';
            }
            $status_filter_override = $filter_status;
            $did_process_bulk_action = false;

            if ( current_user_can( 'manage_options' ) && ! empty( $posted_bulk_nonce ) && wp_verify_nonce( $posted_bulk_nonce, 'orphanix_bulk_action' ) ) {
                $bulk_action = isset( $_POST['orphanix_bulk_action'] ) ? sanitize_text_field( wp_unslash( $_POST['orphanix_bulk_action'] ) ) : '';
                $item_ids = isset( $_POST['item_ids'] ) && is_array( $_POST['item_ids'] ) ? array_map( 'absint', wp_unslash( $_POST['item_ids'] ) ) : array();

                if ( 'trash' === $bulk_action && ! empty( $item_ids ) ) {
                    foreach ( $item_ids as $iid ) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->update( "{$wpdb->prefix}orphanix_scan_items", array( 'status' => 'trashed' ), array( 'id' => $iid ) );
                    }
                    $did_process_bulk_action = true;
                } elseif ( 'delete' === $bulk_action && ! empty( $item_ids ) ) {
                    $repo = new ORPHANIX_Scan_Items_Repository();
                    foreach ( $item_ids as $iid ) {
                        $repo->delete_item_with_attachment( $iid );
                    }
                    $did_process_bulk_action = true;
                } elseif ( 'restore' === $bulk_action && ! empty( $item_ids ) ) {
                    foreach ( $item_ids as $iid ) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->update( "{$wpdb->prefix}orphanix_scan_items", array( 'status' => '' ), array( 'id' => $iid ) );
                    }
                    $did_process_bulk_action = true;
                }
            }

            $did_request_filter = isset( $_POST['filter_action'] ) && 'filter' === sanitize_key( wp_unslash( $_POST['filter_action'] ) );
            if ( $did_process_bulk_action || $did_request_filter ) {
                $redirect_args = array(
                    'page'    => 'orphanix-broken',
                    'action'  => 'results',
                    'scan_id' => $scan_id,
                    'orderby' => $posted_orderby,
                    'order'   => $posted_order,
                );
                if ( '' !== $filter_status ) {
                    $redirect_args['filter_status'] = $filter_status;
                }
                $redirect_args['orefresh'] = time();
                $redirect_url = add_query_arg( $redirect_args, admin_url( 'admin.php' ) );
                if ( ! headers_sent() ) {
                    wp_safe_redirect( $redirect_url );
                    exit;
                }
            }

        }

        // Pagination and filtering
        $per_page = 10;
        $paged = isset( $_GET['paged'] ) ? max( 1, absint( wp_unslash( $_GET['paged'] ) ) ) : 1;
        $offset = ( $paged - 1 ) * $per_page;
        $allowed_results_orderby = array(
            'id'             => 'id',
            'attachment_id'  => 'attachment_id',
            'file_path'      => 'file_path',
            'directory_type' => 'directory_type',
            'file_size'      => 'file_size',
            'status'         => 'status',
        );
        $results_orderby_raw = isset( $_GET['orderby'] ) ? sanitize_key( wp_unslash( $_GET['orderby'] ) ) : '';
        $results_order_raw   = isset( $_GET['order'] ) ? strtoupper( sanitize_text_field( wp_unslash( $_GET['order'] ) ) ) : 'DESC';
        $results_orderby     = ( null !== $results_orderby_override ) ? $results_orderby_override : ( isset( $allowed_results_orderby[ $results_orderby_raw ] ) ? $allowed_results_orderby[ $results_orderby_raw ] : 'id' );
        $results_order       = ( null !== $results_order_override ) ? $results_order_override : ( ( 'ASC' === $results_order_raw ) ? 'ASC' : 'DESC' );
        // Filter by status
        $status_filter = ( null !== $status_filter_override ) ? $status_filter_override : ( isset( $_GET['filter_status'] ) ? sanitize_text_field( wp_unslash( $_GET['filter_status'] ) ) : '' );
        $allowed_result_filters = [ 'published', 'alt_tag_missing', 'in_media_library', 'not_in_media_library', 'used', 'not_used', 'orphan', 'trashed' ];
        if ( ! in_array( $status_filter, $allowed_result_filters, true ) ) {
            $status_filter = '';
        }
        if ( '' === $status_filter ) {
            $status_filter = 'published';
        }
        $where_parts = array( 'scan_id = %d' );
        $where_args  = array( $scan_id );

        if ( 'published' === $status_filter ) {
            $where_parts[] = 'status != %s';
            $where_args[]  = 'trashed';
        } elseif ( 'alt_tag_missing' === $status_filter ) {
            $where_parts[] = "(alt_text IS NULL OR alt_text = '')";
        } elseif ( 'in_media_library' === $status_filter ) {
            $where_parts[] = 'directory_type = %s';
            $where_args[]  = 'media';
        } elseif ( 'not_in_media_library' === $status_filter ) {
            $where_parts[] = 'directory_type != %s';
            $where_args[]  = 'media';
        } elseif ( 'used' === $status_filter ) {
            $where_parts[] = 'status = %s';
            $where_args[]  = 'used';
        } elseif ( 'not_used' === $status_filter ) {
            $where_parts[] = 'status = %s';
            $where_args[]  = 'not_used';
        } elseif ( 'orphan' === $status_filter ) {
            $where_parts[] = 'status = %s';
            $where_args[]  = 'orphan';
        } elseif ( 'trashed' === $status_filter ) {
            $where_parts[] = 'status = %s';
            $where_args[]  = 'trashed';
        }

        $where_sql    = 'WHERE ' . implode( ' AND ', $where_parts );

        // Strict allowlist for ORDER BY
        $allowed_order_columns = array(
            'id', 'attachment_id', 'file_path', 'directory_type', 'file_size', 'status'
        );
        $order_by_col = in_array( $results_orderby, $allowed_order_columns, true ) ? $results_orderby : 'id';
        $order_dir = ( 'ASC' === $results_order ) ? 'ASC' : 'DESC';
        $order_by_sql = $order_by_col . ' ' . $order_dir;

        // Build SQL with only placeholders in WHERE, and strict ORDER BY
        $where_joined = implode( ' AND ', $where_parts );
        // phpcs:disable WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber, PluginCheck.Security.DirectDB.UnescapedDBParameter
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $total_items = intval( $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE {$where_joined}", $where_args ) ) );
        $total_pages = ceil( $total_items / $per_page );
        $items_args  = array_merge( $where_args, array( $per_page, $offset ) );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $items = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}orphanix_scan_items WHERE {$where_joined} ORDER BY {$order_by_sql} LIMIT %d OFFSET %d", $items_args ) );
        // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber, PluginCheck.Security.DirectDB.UnescapedDBParameter

        // Publish/Trash counters for this result table.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $published_count = intval( $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d AND status != %s",
            $scan_id,
            'trashed'
        ) ) );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $trash_count = intval( $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d AND status = %s",
            $scan_id,
            'trashed'
        ) ) );
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__( 'Broken Media Scan Results', 'orphanix-media-cleanup' ); ?></h1>
            <p><a href="<?php echo esc_url( admin_url( 'admin.php?page=orphanix-broken' ) ); ?>" class="button"><?php echo esc_html__( 'Back', 'orphanix-media-cleanup' ); ?></a></p>
            <?php
            $results_base_args = array(
                'page'   => 'orphanix-broken',
                'action' => 'results',
                'scan_id'=> $scan_id,
                'orderby'=> $results_orderby,
                'order'  => $results_order,
            );
            if ( '' !== $status_filter ) {
                $results_base_args['filter_status'] = $status_filter;
            }
            $results_base_url = add_query_arg( $results_base_args, admin_url( 'admin.php' ) );
            ?>
            <ul class="subsubsub">
                <li>
                    <a href="<?php echo esc_url( add_query_arg( array( 'filter_status' => 'published' ), $results_base_url ) ); ?>" class="<?php echo ( '' === $status_filter || 'published' === $status_filter ) ? 'current' : ''; ?>">
                        <?php echo esc_html__( 'Publish', 'orphanix-media-cleanup' ); ?> (<?php echo esc_html( $published_count ); ?>)
                    </a> |
                </li>
                <li>
                    <a href="<?php echo esc_url( add_query_arg( array( 'filter_status' => 'trashed' ), $results_base_url ) ); ?>" class="<?php echo ( 'trashed' === $status_filter ) ? 'current' : ''; ?>">
                        <?php echo esc_html__( 'Trash', 'orphanix-media-cleanup' ); ?> (<?php echo esc_html( $trash_count ); ?>)
                    </a>
                </li>
            </ul>
            <br class="clear" />

            <form method="post" action="<?php echo esc_url( $results_base_url ); ?>">
                <?php wp_nonce_field( 'orphanix_bulk_action', 'orphanix_bulk_nonce' ); ?>
                <input type="hidden" name="orderby" value="<?php echo esc_attr( $results_orderby ); ?>">
                <input type="hidden" name="order" value="<?php echo esc_attr( $results_order ); ?>">
                <div data-orphanix-style="orphanix-s-0018">
                    <select name="orphanix_bulk_action">
                        <option value="">-- <?php esc_html_e('Bulk Actions', 'orphanix-media-cleanup'); ?> --</option>
                        <?php if ( $status_filter !== 'trashed' ) : ?>
                            <option value="trash"><?php esc_html_e('Move to Trash', 'orphanix-media-cleanup'); ?></option>
                        <?php else : ?>
                            <option value="restore"><?php esc_html_e('Restore', 'orphanix-media-cleanup'); ?></option>
                            <option value="delete"><?php esc_html_e('Delete Permanently', 'orphanix-media-cleanup'); ?></option>
                        <?php endif; ?>
                    </select>
                    <button class="button" type="submit"><?php esc_html_e('Apply', 'orphanix-media-cleanup'); ?></button>
                    <span data-orphanix-style="orphanix-s-0019">
                        <label><?php esc_html_e('Filter:', 'orphanix-media-cleanup'); ?>
                            <select name="filter_status">
                                <option value="">-- <?php esc_html_e('All', 'orphanix-media-cleanup'); ?> --</option>
                                <option value="published" <?php selected( $status_filter, 'published' ); ?>><?php esc_html_e('Publish', 'orphanix-media-cleanup'); ?></option>
                                <option value="alt_tag_missing" <?php selected( $status_filter, 'alt_tag_missing' ); ?>><?php esc_html_e('Alt Tag Missing', 'orphanix-media-cleanup'); ?></option>
                                <option value="in_media_library" <?php selected( $status_filter, 'in_media_library' ); ?>><?php esc_html_e('In Media Library', 'orphanix-media-cleanup'); ?></option>
                                <option value="not_in_media_library" <?php selected( $status_filter, 'not_in_media_library' ); ?>><?php esc_html_e('Not in Media Library', 'orphanix-media-cleanup'); ?></option>
                                <option value="used" <?php selected( $status_filter, 'used' ); ?>><?php esc_html_e('Used', 'orphanix-media-cleanup'); ?></option>
                                <option value="not_used" <?php selected( $status_filter, 'not_used' ); ?>><?php esc_html_e('Not Used', 'orphanix-media-cleanup'); ?></option>
                                <option value="orphan" <?php selected( $status_filter, 'orphan' ); ?>><?php esc_html_e('Orphan', 'orphanix-media-cleanup'); ?></option>
                                <option value="trashed" <?php selected( $status_filter, 'trashed' ); ?>><?php esc_html_e('Trashed', 'orphanix-media-cleanup'); ?></option>
                            </select>
                        </label>
                        <button class="button" type="submit" name="filter_action" value="filter"><?php esc_html_e('Filter', 'orphanix-media-cleanup'); ?></button>
                    </span>
                </div>

            <table class="widefat striped" data-orphanix-style="orphanix-s-0082">
                    <thead>
                        <tr>
                            <th data-orphanix-style="orphanix-s-0020"><input type="checkbox" class="orphanix-select-all"></th>
                            <th data-orphanix-style="orphanix-s-0083"><?php esc_html_e('Thumb', 'orphanix-media-cleanup'); ?></th>
                            <th><a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'id', 'order' => ( 'id' === $results_orderby && 'ASC' === $results_order ) ? 'DESC' : 'ASC', 'paged' => 1 ), $results_base_url ) ); ?>"><?php esc_html_e('ID', 'orphanix-media-cleanup'); ?></a></th>
                            <th><?php esc_html_e('Title', 'orphanix-media-cleanup'); ?></th>
                            <th><a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'file_path', 'order' => ( 'file_path' === $results_orderby && 'ASC' === $results_order ) ? 'DESC' : 'ASC', 'paged' => 1 ), $results_base_url ) ); ?>"><?php esc_html_e('Path', 'orphanix-media-cleanup'); ?></a></th>
                            <th><a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'directory_type', 'order' => ( 'directory_type' === $results_orderby && 'ASC' === $results_order ) ? 'DESC' : 'ASC', 'paged' => 1 ), $results_base_url ) ); ?>"><?php esc_html_e('Directory', 'orphanix-media-cleanup'); ?></a></th>
                            <th><?php esc_html_e('Alt Tag', 'orphanix-media-cleanup'); ?></th>
                            <th><a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'file_size', 'order' => ( 'file_size' === $results_orderby && 'ASC' === $results_order ) ? 'DESC' : 'ASC', 'paged' => 1 ), $results_base_url ) ); ?>"><?php esc_html_e('Size', 'orphanix-media-cleanup'); ?></a></th>
                            <th><?php esc_html_e('Belongs to', 'orphanix-media-cleanup'); ?></th>
                            <th><a href="<?php echo esc_url( add_query_arg( array( 'orderby' => 'status', 'order' => ( 'status' === $results_orderby && 'ASC' === $results_order ) ? 'DESC' : 'ASC', 'paged' => 1 ), $results_base_url ) ); ?>"><?php esc_html_e('Status', 'orphanix-media-cleanup'); ?></a></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ( $items ) : 
                            foreach ( $items as $item ) : 
                                $thumbnail = '';
                                if ( ! empty( $item->file_url ) && preg_match( '/\.(jpg|jpeg|png|gif|webp)$/i', $item->file_url ) ) {
                                    $thumbnail = '<img src="' . esc_url( $item->file_url ) . '" data-orphanix-style="orphanix-s-0084" />';
                                } else {
                                    $thumbnail = '&mdash;';
                                }
                        ?>
                            <tr>
                                <td><input type="checkbox" name="item_ids[]" value="<?php echo esc_attr( $item->id ); ?>" /></td>
                                <td><?php echo wp_kses_post( $thumbnail ); ?></td>
                                <td><?php echo esc_html( (string) $item->id ); ?></td>
                                <td><?php echo esc_html( basename( $item->file_path ) ); ?></td>
                                <td><a href="<?php echo esc_url( $item->file_url ); ?>" target="_blank" title="<?php echo esc_attr( $item->file_url ); ?>" data-orphanix-style="orphanix-s-0085"><?php echo esc_html( wp_parse_url( $item->file_url, PHP_URL_PATH ) ); ?></a></td>
                                <td><?php echo esc_html( 'media' === $item->directory_type ? esc_html__( 'In Media Library', 'orphanix-media-cleanup' ) : esc_html__( 'Not in Media Library', 'orphanix-media-cleanup' ) ); ?></td>
                                <td><?php echo ! empty( $item->alt_text ) ? esc_html( $item->alt_text ) : wp_kses_post( '<span data-orphanix-style="orphanix-s-0086">' . esc_html__( 'Missing', 'orphanix-media-cleanup' ) . '</span>' ); ?></td>
                                <td><?php echo esc_html( size_format( $item->file_size ) ); ?></td>
                                <td><?php echo wp_kses_post( $this->render_usage_context( $item->usage_context ) ); ?></td>
                                <td>
                                    <span class="orphanix-status-pill <?php echo esc_attr( 'orphanix-status-pill--' . sanitize_html_class( $item->status ) ); ?>">
                                        <?php echo esc_html( ucfirst( str_replace( '_', ' ', $item->status ) ) ); ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; 
                        else : ?>
                            <tr><td colspan="10"><?php echo esc_html__( 'No results found.', 'orphanix-media-cleanup' ); ?></td></tr>
                        <?php endif; ?>
                    </tbody>
                </table>

                <!-- Pagination -->
                <?php if ( $total_items > 0 ) : ?>
                    <div class="tablenav-pages" data-orphanix-style="orphanix-s-0022">
                        <span class="displaying-num"><?php echo esc_html( intval( $total_items ) ); ?> <?php esc_html_e( 'items', 'orphanix-media-cleanup' ); ?></span>
                        <span class="pagination-links" data-orphanix-style="orphanix-s-0023">
                            <?php
                            $current_url_base = $results_base_url;
                            $current_url_base = add_query_arg(
                                array(
                                    'orderby' => $results_orderby,
                                    'order'   => $results_order,
                                ),
                                $current_url_base
                            );
                            
                            // First page button
                            if ( $paged <= 1 ) {
                                echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&laquo;</span>';
                            } else {
                                echo '<a class="first-page button" href="' . esc_url( add_query_arg( 'paged', 1, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'First page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&laquo;</span></a>';
                            }
                            
                            // Previous page button
                            if ( $paged <= 1 ) {
                                echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&lsaquo;</span>';
                            } else {
                                echo '<a class="prev-page button" href="' . esc_url( add_query_arg( 'paged', $paged - 1, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'Previous page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&lsaquo;</span></a>';
                            }
                            ?>
                            <span class="screen-reader-text"><?php esc_html_e( 'Current Page', 'orphanix-media-cleanup' ); ?></span>
                            <span class="paging-input">
                                <span class="tablenav-paging-text"><?php echo esc_html( intval( $paged ) ); ?> <?php esc_html_e( 'of', 'orphanix-media-cleanup' ); ?> <span class="total-pages"><?php echo esc_html( intval( $total_pages ) ); ?></span></span>
                            </span>
                            <?php
                            // Next page button
                            if ( $paged >= $total_pages ) {
                                echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&rsaquo;</span>';
                            } else {
                                echo '<a class="next-page button" href="' . esc_url( add_query_arg( 'paged', $paged + 1, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'Next page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&rsaquo;</span></a>';
                            }
                            
                            // Last page button
                            if ( $paged >= $total_pages ) {
                                echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&raquo;</span>';
                            } else {
                                echo '<a class="last-page button" href="' . esc_url( add_query_arg( 'paged', $total_pages, $current_url_base ) ) . '"><span class="screen-reader-text">' . esc_html__( 'Last page', 'orphanix-media-cleanup' ) . '</span><span aria-hidden="true">&raquo;</span></a>';
                            }
                            ?>
                        </span>
                    </div>
                <?php endif; ?>
            </form>
        </div>

        <?php
        ob_start();
        ?>
        (function($) {
            $(document).ready(function() {
                $('.orphanix-select-all').on('change', function() {
                    $('input[name="item_ids[]"]').prop('checked', this.checked);
                });
            });
        })(jQuery);
        <?php
        $inline_script = trim( (string) ob_get_clean() );
        wp_add_inline_script( 'orphanix-admin', $inline_script );
        ?>
        <?php
    }

    /**
     * Render "Belongs To" column with formatted usage context
     */
    private function render_usage_context( $usage_context ) {
        if ( empty( $usage_context ) ) {
            return '&ndash;';
        }

        // Decode JSON
        $context = json_decode( $usage_context, true );
        if ( ! is_array( $context ) ) {
            return esc_html( $usage_context );
        }

        $post_ids = $context['post_ids'] ?? [];
        $sources = $context['sources'] ?? [];

        if ( empty( $post_ids ) && empty( $sources ) ) {
            return '<span data-orphanix-style="orphanix-s-0087">Orphan</span>';
        }

        // Source label map
        $source_map = [
            'featured_image' => ['Featured Image', '#e74c3c'],
            'post_content' => ['Post Content', '#3498db'],
            'custom_field' => ['Custom Field', '#9b59b6'],
            'gutenberg' => ['Gutenberg Block', '#f39c12'],
            'elementor' => ['Elementor', '#4ecdc4'],
            'divi' => ['Divi', '#2ecc71'],
            'wpbakery' => ['WPBakery', '#e91e63'],
            'acf' => ['ACF', '#ff6b6b'],
            'metabox' => ['MetaBox', '#4a90e2'],
            'pods' => ['Pods', '#50c878'],
            'woocommerce' => ['WooCommerce', '#9c27b0'],
            'sliders' => ['Sliders', '#ff9800'],
            'widget' => ['Widget', '#795548'],
            'menu' => ['Menu', '#607d8b'],
            'theme' => ['Theme', '#455a64'],
            'deep_db' => ['Database', '#2c3e50'],
        ];

        $html = '<div data-orphanix-style="orphanix-s-0088">';

        // Separate numeric post IDs from table names
        $numeric_post_ids = [];
        $table_names = [];

        foreach ( $post_ids as $pid ) {
            if ( is_numeric( $pid ) ) {
                $numeric_post_ids[] = intval( $pid );
            } else {
                $table_names[] = $pid;
            }
        }

        // Display post titles with links
        if ( ! empty( $numeric_post_ids ) ) {
            $html .= '<div data-orphanix-style="orphanix-s-0089">';
            
            foreach ( array_unique( $numeric_post_ids ) as $post_id ) {
                $post = get_post( $post_id );
                if ( $post ) {
                    $post_type_obj = get_post_type_object( $post->post_type );
                    $post_type_label = $post_type_obj ? $post_type_obj->labels->singular_name : ucfirst( $post->post_type );
                    $edit_url = get_edit_post_link( $post_id );
                    $post_title = ! empty( $post->post_title ) ? $post->post_title : sprintf( '(ID: %d)', $post_id );

                    $html .= '<div data-orphanix-style="orphanix-s-0090">';
                    $html .= '<a href="' . esc_url( $edit_url ) . '" data-orphanix-style="orphanix-s-0091">' . esc_html( $post_title ) . '</a>';
                    $html .= '<span data-orphanix-style="orphanix-s-0092">' . esc_html( $post_type_label ) . '</span>';
                    $html .= '</div>';
                }
            }
            
            $html .= '</div>';
        }

        // Display sources as badges
        if ( ! empty( $sources ) ) {
            $html .= '<div data-orphanix-style="orphanix-s-0089">';
            $html .= '<span data-orphanix-style="orphanix-s-0093" title="The source badges showing where the image is used">Source: i</span>';
            
            foreach ( $sources as $source ) {
                $source_info = $source_map[ $source ] ?? [ $source, '#95a5a6' ];
                list( $label, $color ) = $source_info;

                $html .= '<span data-orphanix-style="orphanix-s-0094">' . esc_html( $label ) . '</span>';
            }
            
            $html .= '</div>';
        }

        // Display database table names
        if ( ! empty( $table_names ) ) {
            $html .= '<div data-orphanix-style="orphanix-s-0095">';
            $html .= '<strong data-orphanix-style="orphanix-s-0096" title="Database tables where the image was found">Tables: i</strong> ' . esc_html( implode( ', ', $table_names ) );
            $html .= '</div>';
        }

        $html .= '</div>';

        // Allow necessary HTML tags with style attributes
        $allowed_html = [
            'div' => ['style' => true, 'title' => true, 'data-orphanix-style' => true],
            'span' => ['style' => true, 'title' => true, 'data-orphanix-style' => true],
            'a' => ['href' => true, 'style' => true, 'title' => true, 'data-orphanix-style' => true],
            'strong' => ['style' => true, 'title' => true, 'data-orphanix-style' => true],
        ];

        return wp_kses( $html, $allowed_html );
    }
}
