<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class ORPHANIX_Analytics {
    public function render() {
        global $wpdb;
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Analytics', 'orphanix-media-cleanup'); ?></h1>
            <p><?php esc_html_e('Scan summaries, file usage reports, and storage savings insights.', 'orphanix-media-cleanup'); ?></p>

            <!-- Media Scan Analytics Section -->
            <div class="orphanix-analytics-section" data-orphanix-style="orphanix-s-0001">
                <h2><?php esc_html_e('Media Scan Analytics', 'orphanix-media-cleanup'); ?></h2>
                
                <?php
                // Check if any Media Scan is complete
                $media_scan_completed = $this->get_cached_var(
                    'media_scan_completed',
                    "SELECT id FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status IN (%s, %s) LIMIT %d",
                    [ 'media', 'completed', 'complete', 1 ]
                );

                if ( !$media_scan_completed ) {
                    echo '<p data-orphanix-style="orphanix-s-0002">' . esc_html__('Analytics data will appear here after the first scans are complete.', 'orphanix-media-cleanup') . '</p>';
                } else {
                    // Regular Scan
                    echo '<div class="orphanix-subsection" data-orphanix-style="orphanix-s-0003">';
                    echo '<h3>' . esc_html__('Regular Scan', 'orphanix-media-cleanup') . '</h3>';
                    $this->render_scan_analytics('media', 'regular');
                    echo '</div>';

                    // Deep Scan
                    echo '<div class="orphanix-subsection" data-orphanix-style="orphanix-s-0003">';
                    echo '<h3>' . esc_html__('Deep Scan', 'orphanix-media-cleanup') . '</h3>';
                    $this->render_scan_analytics('media', 'deep');
                    echo '</div>';
                }
                ?>
            </div>

            <!-- Broken Media Scan Analytics Section -->
            <div class="orphanix-analytics-section" data-orphanix-style="orphanix-s-0001">
                <h2><?php esc_html_e('Broken Media Scan Analytics', 'orphanix-media-cleanup'); ?></h2>
                
                <?php
                // Check if any Broken Scan is complete
                $broken_scan_completed = $this->get_cached_var(
                    'broken_scan_completed',
                    "SELECT id FROM {$wpdb->prefix}orphanix_scans WHERE scan_type = %s AND status IN (%s, %s) LIMIT %d",
                    [ 'broken', 'completed', 'complete', 1 ]
                );

                if ( !$broken_scan_completed ) {
                    echo '<p data-orphanix-style="orphanix-s-0002">' . esc_html__('Analytics data will appear here after the first scans are complete.', 'orphanix-media-cleanup') . '</p>';
                } else {
                    // Regular Scan
                    echo '<div class="orphanix-subsection" data-orphanix-style="orphanix-s-0003">';
                    echo '<h3>' . esc_html__('Regular Scan', 'orphanix-media-cleanup') . '</h3>';
                    $this->render_scan_analytics('broken', 'regular');
                    echo '</div>';

                    // Deep Scan
                    echo '<div class="orphanix-subsection" data-orphanix-style="orphanix-s-0003">';
                    echo '<h3>' . esc_html__('Deep Scan', 'orphanix-media-cleanup') . '</h3>';
                    $this->render_scan_analytics('broken', 'deep');
                    echo '</div>';
                }
                ?>
            </div>

        </div>
        <?php
    }

    /**
     * Render scan analytics for a specific scan type and mode
     */
    private function render_scan_analytics( $scan_type, $scan_mode ) {
        global $wpdb;

        // Get latest completed scan for this type and mode
        $latest_scan = $this->get_cached_row(
            'latest_scan_' . $scan_type . '_' . $scan_mode,
            "SELECT * FROM {$wpdb->prefix}orphanix_scans 
                WHERE scan_type = %s AND scan_mode = %s AND status IN ('completed', 'complete')
                ORDER BY completed_at DESC LIMIT 1",
            [$scan_type, $scan_mode]
        );

        if ( !$latest_scan ) {
            $no_scan_message = sprintf(
                /* translators: %s: scan mode label (e.g., Regular, Deep). */
                esc_html__( 'No completed %s scan yet.', 'orphanix-media-cleanup' ),
                ucfirst( $scan_mode )
            );
            echo '<p data-orphanix-style="orphanix-s-0004">' . esc_html( $no_scan_message ) . '</p>';
            return;
        }

        // Get analytics from scan items
        $scan_id = intval( $latest_scan->id );
        
        $total_files = intval( $latest_scan->total_files );
        
        $in_media_library = intval( $this->get_cached_var(
            'scan_count_media_' . $scan_id,
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d AND directory_type = 'media'",
            [$scan_id]
        ) );

        $not_in_media_library = intval( $this->get_cached_var(
            'scan_count_not_media_' . $scan_id,
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d AND directory_type = 'not-media'",
            [$scan_id]
        ) );

        $used = intval( $this->get_cached_var(
            'scan_count_used_' . $scan_id,
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d AND status = 'used'",
            [$scan_id]
        ) );

        $not_used = intval( $this->get_cached_var(
            'scan_count_not_used_' . $scan_id,
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d AND status = 'not-used'",
            [$scan_id]
        ) );

        $orphan = intval( $this->get_cached_var(
            'scan_count_orphan_' . $scan_id,
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d AND status = 'orphan'",
            [$scan_id]
        ) );

        $alt_tag_missing = intval( $this->get_cached_var(
            'scan_count_alt_missing_' . $scan_id,
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d AND (alt_text = '' OR alt_text IS NULL)",
            [$scan_id]
        ) );

        $alt_tag_present = intval( $this->get_cached_var(
            'scan_count_alt_present_' . $scan_id,
            "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d AND alt_text != '' AND alt_text IS NOT NULL",
            [$scan_id]
        ) );

        // Get broken links count (only for broken scans)
        $broken_count = 0;
        if ( 'broken' === $scan_type ) {
            $broken_count = intval( $this->get_cached_var(
                'scan_count_broken_' . $scan_id,
                "SELECT COUNT(*) FROM {$wpdb->prefix}orphanix_scan_items WHERE scan_id = %d AND status = 'broken'",
                [$scan_id]
            ) );
        }

        // Display analytics
        ?>
        <div class="orphanix-analytics-report" data-orphanix-style="orphanix-s-0005">
            <p data-orphanix-style="orphanix-s-0006"><strong><?php esc_html_e('Last Scan Date:', 'orphanix-media-cleanup'); ?></strong> <?php echo esc_html( wp_date( 'Y-m-d H:i:s', strtotime( $latest_scan->completed_at ) ) ); ?></p>
            
            <div data-orphanix-style="orphanix-s-0007">
                <div data-orphanix-style="orphanix-s-0008">
                    <p data-orphanix-style="orphanix-s-0009"><?php esc_html_e('Total Files Scanned', 'orphanix-media-cleanup'); ?></p>
                    <p data-orphanix-style="orphanix-s-0010"><?php echo esc_html( $total_files ); ?></p>
                </div>

                <?php if ( 'broken' === $scan_type ) : ?>
                <div data-orphanix-style="orphanix-s-0011">
                    <p data-orphanix-style="orphanix-s-0009"><?php esc_html_e('Total Broken Links Found', 'orphanix-media-cleanup'); ?></p>
                    <p data-orphanix-style="orphanix-s-0012"><?php echo esc_html( $broken_count ); ?></p>
                </div>
                <?php endif; ?>

                <div data-orphanix-style="orphanix-s-0013">
                    <p data-orphanix-style="orphanix-s-0009"><?php esc_html_e('In Media Library', 'orphanix-media-cleanup'); ?></p>
                    <p data-orphanix-style="orphanix-s-0014"><?php echo esc_html( $in_media_library ); ?></p>
                </div>

                <div data-orphanix-style="orphanix-s-0011">
                    <p data-orphanix-style="orphanix-s-0009"><?php esc_html_e('Not in Media Library', 'orphanix-media-cleanup'); ?></p>
                    <p data-orphanix-style="orphanix-s-0012"><?php echo esc_html( $not_in_media_library ); ?></p>
                </div>

                <div data-orphanix-style="orphanix-s-0008">
                    <p data-orphanix-style="orphanix-s-0009"><?php esc_html_e('Files Used', 'orphanix-media-cleanup'); ?></p>
                    <p data-orphanix-style="orphanix-s-0010"><?php echo esc_html( $used ); ?></p>
                </div>

                <div data-orphanix-style="orphanix-s-0015">
                    <p data-orphanix-style="orphanix-s-0009"><?php esc_html_e('Files Not Used', 'orphanix-media-cleanup'); ?></p>
                    <p data-orphanix-style="orphanix-s-0016"><?php echo esc_html( $not_used ); ?></p>
                </div>

                <div data-orphanix-style="orphanix-s-0011">
                    <p data-orphanix-style="orphanix-s-0009"><?php esc_html_e('Orphan Files', 'orphanix-media-cleanup'); ?></p>
                    <p data-orphanix-style="orphanix-s-0012"><?php echo esc_html( $orphan ); ?></p>
                </div>

                <div data-orphanix-style="orphanix-s-0011">
                    <p data-orphanix-style="orphanix-s-0009"><?php esc_html_e('Missing Alt Tag', 'orphanix-media-cleanup'); ?></p>
                    <p data-orphanix-style="orphanix-s-0012"><?php echo esc_html( $alt_tag_missing ); ?></p>
                </div>

                <div data-orphanix-style="orphanix-s-0013">
                    <p data-orphanix-style="orphanix-s-0009"><?php esc_html_e('Has Alt Tag', 'orphanix-media-cleanup'); ?></p>
                    <p data-orphanix-style="orphanix-s-0014"><?php echo esc_html( $alt_tag_present ); ?></p>
                </div>
            </div>

        </div>
        <?php
    }

    private function get_cached_row( $cache_key, $query, array $args ) {
        $cached = wp_cache_get( $cache_key, 'orphanix' );
        if ( false !== $cached ) {
            return $cached;
        }

        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $row = $wpdb->get_row( $wpdb->prepare( $query, $args ) );
        wp_cache_set( $cache_key, $row, 'orphanix', 300 );
        return $row;
    }

    private function get_cached_var( $cache_key, $query, array $args ) {
        $cached = wp_cache_get( $cache_key, 'orphanix' );
        if ( false !== $cached ) {
            return $cached;
        }

        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        $value = $wpdb->get_var( $wpdb->prepare( $query, $args ) );
        wp_cache_set( $cache_key, $value, 'orphanix', 300 );
        return $value;
    }
}


