<?php

use Automattic\WooCommerce\Utilities\OrderUtil;

if (!defined('ABSPATH')) {
    exit();
}

/**
 * OrkestaPay Gateway.
 *
 * Provides a Checkout Gateway.
 *
 * @class       OrkestaPay_Gateway
 * @extends     WC_Payment_Gateway
 */
class OrkestaPay_Gateway extends WC_Payment_Gateway
{
    protected $test_mode = true;
    protected $client_id;
    protected $client_secret;
    protected $plugin_version = '1.1.2';

    const STATUS_COMPLETED = 'COMPLETED';
    const STATUS_SUCCESS = 'SUCCESS';

    public function __construct()
    {
        $this->id = 'orkestapay'; // Payment gateway plugin ID
        $this->method_title = __('OrkestaPay', 'orkestapay');
        $this->method_description = __('Orchestrate multiple payment gateways for a frictionless, reliable, and secure checkout experience.', 'orkestapay');
        $this->has_fields = true;
        $this->supports = ['products', 'refunds', 'tokenization', 'add_payment_method'];

        $this->init_form_fields();
        $this->init_settings();

        $this->title = $this->settings['title'];
        $this->description = $this->settings['description'];
        $this->enabled = $this->settings['enabled'];
        $this->test_mode = strcmp($this->settings['test_mode'], 'yes') == 0;
        $this->client_id = $this->settings['client_id'];
        $this->client_secret = $this->settings['client_secret'];

        OrkestaPay_API::set_client_id($this->client_id);
        OrkestaPay_API::set_client_secret($this->client_secret);

        if ($this->test_mode) {
            $this->description .= __(' TEST MODE ENABLED. In test mode, you can use the card number 4242424242424242 with any CVC and a valid expiration date.', 'orkestapay');
        }

        add_action('wp_enqueue_scripts', [$this, 'payment_scripts']);
        add_action('admin_notices', [$this, 'admin_notices']);
        add_action('woocommerce_api_orkesta_return_url', [$this, 'orkesta_return_url']);
        // This action hook saves the settings
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
    }

    /**
     * Plugin options
     */
    public function init_form_fields()
    {
        $this->form_fields = [
            'enabled' => [
                'title' => __('Enable OrkestaPay', 'orkestapay'),
                'label' => __('Enable', 'orkestapay'),
                'type' => 'checkbox',
                'default' => 'no',
                'description' => __('Check the box to enable Orkesta as a payment method.', 'orkestapay'),
            ],
            'test_mode' => [
                'title' => __('Enable test mode', 'orkestapay'),
                'label' => __('Enable', 'orkestapay'),
                'type' => 'checkbox',
                'default' => 'yes',
                'description' => __('Check the box to make test payments.', 'orkestapay'),
            ],
            'title' => [
                'title' => __('Title', 'orkestapay'),
                'type' => 'text',
                'default' => __('Checkout', 'orkestapay'),
                'description' => __('Payment method title that the customer will see on your checkout.', 'orkestapay'),
            ],
            'description' => [
                'title' => __('Description', 'orkestapay'),
                'type' => 'textarea',
                'description' => __('Payment method description that the customer will see on your website.', 'orkestapay'),
                'default' => __('Pay with your credit or debit card.', 'orkestapay'),
            ],
            'client_id' => [
                'title' => __('Access Key', 'orkestapay'),
                'type' => 'text',
                'default' => '',
                'custom_attributes' => [
                    'autocomplete' => 'off',
                    'aria-autocomplete' => 'none',
                    'role' => 'presentation',
                ],
            ],
            'client_secret' => [
                'title' => __('Secret Key', 'orkestapay'),
                'type' => 'password',
                'default' => '',
                'custom_attributes' => [
                    'autocomplete' => 'off',
                    'aria-autocomplete' => 'none',
                    'role' => 'presentation',
                ],
            ],
        ];
    }

    /**
     * Handles admin notices
     *
     * @return void
     */
    public function admin_notices()
    {
        if ('no' == $this->enabled) {
            return;
        }

        /**
         * Check if WC is installed and activated
         */
        if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
            // WooCommerce is NOT enabled!
            echo wp_kses_post('<div class="error"><p>');
            echo esc_html_e('OrkestaPay needs WooCommerce plugin is installed and activated to work.', 'orkestapay');
            echo wp_kses_post('</p></div>');
            return;
        }
    }

    function admin_options()
    {
        wp_enqueue_style('font_montserrat', 'https://fonts.googleapis.com/css2?family=Montserrat:wght@400;600&display=swap', ORKESTAPAY_WC_PLUGIN_FILE, [], $this->plugin_version);
        wp_enqueue_style('orkesta_admin_style', plugins_url('assets/css/orkestapay-admin-style.css', ORKESTAPAY_WC_PLUGIN_FILE), [], $this->plugin_version);

        $this->logo = plugins_url('assets/images/orkestapay.svg', ORKESTAPAY_WC_PLUGIN_FILE);

        include_once dirname(__DIR__) . '/templates/admin.php';
    }

    public function process_admin_options()
    {
        // Guardar las opciones normalmente
        parent::process_admin_options();

        $this->test_mode = $this->get_option('test_mode', 'no') === 'yes';
        $client_id = $this->get_option('client_id', '');
        $client_secret = $this->get_option('client_secret', '');

        if (!$this->validateOrkestaCredentials($client_id, $client_secret)) {
            $options['enabled'] = 'no';
            update_option('woocommerce_' . $this->id . '_settings', $options);
            WC_Admin_Settings::add_error(__('Provided credentials are invalid.', 'orkestapay'));
        }
    }

    /**
     * Loads (enqueue) static files (js & css) for the checkout page
     *
     * @return void
     */
    public function payment_scripts()
    {
        if (!is_checkout()) {
            return;
        }

        wp_enqueue_style('orkestapay_checkout_style', plugins_url('assets/css/orkestapay-checkout-style.css', ORKESTAPAY_WC_PLUGIN_FILE), [], $this->plugin_version, 'all');
    }

    public function payment_fields()
    {
        $apiHost = $this->getApiHost();
        $this->brands = OrkestaPay_API::retrieve("$apiHost/v1/merchants/providers/brands");

        include_once dirname(__DIR__) . '/templates/payment.php';
    }

    /**
     * Process payment at checkout
     *
     * @return int $order_id
     */
    public function process_payment($order_id)
    {
        $order = wc_get_order($order_id);

        try {
            $successUrl = esc_url(WC()->api_request_url('orkesta_return_url')) . '?merchant_order_id=' . $order->get_id();
            $cancelUrl = wc_get_checkout_url();

            $apiHost = $this->getApiHost();
            $checkoutDTO = OrkestaPay_Helper::transform_data_4_checkout($order, $successUrl, $cancelUrl);
            $orkestaCheckout = OrkestaPay_API::request($checkoutDTO, "$apiHost/v1/checkouts");

            // Redirect to OrkestaPay checkout
            return [
                'result' => 'success',
                'redirect' => $orkestaCheckout->checkout_redirect_url,
            ];
        } catch (Exception $e) {
            OrkestaPay_Logger::error('#process_payment', ['error' => $e->getMessage()]);

            $order->update_status('failed');
            $order->save();

            wc_add_notice(__('A transaction error occurred. Your credit card has not been charged.', 'orkestapay-card'), 'error');

            return;
        }
    }

    /**
     * Process refund.
     *
     * If the gateway declares 'refunds' support, this will allow it to refund.
     *
     * @param  int        $order_id Order ID.
     * @param  float|null $amount Refund amount.
     * @param  string     $reason Refund reason.
     * @return bool|\WP_Error True or false based on success, or a WP_Error object.
     */
    public function process_refund($order_id, $amount = null, $reason = '')
    {
        $order = wc_get_order($order_id);

        // Se valida que la orden exista
        if (!$order) {
            return false;
        }

        // Verificar si HPOS está habilitado
        if (OrderUtil::custom_orders_table_usage_is_enabled()) {
            $orkestaOrderId = $order->get_meta('_orkestapay_order_id');
            $orkestaOrderId = $order->get_meta('_orkestapay_payment_id');
        } else {
            $orkestaOrderId = get_post_meta($order_id, '_orkestapay_order_id', true);
            $orkestaPaymentId = get_post_meta($order_id, '_orkestapay_payment_id', true);
        }

        OrkestaPay_Logger::log('#process_refund', [
            'order_id' => $order_id,
            'amount' => $amount,
            'reason' => $reason,
            'orkesta_order_id' => $orkestaOrderId,
            'orkesta_payment_id' => $orkestaPaymentId,
        ]);

        // Se valida que la orden tenga el $orkestaOrderId y $orkestaPaymentId
        if (OrkestaPay_Helper::is_null_or_empty_string($orkestaOrderId) || OrkestaPay_Helper::is_null_or_empty_string($orkestaPaymentId)) {
            return false;
        }

        try {
            $apiHost = $this->getApiHost();
            $orderNote = 'Automatic refunds are not enabled in OrkestaPay. Check this configuration in your OrkestaPay account.';
            $isRefundAvailable = OrkestaPay_API::retrieve("$apiHost/v1/merchants/settings?id=AVAILABLE_REFUND_BY_ECOMMERCE");
            $isRefunded = true;
            $refundData = ['description' => $reason, 'amount' => floatval($amount)];

            if ($isRefundAvailable->custom_settings->AVAILABLE_REFUND_BY_ECOMMERCE === true) {
                $refundResponse = OrkestaPay_API::request($refundData, "$apiHost/v1/payments/{$orkestaPaymentId}/refund", 'POST');
                $isRefunded = $refundResponse->status === self::STATUS_SUCCESS;
                $orderNote = $isRefunded ? 'Refund was successfully created.' : $refundResponse->provider->message;
            }

            $order->add_order_note($orderNote);

            return $isRefunded;
        } catch (Exception $e) {
            OrkestaPay_Logger::error('#orkesta_woocommerce_order_refunded', ['error' => $e->getMessage()]);
            $order->add_order_note('There was an error creating the refund: ' . $e->getMessage());

            return false;
        }
    }

    /**
     * Checks if the Orkesta key is valid
     *
     * @return boolean
     */
    protected function validateOrkestaCredentials($client_id, $client_secret)
    {
        $token_result = OrkestaPay_API::get_access_token($client_id, $client_secret, true);
        if (!array_key_exists('access_token', $token_result)) {
            OrkestaPay_Logger::error('#validateOrkestaCredentials', ['error' => 'Error al obtener access_token']);

            return false;
        }

        // Se valida que la respuesta sea un JWT
        $regex = preg_match('/^([a-zA-Z0-9_=]+)\.([a-zA-Z0-9_=]+)\.([a-zA-Z0-9_\-\+\/=]*)/', $token_result['access_token']);
        if ($regex !== 1) {
            return false;
        }

        return true;
    }

    public function getApiHost()
    {
        return $this->test_mode ? ORKESTAPAY_API_SAND_URL : ORKESTAPAY_API_URL;
    }

    public function orkesta_return_url()
    {
        $orkestaPaymentId = isset($_GET['payment_id']) ? $_GET['payment_id'] : $_GET['paymentId'];
        $wooOrderId = isset($_GET['merchant_order_id']) ? $_GET['merchant_order_id'] : $_GET['merchantOrderId'];
        OrkestaPay_Logger::log('#orkesta_return_url', ['orkestaPaymentId' => $orkestaPaymentId, 'wooOrderId' => $wooOrderId]);

        // Obtener el pago de OrkestaPay relacionado a la orden
        $apiHost = $this->getApiHost();
        $orkestaPayment = OrkestaPay_API::retrieve("$apiHost/v1/payments/$orkestaPaymentId");

        $order_id = intval($wooOrderId);
        $order = wc_get_order($order_id);

        if ($orkestaPayment->status !== self::STATUS_COMPLETED) {
            wc_add_notice(__('A transaction error occurred. Your credit card has not been charged.', 'orkestapay-card'), 'error');
            wp_safe_redirect(wc_get_checkout_url());
            exit();
        }

        // COMPLETED - we received the payment
        $order->payment_complete();
        $order->add_order_note(sprintf("%s payment completed with Payment ID '%s'", $this->method_title, $orkestaPayment->payment_id));

        // Verificar si HPOS está habilitado
        if (OrderUtil::custom_orders_table_usage_is_enabled()) {
            $order->update_meta_data('_orkestapay_order_id', $orkestaPayment->order_id);
            $order->update_meta_data('_orkestapay_payment_id', $orkestaPayment->payment_id);
            $order->save();
        } else {
            update_post_meta($order->get_id(), '_orkestapay_order_id', $orkestaPayment->order_id);
            update_post_meta($order->get_id(), '_orkestapay_payment_id', $orkestaPayment->payment_id);
        }

        // Remove cart
        WC()->cart->empty_cart();

        wp_safe_redirect($this->get_return_url($order));
        exit();
    }
}
