<?php

use Automattic\WooCommerce\Utilities\OrderUtil;

/**
 * OrkestaPayCard_Gateway class.
 *
 * @extends WC_Payment_Gateway
 */

if (!defined('ABSPATH')) {
    exit();
}

class OrkestaPayCard_Gateway extends WC_Payment_Gateway
{
    protected $test_mode = true;
    protected $merchant_id;
    protected $client_id;
    protected $client_secret;
    protected $public_key;
    protected $plugin_version = '1.2.0';

    const PAYMENT_ACTION_REQUIRED = 'PAYMENT_ACTION_REQUIRED';
    const STATUS_COMPLETED = 'COMPLETED';
    const STATUS_SUCCESS = 'SUCCESS';
    const STATUS_REJECTED = 'REJECTED';
    const STATUS_FAILED = 'FAILED';
    const THREE_D_SECURE_SPECIFIC = 'THREE_D_SECURE_SPECIFIC';
    const THREE_D_SECURE_AUTHENTICATION = 'THREE_D_SECURE_AUTHENTICATION';

    public function __construct()
    {
        $this->id = 'orkestapay-card'; // Payment gateway plugin ID
        $this->method_title = __('OrkestaPay Card', 'orkestapay-card');
        $this->method_description = __('Orchestrate multiple payment gateways for a frictionless, reliable, and secure checkout experience.', 'orkestapay-card');
        $this->has_fields = true;
        $this->supports = ['products', 'refunds', 'tokenization', 'add_payment_method'];

        $this->init_form_fields();
        $this->init_settings();

        $this->title = $this->settings['title'];
        $this->description = $this->settings['description'];

        $this->enabled = $this->settings['enabled'];
        $this->test_mode = strcmp($this->settings['test_mode'], 'yes') == 0;
        $this->merchant_id = $this->settings['merchant_id'];
        $this->client_id = $this->settings['client_id'];
        $this->client_secret = $this->settings['client_secret'];
        $this->public_key = $this->settings['public_key'];

        OrkestaPayCard_API::set_client_id($this->client_id);
        OrkestaPayCard_API::set_client_secret($this->client_secret);

        if ($this->test_mode) {
            $this->description .= __('TEST MODE ENABLED. In test mode, you can use the card number 4242424242424242 with any CVC and a valid expiration date.', 'orkestapay-card');
        }

        add_action('wp_enqueue_scripts', [$this, 'payment_scripts']);
        add_action('admin_notices', [$this, 'admin_notices']);
        add_action('woocommerce_api_orkestapay_card_return_url', [$this, 'orkestapay_card_return_url']);
        add_action('woocommerce_api_orkestapay_card_create_customer', [$this, 'orkestapay_card_create_customer']);

        // This action hook saves the settings
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
    }

    /**
     * Plugin options
     */
    public function init_form_fields()
    {
        $this->form_fields = [
            'enabled' => [
                'title' => __('Enable OrkestaPay', 'orkestapay-card'),
                'label' => __('Enable', 'orkestapay-card'),
                'type' => 'checkbox',
                'default' => 'no',
                'description' => __('Check the box to enable Orkesta as a payment method.', 'orkestapay-card'),
            ],
            'test_mode' => [
                'title' => __('Enable test mode', 'orkestapay-card'),
                'label' => __('Enable', 'orkestapay-card'),
                'type' => 'checkbox',
                'default' => 'yes',
                'description' => __('Check the box to make test payments.', 'orkestapay-card'),
            ],
            'title' => [
                'title' => __('Title', 'orkestapay-card'),
                'type' => 'text',
                'default' => __('Credit Card', 'orkestapay-card'),
                'description' => __('Payment method title that the customer will see on your checkout.', 'orkestapay-card'),
            ],
            'description' => [
                'title' => __('Description', 'orkestapay-card'),
                'type' => 'textarea',
                'description' => __('Payment method description that the customer will see on your website.', 'orkestapay-card'),
                'default' => __('Pay with your credit or debit card.', 'orkestapay-card'),
            ],
            'merchant_id' => [
                'title' => __('Merchant ID', 'orkestapay-card'),
                'type' => 'text',
                'default' => '',
                'custom_attributes' => [
                    'autocomplete' => 'off',
                    'aria-autocomplete' => 'none',
                ],
            ],
            'client_id' => [
                'title' => __('Client ID', 'orkestapay-card'),
                'type' => 'text',
                'default' => '',
                'custom_attributes' => [
                    'autocomplete' => 'off',
                    'aria-autocomplete' => 'none',
                    'role' => 'presentation',
                ],
            ],
            'client_secret' => [
                'title' => __('Client Secret', 'orkestapay-card'),
                'type' => 'password',
                'default' => '',
                'custom_attributes' => [
                    'autocomplete' => 'off',
                    'aria-autocomplete' => 'none',
                    'role' => 'presentation',
                ],
            ],
            'public_key' => [
                'title' => __('Public Key', 'orkestapay-card'),
                'type' => 'text',
                'default' => '',
                'custom_attributes' => [
                    'autocomplete' => 'off',
                    'aria-autocomplete' => 'none',
                ],
            ],
        ];
    }

    /**
     * Handles admin notices
     *
     * @return void
     */
    public function admin_notices()
    {
        if ('no' == $this->enabled) {
            return;
        }

        /**
         * Check if WC is installed and activated
         */
        if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
            // WooCommerce is NOT enabled!
            echo wp_kses_post('<div class="error"><p>');
            echo esc_html_e('OrkestaPay needs WooCommerce plugin is installed and activated to work.', 'orkestapay-card');
            echo wp_kses_post('</p></div>');
            return;
        }
    }

    function admin_options()
    {
        wp_enqueue_style('font_montserrat', 'https://fonts.googleapis.com/css2?family=Montserrat:wght@400;600&display=swap', ORKESTAPAY_CARD_WC_PLUGIN_FILE, [], $this->plugin_version);
        wp_enqueue_style('orkesta_admin_style', plugins_url('assets/css/admin-style.css', ORKESTAPAY_CARD_WC_PLUGIN_FILE), [], $this->plugin_version);

        $this->logo = plugins_url('assets/images/orkestapay.svg', ORKESTAPAY_CARD_WC_PLUGIN_FILE);

        include_once dirname(__DIR__) . '/templates/admin.php';
    }

    public function process_admin_options()
    {
        // Guardar las opciones normalmente
        parent::process_admin_options();

        $this->test_mode = $this->get_option('test_mode', 'no') === 'yes';
        $client_id = $this->get_option('client_id', '');
        $client_secret = $this->get_option('client_secret', '');

        if (!$this->validateOrkestaCredentials($client_id, $client_secret)) {
            $options['enabled'] = 'no';
            update_option('woocommerce_' . $this->id . '_settings', $options);
            WC_Admin_Settings::add_error(__('Provided credentials are invalid.', 'orkestapay-card'));
        }
    }

    /**
     * Loads (enqueue) static files (js & css) for the checkout page
     *
     * @return void
     */
    public function payment_scripts()
    {
        if (!is_checkout()) {
            return;
        }

        $orkestapay_customer_id = '';
        if (is_user_logged_in()) {
            $user_id = get_current_user_id();
            $orkestapay_customer_id = get_user_meta($user_id, 'orkestapay_customer_id', true);
        }

        $cart = WC()->cart;
        $payment_args = [
            'currency' => get_woocommerce_currency(),
            'total_amount' => $cart->total,
            'plugin_payment_gateway_id' => $this->id,
            'merchant_id' => $this->getMerchantId(),
            'public_key' => $this->getPublicKey(),
            'is_sandbox' => $this->isTestMode(),
            'orkestapay_create_customer_url' => $this->getOrkestapayCreateCustomerUrl(),
            'orkestapay_customer_id' => $orkestapay_customer_id,
        ];

        $js_url = ORKESTAPAY_CARD_JS_URL;

        wp_enqueue_script('wc-credit-card-form');
        wp_enqueue_script('orkestapay_card_js_resource', $js_url . '/script/orkestapay.js', [], $this->plugin_version, true);
        wp_enqueue_script('orkestapay_card_payment_js', plugins_url('assets/js/orkesta-payment.js', ORKESTAPAY_CARD_WC_PLUGIN_FILE), ['jquery'], $this->plugin_version, true);
        wp_enqueue_style('orkestapay_card_checkout_style', plugins_url('assets/css/checkout-style.css', ORKESTAPAY_CARD_WC_PLUGIN_FILE), [], $this->plugin_version, 'all');
        wp_localize_script('orkestapay_card_payment_js', 'orkestapay_card_payment_args', $payment_args);
    }

    public function payment_fields()
    {
        $apiHost = $this->getApiHost();
        $this->brands = OrkestaPayCard_API::retrieve("$apiHost/v1/merchants/providers/brands");

        $tokens = [];
        if (is_user_logged_in()) {
            $tokens = WC_Payment_Tokens::get_customer_tokens(get_current_user_id(), $this->id);
        }

        $this->tokens = $tokens;

        include_once dirname(__DIR__) . '/templates/payment.php';
    }

    public function validate_fields()
    {
        if (empty($_POST['orkestapay_device_session_id'])) {
            wc_add_notice('OrkestaPay Device Session ID is missing.', 'error');
            return false;
        }

        return true;
    }

    /**
     * Process payment at checkout
     *
     * @return int $order_id
     */
    public function process_payment($order_id)
    {
        $order = wc_get_order($order_id);

        try {
            $shouldSaveCard = isset($_POST['wc-' . $this->id . '-new-payment-method']) && $_POST['wc-' . $this->id . '-new-payment-method'] === 'true';
            // $savedTokenId = isset($_POST['wc-' . $this->id . '-payment-token']) ? wc_clean($_POST['wc-' . $this->id . '-payment-token']) : '';

            $orkestaPaymentMethodId = wc_clean($_POST['orkestapay_payment_method_id']);
            $deviceSessionId = wc_clean($_POST['orkestapay_device_session_id']);
            $successUrl = esc_url(WC()->api_request_url('orkestapay_card_return_url')) . '?merchant_order_id=' . $order->get_id();
            $cancelUrl = wc_get_checkout_url();

            // Si el cliente ha seleccionado guardar la tarjeta, se guarda el token
            if ($shouldSaveCard && is_user_logged_in()) {
                $this->saveCreditCardToken($orkestaPaymentMethodId);
            }

            // Si el cliente no ha seleccionado pagar con una tarjeta guardada
            // if ($savedTokenId && $savedTokenId !== 'new') {
            //     $token = WC_Payment_Tokens::get($savedTokenId);
            //     if ($token && $token->get_user_id() === get_current_user_id()) {
            //         $orkestaPaymentMethodId = $token->get_token(); // Se utiliza el token guardado para realizar el pago
            //     }
            // }

            $apiHost = $this->getApiHost();

            $orderDTO = OrkestaPayCard_Helper::transform_data_4_orders($order);
            $orkestaOrder = OrkestaPayCard_API::request($orderDTO, "$apiHost/v1/orders");

            $paymentDTO = OrkestaPayCard_Helper::transform_data_4_payment($orkestaOrder->order_id, $orkestaPaymentMethodId, $order->get_id(), $deviceSessionId, $successUrl, $cancelUrl);
            $orkestaPayment = OrkestaPayCard_API::request($paymentDTO, "$apiHost/v1/payments");

            // REJECTED - FAILED
            if ($orkestaPayment->status === self::STATUS_REJECTED || $orkestaPayment->status === self::STATUS_FAILED) {
                wc_add_notice('Hubo un problema con tu pago. Por favor, intenta nuevamente.', 'error');
                return [
                    'result' => 'failure',
                ];
            }

            // PAYMENT_ACTION_REQUIRED - 3DS SPECIFIC
            if ($orkestaPayment->status === self::PAYMENT_ACTION_REQUIRED && $orkestaPayment->user_action_required->type === self::THREE_D_SECURE_SPECIFIC) {
                return [
                    'result' => 'success',
                    'redirect' => $orkestaPayment->user_action_required->three_d_secure_specific->three_ds_redirect_url,
                ];
            }

            // PAYMENT_ACTION_REQUIRED - 3DS GLOBAL
            if ($orkestaPayment->status === self::PAYMENT_ACTION_REQUIRED && $orkestaPayment->user_action_required->type === self::THREE_D_SECURE_AUTHENTICATION) {
                return [
                    'result' => 'success',
                    'redirect' => $orkestaPayment->user_action_required->three_d_secure_authentication->three_ds_redirect_url,
                ];
            }

            $order->payment_complete();
            $order->add_order_note(sprintf("%s payment completed with Payment ID '%s'", $this->method_title, $orkestaPayment->payment_id));

            // Verificar si HPOS está habilitado
            if (OrderUtil::custom_orders_table_usage_is_enabled()) {
                $order->update_meta_data('_orkestapay_order_id', $orkestaPayment->order_id);
                $order->update_meta_data('_orkestapay_payment_id', $orkestaPayment->payment_id);
                $order->save();
            } else {
                update_post_meta($order->get_id(), '_orkestapay_order_id', $orkestaPayment->order_id);
                update_post_meta($order->get_id(), '_orkestapay_payment_id', $orkestaPayment->payment_id);
            }

            // Remove cart
            WC()->cart->empty_cart();

            // Redirect to the thank you page
            return [
                'result' => 'success',
                'redirect' => $this->get_return_url($order),
            ];
        } catch (Exception $e) {
            OrkestaPayCard_Logger::error('#process_payment', ['error' => $e->getMessage()]);

            $order->add_order_note(sprintf("%s Credit card payment failed with message: '%s'", $this->method_title, $e->getMessage()));
            $order->update_status('failed');
            $order->save();

            wc_add_notice(__('A transaction error occurred. Your credit card has not been charged.', 'orkestapay-card'), 'error');

            return;
        }
    }

    /**
     * Process refund.
     *
     * If the gateway declares 'refunds' support, this will allow it to refund.
     *
     * @param  int        $order_id Order ID.
     * @param  float|null $amount Refund amount.
     * @param  string     $reason Refund reason.
     * @return bool|\WP_Error True or false based on success, or a WP_Error object.
     */
    public function process_refund($order_id, $amount = null, $reason = '')
    {
        $order = wc_get_order($order_id);

        // Se valida que la orden exista
        if (!$order) {
            return false;
        }

        // Verificar si HPOS está habilitado
        if (OrderUtil::custom_orders_table_usage_is_enabled()) {
            $orkestaOrderId = $order->get_meta('_orkestapay_order_id', true);
            $orkestaOrderId = $order->get_meta('_orkestapay_payment_id', true);
        } else {
            $orkestaOrderId = get_post_meta($order_id, '_orkestapay_order_id', true);
            $orkestaPaymentId = get_post_meta($order_id, '_orkestapay_payment_id', true);
        }

        OrkestaPayCard_Logger::log('#process_refund', [
            'order_id' => $order_id,
            'amount' => $amount,
            'reason' => $reason,
            'orkesta_order_id' => $orkestaOrderId,
            'orkesta_payment_id' => $orkestaPaymentId,
        ]);

        if (OrkestaPayCard_Helper::is_null_or_empty_string($orkestaOrderId) || OrkestaPayCard_Helper::is_null_or_empty_string($orkestaPaymentId)) {
            return false;
        }

        try {
            $apiHost = $this->getApiHost();
            $orderNote = 'Automatic refunds are not enabled in OrkestaPay. Check this configuration in your OrkestaPay account.';
            $isRefundAvailable = OrkestaPayCard_API::retrieve("$apiHost/v1/merchants/settings?id=AVAILABLE_REFUND_BY_ECOMMERCE");
            $isRefunded = true;
            $refundData = ['description' => $reason, 'amount' => floatval($amount)];

            if ($isRefundAvailable->custom_settings->AVAILABLE_REFUND_BY_ECOMMERCE === true) {
                $refundResponse = OrkestaPayCard_API::request($refundData, "$apiHost/v1/payments/{$orkestaPaymentId}/refund", 'POST');
                $isRefunded = $refundResponse->status === self::STATUS_SUCCESS;
                $orderNote = $isRefunded ? 'Refund was successfully created.' : $refundResponse->provider->message;
            }

            $order->add_order_note($orderNote);

            return $isRefunded;
        } catch (Exception $e) {
            OrkestaPayCard_Logger::error('#orkesta_woocommerce_order_refunded', ['error' => $e->getMessage()]);
            $order->add_order_note('There was an error creating the refund: ' . $e->getMessage());

            return false;
        }
    }

    /**
     * Checks if the Orkesta key is valid
     *
     * @return boolean
     */
    protected function validateOrkestaCredentials($client_id, $client_secret)
    {
        $token_result = OrkestaPayCard_API::get_access_token($client_id, $client_secret, true);
        if (!array_key_exists('access_token', $token_result)) {
            OrkestaPayCard_Logger::error('#validateOrkestaCredentials', ['error' => 'Error al obtener access_token']);

            return false;
        }

        // Se valida que la respuesta sea un JWT
        $regex = preg_match('/^([a-zA-Z0-9_=]+)\.([a-zA-Z0-9_=]+)\.([a-zA-Z0-9_\-\+\/=]*)/', $token_result['access_token']);
        if ($regex !== 1) {
            return false;
        }

        return true;
    }

    public function getApiHost()
    {
        return $this->test_mode ? ORKESTAPAY_CARD_API_SAND_URL : ORKESTAPAY_CARD_API_URL;
    }

    public function orkestapay_card_return_url()
    {
        $orkestaPaymentId = isset($_GET['payment_id']) ? $_GET['payment_id'] : $_GET['paymentId'];
        $wooOrderId = isset($_GET['merchant_order_id']) ? $_GET['merchant_order_id'] : $_GET['merchantOrderId'];

        // Obtener el pago de OrkestaPay relacionado a la orden
        $apiHost = $this->getApiHost();
        $orkestaPayment = OrkestaPayCard_API::retrieve("$apiHost/v1/payments/$orkestaPaymentId");

        $order_id = intval($wooOrderId);
        $order = wc_get_order($order_id);

        if ($orkestaPayment->status !== self::STATUS_COMPLETED) {
            wc_add_notice(__('A transaction error occurred. Your credit card has not been charged.', 'orkestapay-card'), 'error');
            wp_safe_redirect(wc_get_checkout_url());
            exit();
        }

        // COMPLETED - we received the payment
        $order->payment_complete();
        $order->add_order_note(sprintf("%s payment completed with Payment ID '%s'", $this->method_title, $orkestaPayment->payment_id));

        // Verificar si HPOS está habilitado
        if (OrderUtil::custom_orders_table_usage_is_enabled()) {
            $order->update_meta_data('_orkestapay_order_id', $orkestaPayment->order_id);
            $order->update_meta_data('_orkestapay_payment_id', $orkestaPayment->payment_id);
            $order->save();
        } else {
            update_post_meta($order->get_id(), '_orkestapay_order_id', $orkestaPayment->order_id);
            update_post_meta($order->get_id(), '_orkestapay_payment_id', $orkestaPayment->payment_id);
        }

        // Remove cart
        WC()->cart->empty_cart();

        wp_safe_redirect($this->get_return_url($order));
        exit();
    }

    public function getMerchantId()
    {
        return $this->merchant_id;
    }

    public function getPublicKey()
    {
        return $this->public_key;
    }

    public function isTestMode()
    {
        return $this->test_mode;
    }

    public function getOrkestapayForm()
    {
        $apiHost = $this->getApiHost();
        $this->brands = OrkestaPayCard_API::retrieve("$apiHost/v1/merchants/providers/brands");

        // Captura el contenido del archivo PHP
        ob_start();
        include_once dirname(__DIR__) . '/templates/payment.php';
        $htmlContent = ob_get_clean(); // Obtiene la salida y la guarda en $htmlContent
        return $htmlContent; // Devuelve el HTML generado
    }

    /**
     * Create customer. Security is handled by WC.
     *
     */
    public function orkestapay_card_create_customer()
    {
        header('HTTP/1.1 200 OK');

        // Solo se puede crear customers para usuarios logueados
        if (!is_user_logged_in()) {
            wp_send_json_error(
                [
                    'result' => 'fail',
                    'message' => 'User not logged in.',
                ],
                400
            );

            die();
        }

        try {
            $user_id = get_current_user_id();
            $first_name = get_user_meta($user_id, 'first_name', true);
            $last_name = get_user_meta($user_id, 'last_name', true);
            $email = wp_get_current_user()->user_email;
            $data = [
                'email' => $email,
                'first_name' => $first_name,
                'last_name' => $last_name,
            ];

            // Se envía la petición a la API de OrkestaPay
            $apiHost = $this->getApiHost();
            $customerResponse = OrkestaPayCard_API::request($data, "$apiHost/v1/customers", 'POST');

            // Se guarda el customer_id en el meta del usuario
            $user_id = get_current_user_id();
            update_user_meta($user_id, 'orkestapay_customer_id', $customerResponse->customer_id);

            // Response
            wp_send_json_success([
                'customer_id' => $customerResponse->customer_id,
            ]);

            die();
        } catch (Exception $e) {
            OrkestaPayCard_Logger::error('#orkestapay_card_create_customer', ['error' => $e->getMessage()]);

            wp_send_json_error(
                [
                    'result' => 'fail',
                    'message' => $e->getMessage(),
                ],
                400
            );

            die();
        }
    }

    public function getOrkestapayCreateCustomerUrl()
    {
        return esc_url(WC()->api_request_url('orkestapay_card_create_customer'));
    }

    private function saveCreditCardToken($orkestaPaymentMethodId)
    {
        $apiHost = $this->getApiHost();
        $orkestaPaymentMethod = OrkestaPayCard_API::retrieve("$apiHost/v1/payment-methods/$orkestaPaymentMethodId");

        $token = new WC_Payment_Token_CC();
        $token->set_token($orkestaPaymentMethodId);
        $token->set_gateway_id($this->id);
        $token->set_card_type($orkestaPaymentMethod->card->brand);
        $token->set_last4($orkestaPaymentMethod->card->last_four);
        $token->set_expiry_month($orkestaPaymentMethod->card->expiration_month);
        $token->set_expiry_year($orkestaPaymentMethod->card->expiration_year);
        $token->set_user_id(get_current_user_id());
        $token->set_default(true);
        $token->save();

        return;
    }
}
