<?php
/*
Plugin Name: OrigiSafe — Advanced Image Optimizer (WebP) — Keep Originals Safe
Plugin URI: https://wordpress.org/plugins/origisafe-advanced-image-optimizer/
Description: Converts JPG/PNG uploads (and existing library) to WebP, moves originals to /uploads/_originals/, and updates attachment metadata so WordPress serves .webp.
Version: 0.0.122
Author: ipodguy79
Author URI: https://profiles.wordpress.org/ipodguy79/
Requires at least: 5.8
Tested up to: 6.9
Requires PHP: 7.4
License: GPL2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: origisafe-advanced-image-optimizer
*/


if (!defined('ABSPATH')) exit;

class HSBC_WebP_Only {
    const META_CONVERTED = '_hsbc_webp_converted';
    const META_ORIG_MAP  = '_hsbc_webp_orig_map';
    const META_PREV_META = '_hsbc_webp_prev_meta';
    const META_FAILED    = '_hsbc_webp_failed';

    const OPT_QUALITY       = 'hsbc_webp_quality';
    const OPT_SWEEP_CURSOR  = 'hsbc_webp_sweep_cursor';
    const OPT_REPAIR_CURSOR = 'hsbc_webp_repair_cursor';

    const OPT_BATCH_SIZE    = 'hsbc_webp_batch_size';

    // Sweep scope (wp-content relative folders, one per line)
    const OPT_SWEEP_INCLUDE = 'hsbc_webp_sweep_include';
    const OPT_SWEEP_EXCLUDE = 'hsbc_webp_sweep_exclude';

    // Prefer existing archived WebPs in uploads/_webp/ when re-converting
    const OPT_REUSE_ARCHIVED = 'hsbc_webp_reuse_archived_webp';



    // Optional: HTTP Basic Auth for loopback (needed if the site is password-protected at the web server)
    const OPT_LOOPBACK_USER = 'hsbc_webp_loopback_user';
    const OPT_LOOPBACK_PASS = 'hsbc_webp_loopback_pass';
    const OPT_JOB_STATE     = 'hsbc_webp_job_state';
    const OPT_STOP_REQUEST  = 'hsbc_webp_stop_request';



    const CRON_HOOK         = 'hsbc_webp_job_tick';

    const LOG_SUBDIR        = 'hsbc-webp-only/logs';
    const LOG_FILE          = 'webp-only.log';

    const OPT_REVERT_CURSOR = 'hsbc_webp_revert_cursor';
    const OPT_REVERT_ORIG_SWEEP_CURSOR = 'hsbc_webp_revert_orig_sweep_cursor';
    const OPT_REVERT_WPCONTENT_SWEEP_CURSOR = 'hsbc_webp_revert_wpcontent_sweep_cursor';
    const OPT_CLEAN_ORIG_CURSOR = 'hsbc_webp_clean_orig_cursor';
    const OPT_CLEAN_ORIG_RESET  = 'hsbc_webp_clean_orig_reset';
    const OPT_CLEAN_WEBP_CURSOR = 'hsbc_webp_clean_webp_cursor';
    const OPT_CLEAN_WEBP_RESET  = 'hsbc_webp_clean_webp_reset';
    const OPT_CLEAN_ORIG_DIR_CURSOR  = 'hsbc_webp_clean_orig_dir_cursor';
    const OPT_CLEAN_ORIG_FILE_CURSOR = 'hsbc_webp_clean_orig_file_cursor';

    // Marker used to confirm a loopback request actually reached WordPress (not cache/WAF/redirect HTML).
    const TICK_OK_MARKER    = 'HSBC_TICK_OK';

    public static function init() {
        register_activation_hook(__FILE__, [__CLASS__, 'on_activate']);

        add_filter('wp_generate_attachment_metadata', [__CLASS__, 'convert_on_generate_metadata'], 20, 2);

        // Background runner (WP-Cron)
        add_action(self::CRON_HOOK, [__CLASS__, 'cron_job_tick']);

        add_action('admin_menu', [__CLASS__, 'admin_menu']);
        add_action('admin_init', [__CLASS__, 'admin_init']);

        add_action('wp_ajax_hsbc_webp_bulk',   [__CLASS__, 'ajax_bulk_convert']);
        add_action('wp_ajax_hsbc_webp_repair', [__CLASS__, 'ajax_repair_existing']);
        add_action('wp_ajax_hsbc_webp_sweep',  [__CLASS__, 'ajax_sweep_uploads']);

        // Background controls + log viewer
        add_action('wp_ajax_hsbc_webp_job_start',  [__CLASS__, 'ajax_job_start']);
        add_action('wp_ajax_hsbc_webp_job_stop',   [__CLASS__, 'ajax_job_stop']);
        add_action('wp_ajax_hsbc_webp_job_status', [__CLASS__, 'ajax_job_status']);

        add_action('wp_ajax_hsbc_webp_log_tail',   [__CLASS__, 'ajax_log_tail']);
        add_action('wp_ajax_hsbc_webp_log_reset',  [__CLASS__, 'ajax_log_reset']);
        add_action('init', [__CLASS__, 'public_tick_endpoint']);
        add_action('rest_api_init', [__CLASS__, 'register_rest_endpoints']);

        // Background tick (self-runner loopback)
        add_action('wp_ajax_hsbc_webp_job_tick_async', [__CLASS__, 'ajax_job_tick_async']);
        add_action('wp_ajax_nopriv_hsbc_webp_job_tick_async', [__CLASS__, 'ajax_job_tick_async']);
        add_action('admin_post_hsbc_webp_job_tick_async', [__CLASS__, 'ajax_job_tick_async']);
        add_action('admin_post_nopriv_hsbc_webp_job_tick_async', [__CLASS__, 'ajax_job_tick_async']);

        // Revert (background)
        add_action('wp_ajax_hsbc_webp_revert', [__CLASS__, 'ajax_revert_batch']);
    }

    public static function on_activate() {
        self::ensure_originals_root();
        self::ensure_log_dir();
        if (get_option(self::OPT_QUALITY, null) === null) {
            update_option(self::OPT_QUALITY, 82);
        }
        if (get_option(self::OPT_BATCH_SIZE, null) === null) {
            update_option(self::OPT_BATCH_SIZE, 8);
        }

        if (get_option(self::OPT_REUSE_ARCHIVED, null) === null) {
            update_option(self::OPT_REUSE_ARCHIVED, 1);
        }
        if (get_option(self::OPT_SWEEP_INCLUDE, null) === null) {
            update_option(self::OPT_SWEEP_INCLUDE, '');
        }
        if (get_option(self::OPT_SWEEP_EXCLUDE, null) === null) {
            update_option(self::OPT_SWEEP_EXCLUDE, '');
        }
        if (get_option(self::OPT_SWEEP_CURSOR, null) === null) {
            update_option(self::OPT_SWEEP_CURSOR, '');
        }
        if (get_option(self::OPT_REPAIR_CURSOR, null) === null) {
            update_option(self::OPT_REPAIR_CURSOR, 0);
        }
        if (get_option(self::OPT_REVERT_CURSOR, null) === null) {
            update_option(self::OPT_REVERT_CURSOR, 0);
        }
        if (get_option(self::OPT_JOB_STATE, null) === null) {
            update_option(self::OPT_JOB_STATE, []);
        }
    }

    public static function admin_init() {
        register_setting('hsbc_webp_only', self::OPT_QUALITY, [
            'type' => 'integer',
            'sanitize_callback' => function($v) {
                $v = (int)$v;
                if ($v < 40) $v = 40;
                if ($v > 95) $v = 95;
                return $v;
            },
            'default' => 82,
        ]);

        register_setting('hsbc_webp_only', self::OPT_BATCH_SIZE, [
            'type' => 'integer',
            'sanitize_callback' => function($v) {
                $v = (int)$v;
                if ($v < 1) $v = 1;
                if ($v > 50) $v = 50;
                return $v;
            },
            'default' => 8,
        ]);

register_setting('hsbc_webp_only', self::OPT_REUSE_ARCHIVED, [
    'type' => 'integer',
    'sanitize_callback' => function($v) {
        return empty($v) ? 0 : 1;
    },
    'default' => 1,
]);

register_setting('hsbc_webp_only', self::OPT_SWEEP_INCLUDE, [
    'type' => 'string',
    'sanitize_callback' => function($v) {
        $v = is_string($v) ? $v : '';
        $v = wp_unslash($v);
        $v = str_replace(["\r"], ["\n"], $v);
        $lines = preg_split('/\n+/', $v);
        $out = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '') continue;
            $line = ltrim($line, "/\t ");
            if (strpos($line, '..') !== false) continue;
            $line = rtrim($line, '/');
            if ($line === '') continue;
            $out[] = $line;
        }
        $out = array_values(array_unique($out));
        return implode("\n", $out);
    },
    'default' => '',
]);

register_setting('hsbc_webp_only', self::OPT_SWEEP_EXCLUDE, [
    'type' => 'string',
    'sanitize_callback' => function($v) {
        $v = is_string($v) ? $v : '';
        $v = wp_unslash($v);
        $v = str_replace(["\r"], ["\n"], $v);
        $lines = preg_split('/\n+/', $v);
        $out = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '') continue;
            $line = ltrim($line, "/\t ");
            if (strpos($line, '..') !== false) continue;
            $line = rtrim($line, '/');
            if ($line === '') continue;
            $out[] = $line;
        }
        $out = array_values(array_unique($out));
        return implode("\n", $out);
    },
    'default' => '',
]);


        

        
    }

    public static function admin_menu() {
        add_media_page('OrigiSafe', 'OrigiSafe', 'manage_options', 'origisafe-advanced-image-optimizer', [__CLASS__, 'render_admin_page']);
    }

    public static function render_admin_page() {
        if (!current_user_can('manage_options')) return;
        $quality = (int) get_option(self::OPT_QUALITY, 82);
        $reuse_archived = !empty(get_option(self::OPT_REUSE_ARCHIVED, 1));
        $batch_size = (int) get_option(self::OPT_BATCH_SIZE, 8);
        $reuse_archived = (int) get_option(self::OPT_REUSE_ARCHIVED, 1);
        $sweep_include = (string) get_option(self::OPT_SWEEP_INCLUDE, '');
        $sweep_exclude = (string) get_option(self::OPT_SWEEP_EXCLUDE, '');
$job = self::get_job_state();
        ?>
        <div class="wrap">
            <h1>OrigiSafe</h1>
            <p>
                Converts JPEG/PNG to WebP, moves originals to <code>uploads/_originals/</code>,
                and updates attachment metadata so WordPress outputs <code>.webp</code> URLs.
            </p>

            <form method="post" action="options.php">
                <?php settings_fields('hsbc_webp_only'); ?>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="<?php echo esc_attr(self::OPT_QUALITY); ?>">WebP Quality</label></th>
                        <td>
                            <input type="number" min="40" max="95"
                                   name="<?php echo esc_attr(self::OPT_QUALITY); ?>"
                                   id="<?php echo esc_attr(self::OPT_QUALITY); ?>"
                                   value="<?php echo esc_attr($quality); ?>" />
                            <p class="description">Typical: 80–85.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="<?php echo esc_attr(self::OPT_BATCH_SIZE); ?>">Batch Size</label></th>
                        <td>
                            <input type="number" min="1" max="50"
                                   name="<?php echo esc_attr(self::OPT_BATCH_SIZE); ?>"
                                   id="<?php echo esc_attr(self::OPT_BATCH_SIZE); ?>"
                                   value="<?php echo esc_attr($batch_size); ?>" />
                            <p class="description">Default: 8. Lower this if you hit timeouts with huge images.</p>
                        </td>
                    </tr>

<tr>
    <th scope="row"><label for="<?php echo esc_attr(self::OPT_REUSE_ARCHIVED); ?>">Reuse archived WebP</label></th>
    <td>
        <label>
            <input type="checkbox"
                   name="<?php echo esc_attr(self::OPT_REUSE_ARCHIVED); ?>"
                   id="<?php echo esc_attr(self::OPT_REUSE_ARCHIVED); ?>"
                   value="1" <?php checked(!empty($reuse_archived)); ?> />
            Prefer existing files in <code>uploads/_webp/</code> when converting again (faster reruns).
        </label>
    </td>
</tr>
<tr>
    <th scope="row"><label for="<?php echo esc_attr(self::OPT_SWEEP_INCLUDE); ?>">Sweep include folders</label></th>
    <td>
        <textarea name="<?php echo esc_attr(self::OPT_SWEEP_INCLUDE); ?>"
                  id="<?php echo esc_attr(self::OPT_SWEEP_INCLUDE); ?>"
                  rows="5" cols="60"
                  placeholder="uploads/2016/07&#10;uploads/2024&#10;gallery"><?php echo esc_textarea($sweep_include); ?></textarea>
        <p class="description">
            Optional. One folder per line, relative to <code>wp-content/</code>. If empty, Sweep scans all of <code>uploads/</code>.
        </p>
    </td>
</tr>
<tr>
    <th scope="row"><label for="<?php echo esc_attr(self::OPT_SWEEP_EXCLUDE); ?>">Sweep exclude folders</label></th>
    <td>
        <textarea name="<?php echo esc_attr(self::OPT_SWEEP_EXCLUDE); ?>"
                  id="<?php echo esc_attr(self::OPT_SWEEP_EXCLUDE); ?>"
                  rows="5" cols="60"
                  placeholder="uploads/cache&#10;uploads/_originals&#10;uploads/_webp"><?php echo esc_textarea($sweep_exclude); ?></textarea>
        <p class="description">
            Optional. One folder per line, relative to <code>wp-content/</code>. Excludes always win.
        </p>
    </td>
</tr>

</table>
                <?php submit_button('Save Settings'); ?>
            </form>

            <hr />

            <h2>Conversion Jobs</h2>
            <p>
                <label>
                    <input type="checkbox" id="hsbc_replace_everywhere" checked />
                    Replace old <code>.jpg/.png</code> URLs across content + postmeta (serialized-safe). (Recommended)
                </label>
            </p>

            <p class="description" style="max-width: 900px;">
                These jobs can self-run in the background, meaning they keep going even if you reload the page.
            </p>

            <p>
                <button class="button button-primary" id="hsbc_bg_bulk">Start Bulk Convert</button>
                <button class="button" id="hsbc_bg_repair">Start Repair/Cleanup </button>
                <button class="button" id="hsbc_bg_sweep">Start Folder Sweep </button>
                <button class="button button-secondary" id="hsbc_bg_stop" disabled>Stop Job</button>
            </p>

            

            <div id="hsbc_status" style="margin-top:12px; padding:10px; background:#fff; border:1px solid #ccd0d4;">
                <?php
                if (!empty($job['running'])) {
                    echo 'Job running: ' . esc_html($job['mode']) . ' (batch size ' . (int)$batch_size . ').';
                } else {
                    echo 'Ready.';
                }
                ?>
            </div>

            <h2 style="margin-top: 24px;">Log</h2>
            <p>
                <button class="button" id="hsbc_log_refresh">Refresh Log</button>
                <button class="button" id="hsbc_log_reset" style="border-color:#dc3232; color:#dc3232;">Reset Log</button>
            </p>
            <div id="hsbc_terminal" style="
                background:#0b0f0b;
                color:#39ff14;
                border:1px solid #1f2a1f;
                padding:12px;
                border-radius:4px;
                max-height: 420px;
                overflow:auto;
                font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, 'Liberation Mono', 'Courier New', monospace;
                font-size: 12px;
                line-height: 1.35;
                white-space: pre;
            "><?php echo esc_html(self::log_tail(250)); ?></div>
        </div>

        <hr />

            <h2>Danger Zone</h2>

            <p style="margin-top:14px; max-width: 900px;">
              <strong>Cleanup (uploads only)</strong><br>
              These tools delete only files that are safe to remove (duplicates/orphans). They do not delete “live” originals. Use these if you reverted and are heading back to WebP, or if you want to clean up leftovers from testing. The plugin will not overwrite files. The plugin will not delete any original unless it finds a verified match (duplicate).
              <br>• <strong>_originals cleanup</strong> Removes <em>duplicate backup copies</em> from <code>uploads/_originals/</code>, <em>only when the same file already exists </em> in <code>uploads/</code>.(same relative path and filename). For “derived image names” (WP resized filenames), it deletes them if a master exists in uploads or originals. 
              <br>• <strong>_webp cleanup</strong> Removes <em>ALL WebP files</em> inside <code>uploads/</code> or <code>wp-content/"each sweep root"/_webp/</code>. There is no “orphan” check and no “duplicate” check in this function.
              <br>• <strong>clean_sweep_originals</strong> deletes from <code>wp-content/"each sweep root"/_originals/</code> only when the same file already exists in the root sweep folder.
            </p>

            <p>
              <button class="button button-danger" id="hsbc_bg_clean_originals" style="border-color:#dc3232;color:#dc3232;">
                Delete duplicate _originals
              </button>
              <button class="button button-danger" id="hsbc_bg_clean_webp" style="border-color:#dc3232;color:#dc3232; margin-left:8px;">
                Delete duplicate _webp 
              </button>
              <button class="button button-danger" id="hsbc_bg_clean_sweep_originals" style="border-color:#dc3232;color:#dc3232; margin-left:8px;">
                Delete duplicate sweep _originals
              </button>
            </p>

            <p style="max-width: 900px; margin-top: 14px;">
              <strong>Revert Everything</strong><br>
              Restores originals and archives WebPs. Reverts <strong>uploads</strong> (restores JPG/PNG from <code>uploads/_originals/</code> back into <code>uploads/</code>, moves WebPs into <code>uploads/_webp/</code>, and restores Media Library metadata/mime). It also reverts any enabled <strong>sweep folders</strong> by restoring files from <code>&lt;sweep-root&gt;/_originals/</code> back into <code>&lt;sweep-root&gt;/</code> and moving matching WebPs into <code>&lt;sweep-root&gt;/_webp/</code>. <strong>Does not overwrite existing originals.</strong>
            </p>

            <p>
              <button class="button button-danger" id="hsbc_bg_revert" style="border-color:#dc3232; color:#dc3232;">
                Revert Everything
              </button>
            </p>

        <script>
        (function(){
            const bulkBtn   = document.getElementById('hsbc_bg_bulk');
            const repairBtn = document.getElementById('hsbc_bg_repair');
            const sweepBtn  = document.getElementById('hsbc_bg_sweep');
            const stopBtn   = document.getElementById('hsbc_bg_stop');
            const revertBtn = document.getElementById('hsbc_bg_revert');
            const statusEl  = document.getElementById('hsbc_status');
            const replaceCb = document.getElementById('hsbc_replace_everywhere');
            const cleanOrigBtn = document.getElementById('hsbc_bg_clean_originals');
            const cleanWebpBtn = document.getElementById('hsbc_bg_clean_webp');
            const cleanSweepOrigBtn = document.getElementById('hsbc_bg_clean_sweep_originals');


            const termEl = document.getElementById('hsbc_terminal');
            const logRefreshBtn = document.getElementById('hsbc_log_refresh');
            const logResetBtn = document.getElementById('hsbc_log_reset');

            let polling = false;

            const NONCE_START  = '<?php echo esc_js(wp_create_nonce('hsbc_webp_job_start')); ?>';
            const NONCE_STOP   = '<?php echo esc_js(wp_create_nonce('hsbc_webp_job_stop')); ?>';
            const NONCE_STATUS = '<?php echo esc_js(wp_create_nonce('hsbc_webp_job_status')); ?>';
            const NONCE_LOG    = '<?php echo esc_js(wp_create_nonce('hsbc_webp_log_tail')); ?>';
            const NONCE_LOGRST = '<?php echo esc_js(wp_create_nonce('hsbc_webp_log_reset')); ?>';

            function setStatus(msg){ statusEl.textContent = msg; }

            function setButtons(running){
                bulkBtn.disabled = running;
                repairBtn.disabled = running;
                sweepBtn.disabled = running;
                revertBtn.disabled = running;
                stopBtn.disabled = !running;
            }

            async function post(action, payload){
                const form = new FormData();
                form.append('action', action);
                for(const k in payload){ form.append(k, payload[k]); }
                const res = await fetch(ajaxurl, { method:'POST', body: form, credentials:'same-origin' });
                return await res.json();
            }

            function renderJobStatus(d){
              if(!d || !d.job){
                setButtons(false);
                setStatus('Ready.');
                stopLogAuto();
                return false;
            }

            const j = d.job;
            if(!j.running){
                setButtons(false);
                setStatus('Ready.');
                stopLogAuto();
                return false;
            }

            setButtons(true);
            setStatus(`Job running: ${j.mode}. (See log below.)`);
            startLogAuto();
            return true;
            }

            async function refreshLog(){
                try{
                    const data = await post('hsbc_webp_log_tail', { _ajax_nonce: NONCE_LOG, lines: '250' });
                    if(data && data.success && data.data){
                        termEl.textContent = data.data.text || '';
                        termEl.scrollTop = termEl.scrollHeight;
                    }
                }catch(e){
                    // ignore
                }
            }

            let pollTimer = null;
            let logTimer = null;
            function startLogAuto(){
              if (logTimer) return;
              logTimer = setInterval(refreshLog, 2000);
            }
            function stopLogAuto(){
              if (!logTimer) return;
              clearInterval(logTimer);
              logTimer = null;
            }

            function stopPolling(){
                if(pollTimer){
                    clearTimeout(pollTimer);
                    pollTimer = null;
                }
                polling = false;
            }

            async function poll(){
                if(polling) return;
                polling = true;

                try{
                    const data = await post('hsbc_webp_job_status', { _ajax_nonce: NONCE_STATUS });
                    if(data && data.success){
                        const running = renderJobStatus(data.data);

                        if(running){
                            await refreshLog();
                            polling = false;
                            pollTimer = setTimeout(poll, 2000);
                            return;
                        } else {
                            // job stopped: refresh log once more and STOP polling completely
                            await refreshLog();
                            stopPolling();
                            return;
                        }
                    }
                }catch(e){
                    // ignore
                }

                // On error: back off *only if a job is running*; otherwise stop.
                polling = false;
                pollTimer = setTimeout(poll, 5000);
            }

            async function start(mode){
                try{
                    const data = await post('hsbc_webp_job_start', {
                        _ajax_nonce: NONCE_START,
                        mode: mode,
                        replace_everywhere: replaceCb.checked ? '1' : '0'
                    });
                    if(!data || !data.success){
                        setStatus('Error: ' + (data && data.data ? data.data : 'Unknown')); 
                        return;
                    }
                    renderJobStatus(data.data);
                    await refreshLog();
                    stopPolling();
                    poll();
                }catch(e){
                    setStatus('Error: ' + e.message);
                }
            }

            bulkBtn.addEventListener('click', () => start('bulk'));
            repairBtn.addEventListener('click', () => start('repair'));
            sweepBtn.addEventListener('click', () => start('sweep'));
            if(cleanOrigBtn){
              cleanOrigBtn.addEventListener('click', async () => {
                const t = prompt('Type DELETE to confirm:\n\nDelete safe duplicates in uploads/_originals (only where uploads already has the file).');
                if(t !== 'DELETE') return;
                await start('clean_originals');
              });
            }
            if(cleanWebpBtn){
            cleanWebpBtn.addEventListener('click', async () => {
                const t = prompt('Type DELETE to confirm:\n\nDelete orphan files in uploads/_webp (no match in uploads or uploads/_originals).');
                if(t !== 'DELETE') return;
                await start('clean_webp');
            });
            }
            if(cleanSweepOrigBtn){
              cleanSweepOrigBtn.addEventListener('click', async () => {
                const t = prompt('Type DELETE to confirm:\n\nDelete safe duplicates in sweep root _originals folders (example: wp-content/gallery/_originals), only where the root already has the file.');
                if(t !== 'DELETE') return;
                await start('clean_sweep_originals');
            });
            }
            revertBtn.addEventListener('click', async () => {
              const t = prompt('Type REVERT to confirm:\n\nREVERT EVERYTHING (restores JPG/PNG from uploads/_originals and moves WebP into uploads/_webp).');
              if(t !== 'REVERT') return;
              await start('revert');
            });

            stopBtn.addEventListener('click', async () => {
                try{
                    const data = await post('hsbc_webp_job_stop', { _ajax_nonce: NONCE_STOP });
                    if(data && data.success){
                        renderJobStatus(data.data);
                        await refreshLog();
                        return;
                    }
                    setStatus('Stop failed: ' + (data && data.data ? data.data : 'Unknown'));
                }catch(e){
                    setStatus('Stop failed: ' + e.message);
                }
            });

            logRefreshBtn.addEventListener('click', refreshLog);
            logResetBtn.addEventListener('click', async () => {
                if(!confirm('Reset the log? This deletes ALL plugin log files. Continue?')) return;
                try{
                    const data = await post('hsbc_webp_log_reset', { _ajax_nonce: NONCE_LOGRST });
                    if(data && data.success){
                        termEl.textContent = '';
                        await refreshLog();
                        return;
                    }
                    setStatus('Log reset failed: ' + (data && data.data ? data.data : 'Unknown'));
                }catch(e){
                    setStatus('Log reset failed: ' + e.message);
                }
            });

            // Boot
            setButtons(<?php echo !empty($job['running']) ? 'true' : 'false'; ?>);
            refreshLog();
            // Always auto-tail while a job is running
            <?php if (!empty($job['running'])): ?>
            startLogAuto();
            <?php endif; ?>

            // Only poll if a job is already running.
            <?php if (!empty($job['running'])): ?>
            poll();
            <?php endif; ?>
        })();
        </script>
        <?php
    }

    // ---------------- Core helpers ----------------

    private static function ensure_originals_root() {
        $u = wp_upload_dir();
        $root = trailingslashit($u['basedir']) . '_originals/';
        if (!file_exists($root)) wp_mkdir_p($root);
        return $root;
    }

    private static function get_upload_baseurl() {
        $u = wp_upload_dir();
        return rtrim($u['baseurl'], '/');
    }

    private static function get_upload_basedir() {
        $u = wp_upload_dir();
        return rtrim($u['basedir'], DIRECTORY_SEPARATOR);
    }

    private static function is_target_mime($mime) {
        return in_array($mime, ['image/jpeg', 'image/png'], true);
    }

    private static function relpath_from_basedir($abs) {
        $basedir = trailingslashit(self::get_upload_basedir());
        if (is_string($abs) && strpos($abs, $basedir) === 0) {
            return ltrim(substr($abs, strlen($basedir)), '/');
        }
        return '';
    }

    private static function change_ext_to_webp($path) {
        return preg_replace('~\.(jpe?g|png)$~i', '.webp', $path);
    }

    private static function image_editor_can_webp() {
        return wp_image_editor_supports(['mime_type' => 'image/webp']);
    }

    private static function convert_file_to_webp($abs_src, $abs_dest, $quality) {
        $editor = wp_get_image_editor($abs_src);
        if (is_wp_error($editor)) return $editor;

        $dir = dirname($abs_dest);
        if (!file_exists($dir)) wp_mkdir_p($dir);

        if (method_exists($editor, 'set_quality')) {
            $editor->set_quality((int)$quality);
        }

        $saved = $editor->save($abs_dest, 'image/webp');
        if (is_wp_error($saved)) return $saved;
        return true;
    }

    private static function originals_abs_path_for_rel($rel) {
        $basedir = self::get_upload_basedir();
        return $basedir . '/_originals/' . ltrim($rel, '/');
    }

    private static function ensure_webp_archive_root() {
        $u = wp_upload_dir();
        $root = trailingslashit($u['basedir']) . '_webp/';
        if (!file_exists($root)) wp_mkdir_p($root);
        return $root;
    }

    private static function webp_archive_abs_path_for_rel($rel) {
        $basedir = self::get_upload_basedir();
        return $basedir . '/_webp/' . ltrim($rel, '/');
    }
    private static function fs_move($from, $to) {
        if (!file_exists($from)) return false;

        $dir = dirname($to);
        if (!is_dir($dir)) wp_mkdir_p($dir);

        if (function_exists('WP_Filesystem')) {
            global $wp_filesystem;

            if (!$wp_filesystem) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
                WP_Filesystem();
            }

            if ($wp_filesystem && method_exists($wp_filesystem, 'move')) {
                return (bool) $wp_filesystem->move($from, $to, false);
            }
        }
        // phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- WP_Filesystem::move attempted above; rename used as local fallback
        return @rename($from, $to);
    }

    private static function fs_delete($path) {
        if (!file_exists($path)) return true;

        if (function_exists('wp_delete_file')) {
            $r = wp_delete_file($path);
            return ($r !== false);
        }

        // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- wp_delete_file attempted above; unlink used as local fallback
        return @unlink($path);
    }

    private static function move_rel_to_webp_archive($rel) {
        $basedir = self::get_upload_basedir();
        $rel = ltrim($rel, '/');
        $src = $basedir . '/' . $rel;
        if (!file_exists($src)) return [false, 'missing'];

        $dst = self::webp_archive_abs_path_for_rel($rel);
        $dst_dir = dirname($dst);
        if (!is_dir($dst_dir)) wp_mkdir_p($dst_dir);

        if (file_exists($dst)) return [true, 'exists'];

        // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- wp_delete_file attempted above; unlink used as local fallback
        if (self::fs_move($src, $dst)) return [true, 'moved'];
        if (@copy($src, $dst) && self::fs_delete($src)) return [true, 'copied'];

        // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- wp_delete_file attempted above; unlink used as local fallback
        return [false, 'move_failed'];
    }

    private static function move_to_webp_archive($abs_path) {
        $rel = self::relpath_from_basedir($abs_path);
        if (!$rel) return [false, 'bad_rel'];
        return self::move_rel_to_webp_archive($rel);
    }

    private static function restore_rel_from_originals($rel) {
        $basedir = self::get_upload_basedir();
        $rel = ltrim($rel, '/');
        $src = self::originals_abs_path_for_rel($rel);
        if (!file_exists($src)) return [false, 'missing'];

        $dst = $basedir . '/' . $rel;
        $dst_dir = dirname($dst);
        if (!is_dir($dst_dir)) wp_mkdir_p($dst_dir);

        if (file_exists($dst)) return [true, 'exists'];

        if (self::fs_move($src, $dst)) return [true, 'restored'];
        if (@copy($src, $dst) && self::fs_delete($src)) return [true, 'copied'];

        return [false, 'restore_failed'];
    }

    private static function move_rel_to_originals($rel) {
        $basedir = self::get_upload_basedir();
        $rel = ltrim($rel, '/');
        $src = $basedir . '/' . $rel;
        if (!file_exists($src)) return [false, 'missing'];

        $dst = self::originals_abs_path_for_rel($rel);
        $dst_dir = dirname($dst);
        if (!is_dir($dst_dir)) wp_mkdir_p($dst_dir);

        if (file_exists($dst)) return [true, 'exists'];

        if (self::fs_move($src, $dst)) return [true, 'moved'];
        if (@copy($src, $dst) && self::fs_delete($src)) return [true, 'copied'];

        return [false, 'move_failed'];
    }

    private static function move_to_originals($abs_path) {
        $rel = self::relpath_from_basedir($abs_path);
        if (!$rel) return [false, 'bad_rel'];
        return self::move_rel_to_originals($rel);
    }

    // ---------------- Logging ----------------

    private static function ensure_log_dir() {
        $basedir = self::get_upload_basedir();
        $dir = trailingslashit($basedir) . self::LOG_SUBDIR;
        if (!file_exists($dir)) {
            wp_mkdir_p($dir);
        }
        return $dir;
    }

    private static function get_log_file_path() {
        $dir = self::ensure_log_dir();
        return trailingslashit($dir) . self::LOG_FILE;
    }

    private static function maybe_rotate_log($path) {
        if (!file_exists($path)) return;
        $max = 1024 * 1024; // 1MB
        $sz = @filesize($path);
        if ($sz !== false && $sz >= $max) {
            $dir = dirname($path);
            $stamp = gmdate('Ymd-His');
            $rot = $dir . '/webp-only-' . $stamp . '.log';
            self::fs_move($path, $rot);
        }
    }

    private static function log_write($message, $level = 'INFO') {
        $path = self::get_log_file_path();
        self::maybe_rotate_log($path);
        $ts = gmdate('Y-m-d H:i:s');
        $line = '[' . $ts . ' UTC] ' . $level . ' ' . trim((string)$message) . "\n";
        @file_put_contents($path, $line, FILE_APPEND);
    }

    private static function log_tail($lines = 250) {
        $path = self::get_log_file_path();
        if (!file_exists($path)) return '';

        $lines = max(1, min(2000, (int)$lines));
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- direct log tail read; WP_Filesystem not suitable for tailing
        $fp = @fopen($path, 'rb');
        if (!$fp) return '';

        $buffer = '';
        $chunk = 4096;
        $pos = -1;
        $line_count = 0;

        fseek($fp, 0, SEEK_END);
        $filesize = ftell($fp);
        if ($filesize === 0) {
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- direct log tail read
            fclose($fp);
            return '';
        }

        while ($line_count <= $lines && -$pos < $filesize) {
            $seek = max(-$filesize, $pos - $chunk);
            fseek($fp, $seek, SEEK_END);
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fread -- direct log tail read
            $data = fread($fp, abs($pos - $seek));
            $buffer = $data . $buffer;
            $line_count = substr_count($buffer, "\n");
            $pos = $seek;
        }

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- direct log tail read
        fclose($fp);
        $parts = preg_split("/\r?\n/", trim($buffer));
        if (!$parts) return '';
        $parts = array_slice($parts, -$lines);
        return implode("\n", $parts);
    }

    private static function log_reset_all() {
        $dir = self::ensure_log_dir();
        $files = glob(trailingslashit($dir) . 'webp-only*.log');
        if (is_array($files)) {
            foreach ($files as $f) {
                self::fs_delete($f);
            }
        }
        $cur = self::get_log_file_path();
        self::fs_delete($cur);
    }

    // ---------------- Background job state ----------------

    private static function get_job_state() {
        $job = get_option(self::OPT_JOB_STATE, []);
        return is_array($job) ? $job : [];
    }

    private static function set_job_state($job) {
        if (!is_array($job)) $job = [];
        update_option(self::OPT_JOB_STATE, $job);
    }
    private static function set_stop_request($reason) {
        update_option(self::OPT_STOP_REQUEST, (string)$reason, false);
    }

    private static function get_stop_request() {
        $v = get_option(self::OPT_STOP_REQUEST, '');
        return is_string($v) ? $v : '';
    }

    private static function clear_stop_request() {
        delete_option(self::OPT_STOP_REQUEST);
    }

    private static function unschedule_ticks() {
        $ts = wp_next_scheduled(self::CRON_HOOK);
        while ($ts) {
            wp_unschedule_event($ts, self::CRON_HOOK);
            $ts = wp_next_scheduled(self::CRON_HOOK);
        }
    }

    private static function schedule_cron_only($delay = 10) {
        $delay = max(5, min(300, (int)$delay));
        if (!wp_next_scheduled(self::CRON_HOOK)) {
            wp_schedule_single_event(time() + $delay, self::CRON_HOOK);
        }

    }

    private static function get_loopback_basic_auth_header() {
        $user = defined('HSBC_WEBP_LOOPBACK_USER') ? (string) HSBC_WEBP_LOOPBACK_USER : '';
        $pass = defined('HSBC_WEBP_LOOPBACK_PASS') ? (string) HSBC_WEBP_LOOPBACK_PASS : '';
        $user = trim($user);
        if ($user === '') return '';
        return 'Basic ' . base64_encode($user . ':' . $pass);
    }

    private static function spawn_async_tick($blocking = false) {
        $job = self::get_job_state();
        if (empty($job['running']) || empty($job['tick_key'])) return false;

        // Stop was requested — don't spawn any more loopbacks.
        if (self::get_stop_request() !== '') return false;

        $key   = (string) $job['tick_key'];
        $stamp = (string) time();

        // Candidates in best-to-worst order (cache-safe first).
        $cands = [
            [
                'label'  => 'admin-post',
                'method' => 'POST',
                'url'    => admin_url('admin-post.php'),
                'body'   => ['action' => 'hsbc_webp_job_tick_async', 'tick_key' => $key, '_' => $stamp],
            ],
            [
                'label'  => 'rest',
                'method' => 'GET',
                'url'    => add_query_arg(['tick_key' => $key, '_' => $stamp], rest_url('hsbc-webp/v1/tick')),
                'body'   => null,
            ],
            [
                'label'  => 'public',
                'method' => 'GET',
                'url'    => add_query_arg(['hsbc_webp_tick' => '1', 'tick_key' => $key, '_' => $stamp], home_url('/')),
                'body'   => null,
            ],
        ];

        foreach ($cands as $c) {
            $args = [
                'timeout'    => $blocking ? 8 : 2,
                'blocking'   => $blocking ? true : false,
                'sslverify'  => false,
                'redirection'=> 2,
                'headers'    => [
                    'Cache-Control' => 'no-cache',
                    'Pragma'        => 'no-cache',
                    'User-Agent'    => 'HSBC-WebP-Only/0.3.0; ' . home_url('/'),
                ],
            ];

            $auth = self::get_loopback_basic_auth_header();
            if ($auth !== '') {
                $args['headers']['Authorization'] = $auth;
            }

            if (!empty($c['body']) && is_array($c['body'])) {
                $args['body'] = $c['body'];
            }

            $res = (strtoupper($c['method']) === 'POST')
                ? wp_remote_post($c['url'], $args)
                : wp_remote_get($c['url'], $args);

            if (is_wp_error($res)) {
                self::log_write('TICK SPAWN FAIL | ' . $c['label'] . ' | ' . $c['url'] . ' | ' . $res->get_error_message(), 'ERROR');
                continue;
            }

            // Non-blocking: we can't validate body; treat transport success as "sent".
            if (!$blocking) {
                self::log_write('TICK SPAWN SENT | ' . $c['label'] . ' | ' . $c['url'], 'INFO');
                return true;
            }

            $code = (int) wp_remote_retrieve_response_code($res);
            $body = (string) wp_remote_retrieve_body($res);
            $ok   = (strpos($body, self::TICK_OK_MARKER) !== false);

            if ($ok) {
                self::log_write('TICK SPAWN OK | ' . $c['label'] . ' | code=' . $code . ' | ' . $c['url'], 'INFO');
                return true;
            }

            $snip = trim(preg_replace('/\s+/', ' ', substr($body, 0, 160)));
            self::log_write('TICK SPAWN BAD | ' . $c['label'] . ' | code=' . $code . ' | body="' . $snip . '" | ' . $c['url'], 'ERROR');
        }

        // Track failures + stop after 3 so it doesn't look "stuck".
        // IMPORTANT: re-read job state so we don't resurrect a job that just finished inside the tick.
        $job_now = self::get_job_state();
        if (empty($job_now['running']) || empty($job_now['tick_key']) || !hash_equals((string)$job_now['tick_key'], (string)$key)) {
            return false;
        }

        $fail = isset($job_now['spawn_failures']) ? (int)$job_now['spawn_failures'] + 1 : 1;
        $job_now['spawn_failures'] = $fail;
        $job_now['last_error'] = 'Background runner loopback failed (no valid tick response).';
        self::set_job_state($job_now);

        self::log_write('TICK SPAWN FAIL #' . $fail . ' | ' . $job_now['last_error'], 'ERROR');
        if ($fail >= 3) self::stop_job($job_now['last_error']);
        return false;
    }

    private static function schedule_tick($delay = 0) {
        // First tick: blocking probe so we log the REAL HTTP result (code + body marker) and stop lying.
        $ok = self::spawn_async_tick(true);

        // Safety net: schedule a cron tick as well.
        self::schedule_cron_only($delay);
        return $ok;
    }

    private static function start_job($mode, $replace_everywhere) {
        $mode = (string)$mode;
        $replace_everywhere = !empty($replace_everywhere);

        // Reset cursors for repeatable background runs.
        if ($mode === 'repair') update_option(self::OPT_REPAIR_CURSOR, 0);
        if ($mode === 'sweep') update_option(self::OPT_SWEEP_CURSOR, '');
        if ($mode === 'revert') update_option(self::OPT_REVERT_CURSOR, 0);
        if ($mode === 'revert') {
            update_option(self::OPT_REVERT_CURSOR, 0);
            update_option(self::OPT_REVERT_ORIG_SWEEP_CURSOR, '');
            update_option(self::OPT_REVERT_WPCONTENT_SWEEP_CURSOR, '');

        }
        if ($mode === 'clean_originals') {
            update_option(self::OPT_CLEAN_ORIG_CURSOR, '');
            update_option(self::OPT_CLEAN_ORIG_RESET, '');
            update_option(self::OPT_CLEAN_ORIG_DIR_CURSOR, '');
            update_option(self::OPT_CLEAN_ORIG_FILE_CURSOR, '');
        }
        if ($mode === 'clean_webp')      update_option(self::OPT_CLEAN_WEBP_CURSOR, '');

        $tick_key = wp_generate_password(32, false, false);

        $job = [
            'running' => true,
            'mode' => $mode,
            'replace_everywhere' => $replace_everywhere ? 1 : 0,
            'tick_key' => $tick_key,
            'spawn_failures' => 0,
            'started_at' => time(),
            'last_run' => 0,
            'stats' => [],
            'last_error' => '',
        ];
        self::set_job_state($job);
        self::log_write('JOB START: ' . $mode . ' | replace_everywhere=' . ($replace_everywhere ? '1' : '0'));
    }

    private static function stop_job($reason = 'stopped') {
        $job = self::get_job_state();
        $job['running'] = false;
        $job['tick_key'] = '';
        $job['last_error'] = (string)$reason;
        $job['last_run'] = time();
        self::set_job_state($job);
        self::unschedule_ticks();
        self::log_write('JOB STOP: ' . $reason);
    }

    private static function get_batch_size() {
        $v = (int) get_option(self::OPT_BATCH_SIZE, 8);
        if ($v < 1) $v = 1;
        if ($v > 50) $v = 50;
        return $v;
    }

    private static function acquire_lock() {
        $key = 'hsbc_webp_job_lock';
        if (get_transient($key)) return false;
        set_transient($key, 1, 60);
        return true;
    }

    private static function release_lock() {
        delete_transient('hsbc_webp_job_lock');
    }

    // ---------------- Background runner (WP-Cron) ----------------

    private static function run_job_tick_once($source = 'async') {
        // Returns array: ['done' => bool, 'scheduled_next' => bool]
        if (!self::acquire_lock()) {
            return ['done' => false, 'scheduled_next' => false, 'locked' => true];
        }

        $needs_next = false;
        try {
            $job = self::get_job_state();
            if (empty($job['running'])) {
                return ['done' => true, 'scheduled_next' => false, 'idle' => true];
            }
            $stop_reason = self::get_stop_request();
            if ($stop_reason !== '') {
                self::clear_stop_request();
                self::stop_job($stop_reason);
                return ['done' => true, 'scheduled_next' => false, 'stopped' => true];
            }

            $mode = isset($job['mode']) ? (string)$job['mode'] : '';
            $replace = !empty($job['replace_everywhere']);
            $limit = self::get_batch_size();

            // WebP support is required for all non-revert modes.
            if ($mode !== 'revert' && !self::image_editor_can_webp()) {
                self::stop_job('Server cannot generate WebP (missing GD/Imagick WebP support).');
                return ['done' => true, 'scheduled_next' => false];
            }

            $result = ['done' => true, 'stats' => []];
            if ($mode === 'bulk') {
                $result = self::process_bulk_batch($limit, $replace, $job['tick_key']);
            } elseif ($mode === 'repair') {
                $result = self::process_repair_batch($limit, $replace);
            } elseif ($mode === 'sweep') {
                $result = self::process_sweep_batch($limit, $replace);
            } elseif ($mode === 'revert') {
                $result = self::process_revert_batch($limit, $replace);
            } elseif ($mode === 'clean_originals') {
                $result = self::process_cleanup_originals_batch($limit);
            } elseif ($mode === 'clean_webp') {
                $result = self::process_cleanup_webp_batch($limit);
            } elseif ($mode === 'clean_sweep_originals') {
                $result = self::process_cleanup_sweep_originals_batch($limit);
            } else {
                self::stop_job('Unknown job mode: ' . $mode);
                return ['done' => true, 'scheduled_next' => false];
            }
            $stop_reason = self::get_stop_request();
            if ($stop_reason !== '') {
                self::clear_stop_request();
                self::stop_job($stop_reason);
                return ['done' => true, 'scheduled_next' => false, 'stopped' => true];
            }

            $job_now = self::get_job_state();
            if (empty($job_now['running']) || empty($job_now['tick_key']) || !hash_equals((string)$job_now['tick_key'], (string)$job['tick_key'])) {
                // Stopped or replaced while this tick was running — do NOT write state back.
                return ['done' => true, 'scheduled_next' => false, 'stopped' => true];
            }

            $job_now['last_run'] = time();
            $job_now['stats'] = is_array($result['stats']) ? $result['stats'] : [];
            $job_now['last_error'] = '';
            self::set_job_state($job_now);

            if (!empty($result['done'])) {
                self::stop_job('done');
                return ['done' => true, 'scheduled_next' => false];
            }

            $needs_next = true;
            return ['done' => false, 'scheduled_next' => false];
        } catch (Throwable $e) {
            self::log_write('JOB ERROR: ' . $e->getMessage(), 'ERROR');
            self::stop_job('error: ' . $e->getMessage());
            return ['done' => true, 'scheduled_next' => false];
        } finally {
            self::release_lock();
        }
        // unreachable
    }

    public static function cron_job_tick() {
        $r = self::run_job_tick_once('cron');
        if (!empty($r['locked'])) {
            self::schedule_cron_only(30);
            return;
        }
        if (empty($r['done'])) {
            self::schedule_cron_only(10);
        }
    }

    // ---------------- REST tick endpoint (bypasses most caching/WAF setups) ----------------

    public static function register_rest_endpoints() {
        if (!function_exists('register_rest_route')) return;

        register_rest_route('hsbc-webp/v1', '/tick', [
            'methods'  => 'GET',
            'callback' => [__CLASS__, 'rest_tick'],
            'permission_callback' => [__CLASS__, 'rest_tick_permission'],
            'args' => [
                'tick_key' => [
                    'required' => true,
                    'type'     => 'string',
                ],
            ],
        ]);
    }

    public static function rest_tick($request) {
        if (function_exists('nocache_headers')) nocache_headers();

        $job = self::get_job_state();
        $key = (string) $request->get_param('tick_key');

        if (empty($job['running']) || empty($job['tick_key']) || !hash_equals((string)$job['tick_key'], (string)$key)) {
            self::log_write('TICK HIT BLOCKED | REST endpoint | bad tick_key', 'ERROR');
            return new WP_REST_Response('forbidden', 403);
        }

        self::log_write('TICK HIT | REST endpoint', 'INFO');

                // Run one batch
                $r = self::run_job_tick_once('rest');

                // If another tick is already working, DO NOT spawn more loopbacks (prevents tick-storm).
                if (!empty($r['locked'])) {
                    status_header(200);
                    header('Content-Type: text/plain; charset=utf-8');
                    echo esc_html(self::TICK_OK_MARKER);
                    exit;
                }

                // Chain next tick if still running
                $job2 = self::get_job_state();
                if (
                    !empty($job2['running']) &&
                    !empty($job2['tick_key']) &&
                    hash_equals((string)$job2['tick_key'], (string)$key)
                ) {
                    self::schedule_cron_only(20);
                    self::spawn_async_tick(false);
                }
                
        return new WP_REST_Response(self::TICK_OK_MARKER, 200);
    }
    public static function public_tick_endpoint() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- public tick uses tick_key auth
        if (empty($_REQUEST['hsbc_webp_tick'])) return;
        if (function_exists('nocache_headers')) nocache_headers();
        if (function_exists('wp_ob_end_flush_all')) wp_ob_end_flush_all();
        while (ob_get_level()) { @ob_end_clean(); }

        $job = self::get_job_state();
        $key = sanitize_text_field( wp_unslash( $_REQUEST['tick_key'] ?? '' ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- protected by per-job tick_key

        if (empty($job['running']) || empty($job['tick_key']) || !hash_equals((string)$job['tick_key'], (string)$key)) {
            self::log_write('TICK HIT BLOCKED | public endpoint | bad tick_key', 'ERROR');
            status_header(403);
            echo 'forbidden';
            exit;
        }

        self::log_write('TICK HIT | public endpoint', 'INFO');

        // Run one batch
        self::run_job_tick_once('public');

        // Chain next tick if still running
        $job2 = self::get_job_state();
        if (
            !empty($job2['running']) &&
            !empty($job2['tick_key']) &&
            hash_equals((string)$job2['tick_key'], (string)$key)
        ) {
            self::schedule_cron_only(20);
            self::spawn_async_tick(false);
        }

        echo esc_html(self::TICK_OK_MARKER);
        exit;
    }

    public static function ajax_job_tick_async() {
        // No-login endpoint used by the self-runner loopback. Secured by a random per-job tick_key.
        $job = self::get_job_state();
        $key = sanitize_text_field( wp_unslash( $_REQUEST['tick_key'] ?? '' ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- protected by per-job tick_key

        if (empty($job['running']) || empty($job['tick_key']) || !hash_equals((string)$job['tick_key'], (string)$key)) {
            self::log_write('TICK HIT BLOCKED | job not running or bad tick_key', 'ERROR');
            status_header(403);            
            echo esc_html( 'forbidden' );
            exit;
        }

        self::log_write('TICK HIT | async endpoint called', 'INFO');

                // Run one batch
                $r = self::run_job_tick_once('async');

                // If another tick is already working, DO NOT spawn more loopbacks (prevents tick-storm).
                if (!empty($r['locked'])) {
                    status_header(200);
                    header('Content-Type: text/plain; charset=utf-8');
                    echo esc_html(self::TICK_OK_MARKER);
                    exit;
                }

                // If still running, spawn the next tick AFTER releasing the lock.
                $job2 = self::get_job_state();
                if (
                    !empty($job2['running']) &&
                    !empty($job2['tick_key']) &&
                    hash_equals((string)$job2['tick_key'], (string)$key)
                ) {
                    self::schedule_cron_only(20);
                    self::spawn_async_tick(false);
                }

                status_header(200);
                header('Content-Type: text/plain; charset=utf-8');
                echo esc_html(self::TICK_OK_MARKER);
                exit;
            }

    // ---------------- AJAX: background controls ----------------

    public static function ajax_job_start() {
        if (!current_user_can('manage_options')) wp_send_json_error('No permission');
        check_ajax_referer('hsbc_webp_job_start');

        $mode = isset($_POST['mode']) ? sanitize_key($_POST['mode']) : '';
        if (!in_array($mode, ['bulk', 'repair', 'sweep', 'revert', 'clean_originals', 'clean_webp', 'clean_sweep_originals'], true)) {
            wp_send_json_error('Bad mode');
        }

        $job = self::get_job_state();
        if (!empty($job['running'])) {
            wp_send_json_error('A job is already running. Stop it first.');
        }

        $replace = !empty($_POST['replace_everywhere']);
        self::start_job($mode, $replace);
        self::schedule_tick(5);

        wp_send_json_success([
            'job' => self::get_job_state(),
        ]);
    }

    public static function ajax_job_stop() {
        if (!current_user_can('manage_options')) wp_send_json_error('No permission');
        check_ajax_referer('hsbc_webp_job_stop');

        // Don't race an in-flight tick: request stop and prevent future cron ticks ASAP.
        self::set_stop_request('stopped_by_user');
        self::unschedule_ticks();

        // If nothing is currently running a tick, stop immediately.
        if (self::acquire_lock()) {
            self::clear_stop_request();
            self::stop_job('stopped_by_user');
            self::release_lock();
        }

        wp_send_json_success([
            'job' => self::get_job_state(),
        ]);
    }

    public static function ajax_job_status() {
        if (!current_user_can('manage_options')) wp_send_json_error('No permission');
        check_ajax_referer('hsbc_webp_job_status');

        wp_send_json_success([
            'job' => self::get_job_state(),
            'next_tick' => wp_next_scheduled(self::CRON_HOOK),
            'batch_size' => self::get_batch_size(),
        ]);
    }

    public static function ajax_log_tail() {
        if (!current_user_can('manage_options')) wp_send_json_error('No permission');
        check_ajax_referer('hsbc_webp_log_tail');

        $lines = isset($_POST['lines']) ? (int)$_POST['lines'] : 250;
        $text = self::log_tail($lines);
        wp_send_json_success([
            'text' => $text,
        ]);
    }

    public static function ajax_log_reset() {
        if (!current_user_can('manage_options')) wp_send_json_error('No permission');
        check_ajax_referer('hsbc_webp_log_reset');

        self::log_reset_all();
        self::log_write('LOG RESET by user');
        wp_send_json_success(true);
    }

    // ---------------- Batch processors (shared by AJAX + background) ----------------

    private static function process_bulk_batch($limit, $replace_everywhere, $tick_key) {
        global $wpdb;

        $limit = max(1, min(50, (int)$limit));
        $replace_everywhere = !empty($replace_everywhere);
        $time_start = microtime(true);
        $time_cap   = 50.0; // keep under PHP 60s timeout

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- admin-only batch tool; prepared query; caching not useful
        $ids = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT p.ID
                 FROM {$wpdb->posts} p
                 LEFT JOIN {$wpdb->postmeta} m1 ON (m1.post_id = p.ID AND m1.meta_key = %s)
                 LEFT JOIN {$wpdb->postmeta} m2 ON (m2.post_id = p.ID AND m2.meta_key = %s)
                 WHERE p.post_type = 'attachment'
                   AND p.post_status = 'inherit'
                   AND (p.post_mime_type = 'image/jpeg' OR p.post_mime_type = 'image/png')
                   AND m1.post_id IS NULL
                   AND m2.post_id IS NULL
                 ORDER BY p.ID ASC
                 LIMIT %d",
                self::META_CONVERTED, self::META_FAILED, $limit
            )
        );

        $processed = 0;
        $converted = 0;
        $skipped   = 0;
        $failed    = 0;
        $url_updates_total = 0;

        foreach ($ids as $id) {

                    $job_now = self::get_job_state();
                    if (empty($job_now['running']) || empty($job_now['tick_key']) || !hash_equals((string)$job_now['tick_key'], (string)$tick_key)) { break; }

                    if ((microtime(true) - $time_start) >= ($time_cap - 2.0)) { break; }
                    $processed++;
            $old_url = wp_get_attachment_url($id);

            $mime = get_post_mime_type($id);
            if (!self::is_target_mime($mime)) { $skipped++; continue; }

            $file = get_attached_file($id);
            if (!$file || !file_exists($file)) {
                update_post_meta($id, self::META_FAILED, 'missing_file');
                $failed++;
                continue;
            }

            $prev_meta = wp_get_attachment_metadata($id);
            if (!is_array($prev_meta) || empty($prev_meta)) $prev_meta = [];
            update_post_meta($id, self::META_PREV_META, wp_json_encode($prev_meta));
            
            if ((microtime(true) - $time_start) >= ($time_cap - 2.0)) { break; }
            $job_now = self::get_job_state();
                        if (empty($job_now['running']) || empty($job_now['tick_key']) || !hash_equals((string)$job_now['tick_key'], (string)$tick_key)) { break; }
                        $meta2 = self::convert_on_generate_metadata($prev_meta, $id);
                        if (is_array($meta2) && !empty($meta2)) {
                            wp_update_attachment_metadata($id, $meta2);
                        }

            if (get_post_meta($id, self::META_CONVERTED, true)) {
                $converted++;

                if ($replace_everywhere) {
                    $new_meta = wp_get_attachment_metadata($id);
                    if (is_array($prev_meta) && !empty($prev_meta) && is_array($new_meta) && !empty($new_meta)) {
                        $map = self::build_url_map_from_meta($prev_meta, $new_meta);
                        foreach ($map as $ou => $nu) {
                            $url_updates_total += self::replace_urls_everywhere_variants($ou, $nu);
                        }
                    } else {
                        $new_url = wp_get_attachment_url($id);
                        if ($new_url && $old_url && $new_url !== $old_url) {
                            $url_updates_total += self::replace_urls_everywhere_variants($old_url, $new_url);
                        }
                    }
                }
            } else {
                $msg = is_wp_error($meta2) ? $meta2->get_error_message() : 'convert_failed';
                update_post_meta($id, self::META_FAILED, $msg);
                $failed++;
            }
        }

        $done = empty($ids);
        $stats = [
            'processed' => $processed,
            'converted' => $converted,
            'skipped' => $skipped,
            'failed' => $failed,
            'url_updates' => $url_updates_total,
        ];

        self::log_write('BATCH bulk | processed=' . $processed . ' converted=' . $converted . ' skipped=' . $skipped . ' failed=' . $failed . ' url_updates=' . $url_updates_total . ' done=' . ($done ? '1' : '0'));

        return ['done' => $done, 'stats' => $stats];
    }

    private static function process_repair_batch($limit, $replace_everywhere) {
        global $wpdb;

        $limit = max(1, min(50, (int)$limit));
        $replace_everywhere = !empty($replace_everywhere);

        $cursor = (int) get_option(self::OPT_REPAIR_CURSOR, 0);
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- admin-only batch tool; prepared query; caching not useful
        $ids = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT p.ID
                 FROM {$wpdb->posts} p
                 INNER JOIN {$wpdb->postmeta} mconv ON (mconv.post_id = p.ID AND mconv.meta_key = %s)
                 WHERE p.post_type = 'attachment'
                   AND p.post_status = 'inherit'
                   AND p.ID > %d
                 ORDER BY p.ID ASC
                 LIMIT %d",
                self::META_CONVERTED, $cursor, $limit
            )
        );

        if (empty($ids)) {
            update_option(self::OPT_REPAIR_CURSOR, 0);
            $stats = [
                'processed' => 0,
                'repaired' => 0,
                'moved_files' => 0,
                'url_updates' => 0,
            ];
            self::log_write('BATCH repair | done=1');
            return ['done' => true, 'stats' => $stats];
        }

        $processed = 0;
        $repaired = 0;
        $moved_files = 0;
        $url_updates_total = 0;

        foreach ($ids as $id) {
            $processed++;
            update_option(self::OPT_REPAIR_CURSOR, (int)$id);

            $did = false;

            $new_meta = wp_get_attachment_metadata($id);
            if (!is_array($new_meta) || empty($new_meta)) continue;

            $orig_map = null;
            $orig_map_json = get_post_meta($id, self::META_ORIG_MAP, true);
            if ($orig_map_json) {
                $tmp = json_decode($orig_map_json, true);
                if (is_array($tmp)) $orig_map = $tmp;
            }

            $prev_meta = null;
            $prev_meta_json = get_post_meta($id, self::META_PREV_META, true);
            if ($prev_meta_json) {
                $tmp = json_decode($prev_meta_json, true);
                if (is_array($tmp)) $prev_meta = $tmp;
            }

            if ($replace_everywhere) {
                if (is_array($prev_meta) && !empty($prev_meta)) {
                    $map = self::build_url_map_from_meta($prev_meta, $new_meta);
                    foreach ($map as $ou => $nu) {
                        $u = self::replace_urls_everywhere_variants($ou, $nu);
                        if ($u) { $url_updates_total += $u; $did = true; }
                    }
                } elseif (is_array($orig_map)) {
                    $map = self::build_url_map_from_orig_map($orig_map, $new_meta);
                    foreach ($map as $ou => $nu) {
                        $u = self::replace_urls_everywhere_variants($ou, $nu);
                        if ($u) { $url_updates_total += $u; $did = true; }
                    }
                }
            }

            if (is_array($orig_map)) {
                $m = self::move_old_files_from_orig_map_to_originals($orig_map);
                if ($m) { $moved_files += $m; $did = true; }
            }
            if (is_array($prev_meta) && !empty($prev_meta)) {
                $m = self::move_old_files_from_meta_to_originals($prev_meta);
                if ($m) { $moved_files += $m; $did = true; }
            }

            [$m2, $u2] = self::move_original_image_leftovers($new_meta, $replace_everywhere);
            if ($m2) { $moved_files += $m2; $did = true; }
            if ($u2) { $url_updates_total += $u2; $did = true; }

            if ($did) $repaired++;
        }

        $stats = [
            'processed' => $processed,
            'repaired' => $repaired,
            'moved_files' => $moved_files,
            'url_updates' => $url_updates_total,
        ];

        self::log_write('BATCH repair | processed=' . $processed . ' repaired=' . $repaired . ' moved_files=' . $moved_files . ' url_updates=' . $url_updates_total . ' done=0');
        return ['done' => false, 'stats' => $stats];
    }

    private static function process_sweep_batch($limit, $replace_everywhere) {
    $limit = max(1, min(50, (int)$limit));
    $replace_everywhere = !empty($replace_everywhere);

    $basedir = self::get_upload_basedir();
    $baseurl = self::get_upload_baseurl();

    // Cursor supports both legacy string and new array state.
    $cursor_state = get_option(self::OPT_SWEEP_CURSOR, '');
    $cursor_i = 0;
    $cursor_p = '';
    if (is_array($cursor_state)) {
        $cursor_i = (int) ($cursor_state['i'] ?? 0);
        $cursor_p = (string) ($cursor_state['p'] ?? '');
    } else {
        $cursor_p = (string) $cursor_state;
    }

    $processed = 0;
    $converted = 0;
    $moved_files = 0;
    $url_updates = 0;

    $quality = (int) get_option(self::OPT_QUALITY, 82);
    $reuse_archived = !empty(get_option(self::OPT_REUSE_ARCHIVED, 1));

    // Include/exclude folders are relative to wp-content/ (one per line).
    $include_raw = (string) get_option(self::OPT_SWEEP_INCLUDE, '');
    $exclude_raw = (string) get_option(self::OPT_SWEEP_EXCLUDE, '');

    $include = array_filter(array_map('trim', preg_split('/\n+/', str_replace(["\r"], ["\n"], $include_raw))));
    $exclude = array_filter(array_map('trim', preg_split('/\n+/', str_replace(["\r"], ["\n"], $exclude_raw))));

    // Normalize to prefix form (no leading slash, no trailing slash; compare as "<prefix>/")
    $norm_prefixes = function($arr) {
        $out = [];
        foreach ($arr as $p) {
            $p = trim($p);
            if ($p === '') continue;
            $p = ltrim($p, "/\t ");
            if (strpos($p, '..') !== false) continue;
            $p = rtrim($p, '/');
            if ($p === '') continue;
            $out[] = $p . '/';
        }
        return array_values(array_unique($out));
    };

    $include_pfx = $norm_prefixes($include);
    $exclude_pfx = $norm_prefixes($exclude);

    // Build roots:
    // - If include is empty: default to uploads/
    // - Else: each include line is a root folder under wp-content/
    $roots = [];
    if (empty($include_pfx)) {
        $roots[] = [
            'id' => 'uploads',
            'type' => 'uploads',
            'abs' => $basedir,
            'wp_root' => 'uploads/',
        ];
    } else {
        foreach ($include_pfx as $pfx) {
            $root_rel = rtrim($pfx, '/'); // without trailing slash
            $abs_root = trailingslashit(WP_CONTENT_DIR) . $root_rel;
            if (!is_dir($abs_root)) continue;
            // If this root is inside uploads, treat it as uploads-type (so originals/webp archives behave like normal)
            $type = (strpos(trailingslashit($abs_root), trailingslashit($basedir)) === 0) ? 'uploads' : 'wpcontent';
            $roots[] = [
                'id' => $root_rel,
                'type' => $type,
                'abs' => $abs_root,
                'wp_root' => $root_rel . '/',
            ];
        }
        if (empty($roots)) {
            // Fallback: nothing valid specified, scan uploads.
            $roots[] = [
                'id' => 'uploads',
                'type' => 'uploads',
                'abs' => $basedir,
                'wp_root' => 'uploads/',
            ];
            $cursor_i = 0;
            $cursor_p = '';
        }
    }

    $matches_prefix = function($path, $prefixes) {
        foreach ($prefixes as $pfx) {
            if (strpos($path, $pfx) === 0) return true;
        }
        return false;
    };

    $root_count = count($roots);

    $cursor_missed = false;

    for ($ri = $cursor_i; $ri < $root_count; $ri++) {
        $root = $roots[$ri];
        $abs_root = $root['abs'];

        $it = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($abs_root, FilesystemIterator::SKIP_DOTS)
        );

        // Cursor: only used for uploads sweeps (uploads can be huge). For wp-content roots (e.g. gallery/), we rescan from the start each tick.
        $started = true;
        $cursor_found = true;
        if ($root['type'] === 'uploads' && $ri === (int)$cursor_i && $cursor_p !== '') {
            $started = false;
            $cursor_found = false;
        }
        foreach ($it as $fileinfo) {
            if ($processed >= $limit) break 2;
            if (!$fileinfo->isFile()) continue;

            $abs = $fileinfo->getPathname();

            // Build wp-content relative path for cursor/exclude comparisons.
            // Example: "uploads/2016/07/a.jpg" or "gallery/album/a.jpg"
            if ($root['type'] === 'uploads') {
                $rel_u = self::relpath_from_basedir($abs);
                if ($rel_u === '') continue;
                $wp_rel = 'uploads/' . ltrim($rel_u, '/');
            } else {
                $rel_in_root = str_replace('\\', '/', substr($abs, strlen(trailingslashit($abs_root))));
                $rel_in_root = ltrim($rel_in_root, '/');
                if ($rel_in_root === '') continue;
                $wp_rel = $root['wp_root'] . $rel_in_root;
            }

            // Safety skips
            if (strpos($wp_rel, 'uploads/_originals/') === 0) continue;
            if (strpos($wp_rel, 'uploads/_webp/') === 0) continue;
            if (strpos($wp_rel, '/_originals/') !== false || strpos($wp_rel, '/_webp/') !== false) continue;

            // Exclude always wins
            if (!empty($exclude_pfx) && $matches_prefix($wp_rel, $exclude_pfx)) continue;
            // Cursor resume: if we have a cursor for this root, skip everything until we encounter it exactly, then start after it.
            if ($root['type'] === 'uploads' && $ri === (int)$cursor_i && !$started) {
                if ($wp_rel === $cursor_p) {
                    $started = true;
                    $cursor_found = true;
                    continue;
                }
                continue;
            }
if (!preg_match('/\.(jpe?g|png)$/i', $wp_rel)) continue;
            if (preg_match('/\.bak\.(jpe?g|png)$/i', $wp_rel)) continue;

            $processed++;
            // Convert (prefer existing webp if present; for uploads also prefer archived webp in uploads/_webp/)
            $abs_webp = self::change_ext_to_webp($abs);

            if (!file_exists($abs_webp)) {
                $did_copy = false;

                if ($reuse_archived && $root['type'] === 'uploads') {
                    $rel_webp = self::relpath_from_basedir($abs_webp);
                    if ($rel_webp) {
                        $arch_webp = self::webp_archive_abs_path_for_rel($rel_webp);
                        if ($arch_webp && file_exists($arch_webp)) {
                            $did_copy = @copy($arch_webp, $abs_webp);
                        }
                    }
                }

                if (!$did_copy) {
                    $res = self::convert_file_to_webp($abs, $abs_webp, $quality);
                    if (!is_wp_error($res)) {
                        $converted++;
                    } else {
                        if ($root['type'] === 'uploads') update_option(self::OPT_SWEEP_CURSOR, ['i' => $ri, 'p' => $wp_rel]);
                        continue;
                    }
                } else {
                    $converted++;
                }
            }
            // Advance cursor (uploads sweeps only): use a stable marker (the .webp path survives after moving originals).
            if ($root['type'] === 'uploads') {
                $cursor_marker = self::change_ext_to_webp($wp_rel);
                if (file_exists($abs_webp)) {
                    update_option(self::OPT_SWEEP_CURSOR, ['i' => $ri, 'p' => $cursor_marker]);
                } else {
                    update_option(self::OPT_SWEEP_CURSOR, ['i' => $ri, 'p' => $wp_rel]);
                }
            }


            // Move original away
            if ($root['type'] === 'uploads') {
                [$ok,] = self::move_to_originals($abs);
                if ($ok) $moved_files++;
            } else {
                // For non-uploads roots, archive originals under "<root>/_originals/<rel>"
                $rel_in_root = str_replace('\\', '/', substr($abs, strlen(trailingslashit($abs_root))));
                $rel_in_root = ltrim($rel_in_root, '/');
                $dst = trailingslashit($abs_root) . '_originals/' . $rel_in_root;
                wp_mkdir_p(dirname($dst));
                if (!file_exists($dst)) {
                    if (self::fs_move($abs, $dst) || (@copy($abs, $dst) && self::fs_delete($abs))) {
                        $moved_files++;
                    }
                }
            }

            // Replace URLs only for uploads (wp has a baseurl for uploads)
            if ($root['type'] === 'uploads' && $replace_everywhere) {
                $rel_u = self::relpath_from_basedir($abs);
                if ($rel_u) {
                    $old_url = $baseurl . '/' . ltrim($rel_u, '/');
                    $new_rel = self::change_ext_to_webp($rel_u);
                    $new_url = $baseurl . '/' . ltrim($new_rel, '/');
                    $url_updates += self::replace_urls_everywhere_variants($old_url, $new_url);
                }
            }
        }

        // If we expected to find a cursor path in this root but didn't, restart from the beginning next tick.
        if ($root['type'] === 'uploads' && $ri === (int)$cursor_i && $cursor_p !== '' && !$cursor_found) {
            $cursor_missed = true;
        }

        // Finished this root; reset cursor_p for the next root
        $cursor_p = '';
    if ($cursor_missed) {
        // Cursor file no longer exists (it may have been moved/deleted). Reset and continue in the next tick.
        update_option(self::OPT_SWEEP_CURSOR, '');
        $stats = [
            'processed' => $processed,
            'converted' => $converted,
            'moved_files' => $moved_files,
            'url_updates' => $url_updates,
        ];
        return ['done' => false, 'stats' => $stats];
    }


    }

    $done = ($processed < $limit);

    if ($done) {
        // Reset cursor when no more eligible files remain
        update_option(self::OPT_SWEEP_CURSOR, '');
    }

    $stats = [
        'processed' => $processed,
        'converted' => $converted,
        'moved_files' => $moved_files,
        'url_updates' => $url_updates,
    ];

    return ['done' => $done, 'stats' => $stats];
}
private static function hsbc_is_derived_image_name($filename) {
    // WP derivatives we want to treat as “safe to remove from _originals”
    // Examples: foo-300x300.jpg, foo-scaled.jpg, foo-rotated.png
    return (bool) preg_match('/-(\d+x\d+|scaled|rotated)\.(jpe?g|png)$/i', $filename);
}

private static function hsbc_originals_has_master_for($basedir, $rel) {
    // For a derived rel path like 2022/02/foo-300x300.jpg,
    // check if _originals/2022/02/foo.jpg|jpeg|png exists.
    $orig_root = trailingslashit($basedir) . '_originals';

    $rel = ltrim(str_replace('\\', '/', $rel), '/');
    $dir = str_replace('\\', '/', dirname($rel));
    if ($dir === '.' || $dir === DIRECTORY_SEPARATOR) $dir = '';

    $pi = pathinfo($rel);
    $name = isset($pi['filename']) ? $pi['filename'] : '';
    if ($name === '') return false;

    // Strip derivative suffix
    $stem = preg_replace('/-(\d+x\d+|scaled|rotated)$/i', '', $name);

    $folder = ($dir === '') ? $orig_root : (trailingslashit($orig_root) . $dir);

    foreach (['jpg','jpeg','png'] as $ext) {
        $candidate = trailingslashit($folder) . $stem . '.' . $ext;
        if (is_file($candidate)) return true;
    }
    return false;
}
private static function process_cleanup_originals_batch($limit) {
    $limit = max(1, min(500, (int)$limit)); // allow larger batches for fast disks
    $u = wp_upload_dir();
    $basedir = isset($u['basedir']) ? (string)$u['basedir'] : '';
    if ($basedir === '') return ['done' => true, 'stats' => ['error' => 'No uploads basedir']];

    $orig_root = trailingslashit($basedir) . '_originals';
    if (!is_dir($orig_root)) return ['done' => true, 'stats' => ['deleted' => 0, 'scanned' => 0, 'note' => 'No _originals folder']];

    $dir_cursor  = (string) get_option(self::OPT_CLEAN_ORIG_DIR_CURSOR, '');
    $file_cursor = (string) get_option(self::OPT_CLEAN_ORIG_FILE_CURSOR, '');

    $deleted = 0;
    $scanned = 0;
    $last = '';
    $time_start = microtime(true);
    $time_cap   = 50.0; // seconds per tick (must be under PHP max_execution_time)  

    $dirs = self::hsbc_list_dirs_sorted($orig_root);

    // Pick current directory to process
    $dir_index = 0;
    if ($dir_cursor !== '') {
        $found = array_search($dir_cursor, $dirs, true);
        $dir_index = ($found === false) ? 0 : (int)$found;
    }
    $dir_rel = $dirs[$dir_index] ?? '';
    $dir_abs = ($dir_rel === '') ? $orig_root : (trailingslashit($orig_root) . $dir_rel);

    // Gather files in this directory (sorted)
    $files = [];
    if (is_dir($dir_abs)) {
        $fit = new FilesystemIterator($dir_abs, FilesystemIterator::SKIP_DOTS);
        foreach ($fit as $fi) {
            if ($fi->isFile()) $files[] = $fi->getFilename();
        }
    }
    sort($files, SORT_STRING);

    // Start after file cursor
    $start_i = 0;
    if ($file_cursor !== '') {
        $pos = array_search($file_cursor, $files, true);
        $start_i = ($pos === false) ? 0 : ((int)$pos + 1);
    }

    // Process up to $limit files in THIS folder only
    for ($i = $start_i; $i < count($files) && $scanned < $limit; $i++) {
        $fname = $files[$i];
        $abs = trailingslashit($dir_abs) . $fname;

        $rel = ($dir_rel === '') ? $fname : ($dir_rel . '/' . $fname);
        $base = $fname;

        // --- YOUR existing delete rules (keep exactly as you have them) ---
        if (self::hsbc_is_derived_image_name($base)) {
            $has_master_in_originals = self::hsbc_originals_has_master_for($basedir, $rel);
            $has_master_in_uploads   = self::hsbc_uploads_has_master_for($basedir, $rel);

            if ($has_master_in_originals || $has_master_in_uploads) {
                self::fs_delete($abs);
                $deleted++;
            }
        } else {
            $uploads_abs = trailingslashit($basedir) . $rel;
            if (is_file($uploads_abs)) {
                self::fs_delete($abs);
                $deleted++;
            }
        }
        // ---------------------------------------------------------------

        $scanned++;
        update_option(self::OPT_CLEAN_ORIG_FILE_CURSOR, $fname);

        if ((microtime(true) - $time_start) >= $time_cap) {
            break;
        }
    }

    // If we still have more files in this dir, keep going next tick
    if ($scanned >= $limit && (count($files) > 0) && ((int)$start_i + $scanned) < count($files)) {
        update_option(self::OPT_CLEAN_ORIG_DIR_CURSOR, $dir_rel);
        $stats = ['deleted' => $deleted, 'scanned' => $scanned, 'dir' => $dir_rel];
        self::log_write('CLEAN ORIGINALS (DIR): dir=' . $dir_rel . ' scanned=' . $scanned . ' deleted=' . $deleted . ' done=0');
        return ['done' => false, 'stats' => $stats];
    }

    // Finished this dir: advance to next dir
    update_option(self::OPT_CLEAN_ORIG_FILE_CURSOR, '');
    $next_index = $dir_index + 1;

    if ($next_index < count($dirs)) {
        update_option(self::OPT_CLEAN_ORIG_DIR_CURSOR, $dirs[$next_index]);
        $stats = ['deleted' => $deleted, 'scanned' => $scanned, 'dir' => $dir_rel, 'note' => 'next_dir'];
        self::log_write('CLEAN ORIGINALS (DIR): dir=' . $dir_rel . ' scanned=' . $scanned . ' deleted=' . $deleted . ' next=' . $dirs[$next_index]);
        return ['done' => false, 'stats' => $stats];
    }

    // All dirs done
    update_option(self::OPT_CLEAN_ORIG_DIR_CURSOR, '');
    update_option(self::OPT_CLEAN_ORIG_FILE_CURSOR, '');
    update_option(self::OPT_CLEAN_ORIG_CURSOR, '');
    update_option(self::OPT_CLEAN_ORIG_RESET, '1');

    $stats = ['deleted' => $deleted, 'scanned' => $scanned, 'note' => 'done'];
    self::log_write('CLEAN ORIGINALS (DIR): done scanned=' . $scanned . ' deleted=' . $deleted, 'INFO');
    return ['done' => true, 'stats' => $stats];

    // If we scanned anything, advance cursor and clear reset flag.
    if ($last !== '') {
        update_option(self::OPT_CLEAN_ORIG_CURSOR, $last);
        update_option(self::OPT_CLEAN_ORIG_RESET, '');
        $done = ($scanned < $limit);
        $stats = ['deleted' => $deleted, 'scanned' => $scanned, 'cursor' => $last];
        self::log_write('CLEAN ORIGINALS: root=' . $orig_root . ' scanned=' . $scanned . ' deleted=' . $deleted . ' cursor=' . $last);
        return ['done' => $done, 'stats' => $stats];
    }

    // scanned == 0: either finished, or cursor is ahead. Reset ONCE only.
    $did_reset = (string) get_option(self::OPT_CLEAN_ORIG_RESET, '');
    if ($cursor !== '' && $did_reset !== '1') {
        update_option(self::OPT_CLEAN_ORIG_CURSOR, '');
        update_option(self::OPT_CLEAN_ORIG_RESET, '1');
        $stats = ['deleted' => 0, 'scanned' => 0, 'cursor' => '-', 'note' => 'cursor_reset'];
        self::log_write('CLEAN ORIGINALS: scanned=0 (cursor reset) root=' . $orig_root, 'INFO');
        return ['done' => false, 'stats' => $stats];
    }

    // Already reset once (or cursor was empty): stop cleanly.
    update_option(self::OPT_CLEAN_ORIG_CURSOR, '');
    update_option(self::OPT_CLEAN_ORIG_RESET, '1');
    $stats = ['deleted' => 0, 'scanned' => 0, 'cursor' => '-', 'note' => 'done'];
    self::log_write('CLEAN ORIGINALS: root=' . $orig_root . ' scanned=0 deleted=0 cursor=- (done)', 'INFO');
    return ['done' => true, 'stats' => $stats];

}
private static function hsbc_list_dirs_sorted($root_abs) {
    $dirs = ['']; // include root itself as ""
    $it = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($root_abs, FilesystemIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );
    foreach ($it as $fi) {
        if (!$fi->isDir()) continue;
        $abs = $fi->getPathname();
        $rel = str_replace('\\', '/', substr($abs, strlen($root_abs)));
        $rel = trim($rel, '/');
        if ($rel !== '') $dirs[] = $rel;
    }
    $dirs = array_values(array_unique($dirs));
    sort($dirs, SORT_STRING);
    return $dirs;
}
private static function hsbc_uploads_has_master_for($basedir, $rel) {
    $rel = ltrim(str_replace('\\', '/', $rel), '/');
    $dir = str_replace('\\', '/', dirname($rel));
    if ($dir === '.' || $dir === DIRECTORY_SEPARATOR) $dir = '';

    $pi = pathinfo($rel);
    $name = $pi['filename'] ?? '';
    if ($name === '') return false;

    // Strip size then scaled/rotated (handles foo-scaled-1024x768)
    $stem1 = preg_replace('/-\d+x\d+$/', '', $name);
    $stem2 = preg_replace('/-(scaled|rotated)$/i', '', $stem1);
    $stems = array_values(array_unique([$stem1, $stem2]));

    $folder = ($dir === '') ? rtrim($basedir, '/\\') : rtrim($basedir, '/\\') . '/' . $dir;

    foreach ($stems as $stem) {
        foreach (['jpg','jpeg','png'] as $ext) {
            $candidate = $folder . '/' . $stem . '.' . $ext;
            if (is_file($candidate)) return true;
        }
    }
    return false;
}

private static function process_cleanup_webp_batch($limit) {
    $limit = max(1, min(500, (int)$limit));

    $u = wp_upload_dir();
    $basedir = isset($u['basedir']) ? (string)$u['basedir'] : '';
    if ($basedir === '') return ['done' => true, 'stats' => ['error' => 'No uploads basedir']];

    // Roots: uploads/_webp + any "<wp-content>/<sweep-root>/_webp"
    $roots = [];

    $uploads_webp = trailingslashit($basedir) . '_webp';
    if (is_dir($uploads_webp)) {
        $roots[] = ['id' => 'uploads', 'dir' => $uploads_webp];
    }

    $include_raw = (string)get_option(self::OPT_SWEEP_INCLUDE, '');
    $include = array_filter(array_map('trim', preg_split('/\n+/', str_replace(["\r"], ["\n"], $include_raw))));

    $norm_prefixes = function($arr) {
        $out = [];
        foreach ($arr as $p) {
            $p = trim($p);
            if ($p === '') continue;
            $p = ltrim($p, "/\t ");
            if (strpos($p, '..') !== false) continue;
            $p = rtrim($p, '/');
            if ($p === '') continue;
            $out[] = $p;
        }
        return array_values(array_unique($out));
    };

    $include_roots = $norm_prefixes($include);

    foreach ($include_roots as $root_rel) {
        $abs_root = trailingslashit(WP_CONTENT_DIR) . $root_rel;
        if (!is_dir($abs_root)) continue;

        // Skip anything inside uploads
        if (strpos(trailingslashit($abs_root), trailingslashit($basedir)) === 0) continue;

        $root_webp = trailingslashit($abs_root) . '_webp';
        if (!is_dir($root_webp)) continue;

        $roots[] = ['id' => $root_rel, 'dir' => $root_webp];
    }

    if (empty($roots)) {
        return ['done' => true, 'stats' => ['deleted' => 0, 'scanned' => 0, 'note' => 'No _webp folders']];
    }

    $deleted = 0;
    $scanned = 0;
    $last = '';

    // IMPORTANT: behave like uploads-side nuke — NO cursor skipping/resume logic.
    foreach ($roots as $r) {
        $webp_root = trailingslashit($r['dir']);

        $it = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($webp_root, FilesystemIterator::SKIP_DOTS),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        foreach ($it as $fi) {
            if (!$fi->isFile()) continue;
            if ($scanned >= $limit) break 2;

            $abs = $fi->getPathname();
            $rel = str_replace('\\', '/', substr($abs, strlen($webp_root)));
            $rel = ltrim($rel, '/');
            if ($rel === '') continue;

            if (strtolower(substr($rel, -5)) !== '.webp') continue;

            self::fs_delete($abs);
            $deleted++;
            $scanned++;

            $last = $r['id'] . '|'.$rel;
        }
    }

    if ($last !== '') {
        update_option(self::OPT_CLEAN_WEBP_CURSOR, $last);
        update_option(self::OPT_CLEAN_WEBP_RESET, '');
        $done = ($scanned < $limit);
        $stats = ['deleted' => $deleted, 'scanned' => $scanned, 'cursor' => $last];
        self::log_write('CLEAN WEBP NUKE: roots=' . count($roots) . ' scanned=' . $scanned . ' deleted=' . $deleted . ' cursor=' . $last, 'INFO');
        return ['done' => $done, 'stats' => $stats];
    }

    // No files left anywhere
    update_option(self::OPT_CLEAN_WEBP_CURSOR, '');
    update_option(self::OPT_CLEAN_WEBP_RESET, '');
    self::log_write('CLEAN WEBP NUKE: done', 'INFO');

    // Optional: remove empty directories inside each _webp
    if (method_exists(__CLASS__, 'prune_empty_dirs')) {
        foreach ($roots as $r) {
            $dir = $r['dir'];
            if (is_dir($dir)) self::prune_empty_dirs($dir);
        }
    }

    return ['done' => true, 'stats' => ['deleted' => 0, 'scanned' => 0, 'note' => 'done']];
}
private static function process_cleanup_sweep_originals_batch($limit) {
    $limit = max(1, min(200, (int)$limit)); // deletions per tick

    $basedir = self::get_upload_basedir();

    // Build roots from Sweep Include (relative to wp-content/), but ONLY non-uploads roots (e.g. gallery/)
    $include_raw = (string)get_option(self::OPT_SWEEP_INCLUDE, '');
    $include = array_filter(array_map('trim', preg_split('/\n+/', str_replace(["\r"], ["\n"], $include_raw))));

    $norm_prefixes = function($arr) {
        $out = [];
        foreach ($arr as $p) {
            $p = trim($p);
            if ($p === '') continue;
            $p = ltrim($p, "/\t ");
            if (strpos($p, '..') !== false) continue;
            $p = rtrim($p, '/');
            if ($p === '') continue;
            $out[] = $p;
        }
        return array_values(array_unique($out));
    };

    $include_roots = $norm_prefixes($include);

    $roots = [];
    foreach ($include_roots as $root_rel) {
        $abs_root = trailingslashit(WP_CONTENT_DIR) . $root_rel;
        if (!is_dir($abs_root)) continue;

        // Skip anything inside uploads (uploads has its own cleanup button)
        if ($basedir && strpos(trailingslashit($abs_root), trailingslashit($basedir)) === 0) continue;

        $orig_root = trailingslashit($abs_root) . '_originals';
        if (!is_dir($orig_root)) continue;

        $roots[] = [
            'id'   => $root_rel,
            'abs'  => $abs_root,
            'orig' => $orig_root,
        ];
    }

    // Always clear any old cursor state (this routine is intentionally cursorless)
    update_option('hsbc_webp_clean_sweep_originals_cursor', '');

    if (empty($roots)) {
        self::log_write('CLEAN SWEEP ORIGINALS: no non-uploads roots with _originals found (nothing to do).', 'INFO');
        return ['done' => true, 'stats' => ['deleted' => 0, 'checked' => 0, 'skipped' => 0, 'mismatch' => 0]];
    }

    $deleted  = 0;
    $checked  = 0;
    $skipped  = 0;
    $mismatch = 0;
    $last_path = '';
    $root_id = '';

    foreach ($roots as $root) {
        if ($deleted >= $limit) break;

        $root_id = $root['id'];

        $orig_base = trailingslashit($root['orig']);
        $abs_root  = trailingslashit($root['abs']);

        $it = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($orig_base, FilesystemIterator::SKIP_DOTS),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        foreach ($it as $fileinfo) {
            if ($deleted >= $limit) break 2;
            if (!$fileinfo->isFile()) continue;

            $abs = $fileinfo->getPathname();

            // rel path inside _originals
            $rel = str_replace('\\', '/', substr($abs, strlen($orig_base)));
            $rel = ltrim($rel, '/');
            if ($rel === '') continue;

            $marker = $root['id'] . '/_originals/' . $rel;

            $checked++;
            $last_path = $marker;

            // Only delete when the same file exists in the root AND sizes match (safe duplicate)
            $candidate = $abs_root . $rel;
            if (!file_exists($candidate)) {
                $skipped++;
                continue;
            }

            $s1 = @filesize($abs);
            $s2 = @filesize($candidate);
            if ($s1 === false || $s2 === false || $s1 !== $s2) {
                $mismatch++;
                continue;
            }

            if (self::fs_delete($abs)) {
                $deleted++;
            } else {
                // treat delete failure as mismatch so it shows up in stats
                $mismatch++;
            }
        }
    }

    // Done only when we couldn't delete anything this tick (prevents “pockets left behind”)
    $done = ($deleted === 0);

    self::log_write("CLEAN SWEEP ORIGINALS: checked={$checked} deleted={$deleted} skipped={$skipped} mismatch={$mismatch}", 'INFO');

    return [
        'done' => $done,
        'stats' => [
            'checked'   => $checked,
            'deleted'   => $deleted,
            'skipped'   => $skipped,
            'mismatch'  => $mismatch,
            'last_path' => $last_path,
            'root'      => $root_id,
        ]
    ];
}


    private static function guess_mime_from_rel($rel) {
        $rel = (string)$rel;
        if (preg_match('/\.png$/i', $rel)) return 'image/png';
        return 'image/jpeg';
    }

    private static function process_revert_batch($limit, $replace_everywhere) {
        global $wpdb;

        $limit = max(1, min(50, (int)$limit));
        $replace_everywhere = !empty($replace_everywhere);

        $cursor = (int) get_option(self::OPT_REVERT_CURSOR, 0);

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- admin-only batch tool; prepared query; caching not useful
        $ids = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT p.ID
                 FROM {$wpdb->posts} p
                 INNER JOIN {$wpdb->postmeta} mconv ON (mconv.post_id = p.ID AND mconv.meta_key = %s)
                 WHERE p.post_type = 'attachment'
                   AND p.post_status = 'inherit'
                   AND p.ID > %d
                 ORDER BY p.ID ASC
                 LIMIT %d",
                self::META_CONVERTED, $cursor, $limit
            )
        );

        if (empty($ids)) {
            update_option(self::OPT_REVERT_CURSOR, 0);

            $sweep = self::process_revert_originals_sweep_batch(500);
            $wpc   = self::process_revert_wpcontent_sweep_batch(500);

            $stats = [
                'processed'     => 0,
                'reverted'      => 0,
                'moved_webp'    => (int)($wpc['stats']['moved_webp'] ?? 0),
                'restored'      => (int)($sweep['stats']['restored'] ?? 0) + (int)($wpc['stats']['restored'] ?? 0),
                'url_updates'   => 0,
                'sweep_scanned' => (int)($sweep['stats']['scanned'] ?? 0),
                'wpc_processed' => (int)($wpc['stats']['processed'] ?? 0),
            ];

            $done = (!empty($sweep['done']) && !empty($wpc['done']));
            self::log_write(
                'BATCH revert | uploads_sweep_done=' . (!empty($sweep['done']) ? '1' : '0') .
                ' wpc_done=' . (!empty($wpc['done']) ? '1' : '0'),
                'INFO'
            );

            return ['done' => $done, 'stats' => $stats];
        }

        $processed = 0;
        $reverted = 0;
        $moved_webp = 0;
        $restored = 0;
        $url_updates = 0;

        foreach ($ids as $id) {
            $processed++;
            update_option(self::OPT_REVERT_CURSOR, (int)$id);

            $prev_meta_json = get_post_meta($id, self::META_PREV_META, true);
            $prev_meta = $prev_meta_json ? json_decode($prev_meta_json, true) : null;
            if (!is_array($prev_meta) || empty($prev_meta['file'])) {
                self::log_write('REVERT skip attachment ' . $id . ' (missing prev metadata)', 'WARN');
                continue;
            }

            $new_meta = wp_get_attachment_metadata($id);
            if (!is_array($new_meta) || empty($new_meta['file'])) {
                self::log_write('REVERT skip attachment ' . $id . ' (missing current metadata)', 'WARN');
                continue;
            }

            // Build URL map new->old (reverse of stored prev->new)
            if ($replace_everywhere) {
                $map = self::build_url_map_from_meta($prev_meta, $new_meta);
                foreach ($map as $old_url => $new_url) {
                    $u = self::replace_urls_everywhere_variants($new_url, $old_url);
                    if ($u) $url_updates += $u;
                }
            }

            // Move current WebP files into uploads/_webp/
            $webp_rels = [];
            $webp_rels[] = $new_meta['file'];
            if (!empty($new_meta['sizes']) && is_array($new_meta['sizes'])) {
                $dir = dirname($new_meta['file']);
                foreach ($new_meta['sizes'] as $s) {
                    if (empty($s['file'])) continue;
                    $rel = ($dir && $dir !== '.') ? ($dir . '/' . $s['file']) : $s['file'];
                    $webp_rels[] = $rel;
                }
            }
            $webp_rels = array_values(array_unique(array_filter($webp_rels)));
            foreach ($webp_rels as $rel) {
                if (!preg_match('/\.webp$/i', $rel)) continue;
                [$ok,] = self::move_rel_to_webp_archive($rel);
                if ($ok) $moved_webp++;
            }

            // Restore originals back from uploads/_originals/
            $orig_rels = [];
            $orig_rels[] = $prev_meta['file'];
            if (!empty($prev_meta['sizes']) && is_array($prev_meta['sizes'])) {
                $dir = dirname($prev_meta['file']);
                foreach ($prev_meta['sizes'] as $s) {
                    if (empty($s['file'])) continue;
                    $rel = ($dir && $dir !== '.') ? ($dir . '/' . $s['file']) : $s['file'];
                    $orig_rels[] = $rel;
                }
            }
            $orig_rels = array_values(array_unique(array_filter($orig_rels)));
            foreach ($orig_rels as $rel) {
                if (!preg_match('/\.(jpe?g|png)$/i', $rel)) continue;
                [$ok,] = self::restore_rel_from_originals($rel);
                if ($ok) $restored++;
            }

            // Restore attachment to original
            $basedir = self::get_upload_basedir();
            $abs_full = $basedir . '/' . ltrim($prev_meta['file'], '/');
            update_attached_file($id, $abs_full);
            wp_update_attachment_metadata($id, $prev_meta);

            $mime = self::guess_mime_from_rel($prev_meta['file']);
            wp_update_post([
                'ID' => $id,
                'post_mime_type' => $mime,
            ]);

            delete_post_meta($id, self::META_CONVERTED);
            delete_post_meta($id, self::META_ORIG_MAP);
            delete_post_meta($id, self::META_FAILED);

            $reverted++;
            self::log_write('REVERT attachment ' . $id . ' OK | restored=' . count($orig_rels) . ' moved_webp=' . count($webp_rels));
        }

        $stats = [
            'processed' => $processed,
            'reverted' => $reverted,
            'moved_webp' => $moved_webp,
            'restored' => $restored,
            'url_updates' => $url_updates,
        ];

        self::log_write('BATCH revert | processed=' . $processed . ' reverted=' . $reverted . ' moved_webp=' . $moved_webp . ' restored=' . $restored . ' url_updates=' . $url_updates . ' done=0');
        return ['done' => false, 'stats' => $stats];
    }
    private static function process_revert_originals_sweep_batch($limit) {
        $limit = max(1, min(500, (int)$limit));

        $u = wp_upload_dir();
        $basedir = isset($u['basedir']) ? (string)$u['basedir'] : '';
        if ($basedir === '') return ['done' => true, 'stats' => ['restored' => 0, 'scanned' => 0, 'error' => 'No uploads basedir']];

        $orig_root = trailingslashit($basedir) . '_originals';
        if (!is_dir($orig_root)) return ['done' => true, 'stats' => ['restored' => 0, 'scanned' => 0, 'note' => 'No _originals folder']];

        $cursor = (string) get_option(self::OPT_REVERT_ORIG_SWEEP_CURSOR, '');

        $restored = 0;
        $scanned = 0;
        $last = '';

        $it = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($orig_root, FilesystemIterator::SKIP_DOTS),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        $started = ($cursor === '');

        foreach ($it as $fi) {
            if (!$fi->isFile()) continue;

            $abs = $fi->getPathname();
            $rel = str_replace('\\', '/', substr($abs, strlen($orig_root)));
            $rel = ltrim($rel, '/');

            if (!$started) {
                if ($rel === $cursor) { $started = true; continue; }
                continue;
            }

            // Only base/master images (no -300x300, no -scaled, no -rotated)
            if (!preg_match('/\.(jpe?g|png)$/i', $rel)) { $scanned++; $last = $rel; if ($scanned >= $limit) break; continue; }
            if (preg_match('/-(\d+x\d+|scaled|rotated)\.(jpe?g|png)$/i', basename($rel))) { $scanned++; $last = $rel; if ($scanned >= $limit) break; continue; }

            $dest = trailingslashit($basedir) . $rel;

            if (!is_file($dest)) {
                $dir = dirname($dest);
                if (!is_dir($dir)) wp_mkdir_p($dir);

                if (@copy($abs, $dest)) {
                    $restored++;
                } else {
                    self::log_write('REVERT SWEEP: FAILED copy ' . $abs . ' -> ' . $dest, 'ERROR');
                }
            }

            $scanned++;
            $last = $rel;
            if ($scanned >= $limit) break;
        }

        if ($last !== '') {
            update_option(self::OPT_REVERT_ORIG_SWEEP_CURSOR, $last);
            $done = ($scanned < $limit);
            self::log_write('REVERT SWEEP: scanned=' . $scanned . ' restored=' . $restored . ' cursor=' . $last, 'INFO');
            return ['done' => $done, 'stats' => ['scanned' => $scanned, 'restored' => $restored, 'cursor' => $last]];
        }

        update_option(self::OPT_REVERT_ORIG_SWEEP_CURSOR, '');
        self::log_write('REVERT SWEEP: done', 'INFO');
        return ['done' => true, 'stats' => ['scanned' => 0, 'restored' => 0, 'note' => 'done']];
    }
    private static function process_revert_wpcontent_sweep_batch($limit) {
        $limit = max(1, min(500, (int)$limit));

        // Same include/exclude system as process_sweep_batch(): wp-content relative, one per line
        $include_raw = (string) get_option(self::OPT_SWEEP_INCLUDE, '');
        $exclude_raw = (string) get_option(self::OPT_SWEEP_EXCLUDE, '');

        $include = array_filter(array_map('trim', preg_split('/\n+/', str_replace(["\r"], ["\n"], $include_raw))));
        $exclude = array_filter(array_map('trim', preg_split('/\n+/', str_replace(["\r"], ["\n"], $exclude_raw))));

        $norm_prefixes = function($arr) {
            $out = [];
            foreach ($arr as $p) {
                $p = trim($p);
                if ($p === '') continue;
                $p = ltrim($p, "/\t ");
                if (strpos($p, '..') !== false) continue;
                $p = rtrim($p, '/');
                if ($p === '') continue;
                $out[] = $p . '/';
            }
            return array_values(array_unique($out));
        };

        $include_pfx = $norm_prefixes($include);
        $exclude_pfx = $norm_prefixes($exclude);

        // If include is empty, sweep only targets uploads/ (no wp-content custom roots)
        if (empty($include_pfx)) {
            return ['done' => true, 'stats' => ['processed' => 0, 'restored' => 0, 'moved_webp' => 0, 'note' => 'no wp-content roots (include empty)']];
        }

        $basedir = self::get_upload_basedir();

        // Build wp-content roots the same way sweep does, but KEEP ONLY non-uploads roots
        $roots = [];
        foreach ($include_pfx as $pfx) {
            $root_rel = rtrim($pfx, '/'); // wp-content relative
            $abs_root = trailingslashit(WP_CONTENT_DIR) . $root_rel;
            if (!is_dir($abs_root)) continue;

            $type = (strpos(trailingslashit($abs_root), trailingslashit($basedir)) === 0) ? 'uploads' : 'wpcontent';
            if ($type !== 'wpcontent') continue;

            $roots[] = [
                'abs' => $abs_root,
                'wp_root' => $root_rel . '/',
            ];
        }

        if (empty($roots)) {
            return ['done' => true, 'stats' => ['processed' => 0, 'restored' => 0, 'moved_webp' => 0, 'note' => 'no non-uploads roots found']];
        }

        $matches_prefix = function($path, $prefixes) {
            foreach ($prefixes as $pfx) {
                if (strpos($path, $pfx) === 0) return true;
            }
            return false;
        };

        // Cursor supports both legacy string and new array state (same pattern as OPT_SWEEP_CURSOR)
        $cursor_state = get_option(self::OPT_REVERT_WPCONTENT_SWEEP_CURSOR, '');
        $cursor_i = 0;
        $cursor_p = '';
        if (is_array($cursor_state)) {
            $cursor_i = (int) ($cursor_state['i'] ?? 0);
            $cursor_p = (string) ($cursor_state['p'] ?? '');
        } else {
            $cursor_p = (string) $cursor_state;
        }

        $time_start = microtime(true);
        $time_cap   = 50.0; // keep under PHP 60s timeout

        $processed = 0;
        $restored = 0;
        $moved_webp = 0;

        $root_count = count($roots);

        for ($ri = $cursor_i; $ri < $root_count; $ri++) {
            $root = $roots[$ri];
            $abs_root = rtrim(str_replace('\\','/', $root['abs']), '/');
            $orig_root = $abs_root . '/_originals';

            if (!is_dir($orig_root)) {
                $cursor_p = '';
                continue;
            }

            $it = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($orig_root, FilesystemIterator::SKIP_DOTS),
                RecursiveIteratorIterator::LEAVES_ONLY
            );

            $started = true;
            if ($ri === (int)$cursor_i && $cursor_p !== '') {
                $started = false;
            }

            foreach ($it as $fi) {
                if (!$fi->isFile()) continue;

                $abs = str_replace('\\','/', $fi->getPathname());
                $rel_in_orig = str_replace('\\','/', substr($abs, strlen(trailingslashit($orig_root))));
                $rel_in_orig = ltrim($rel_in_orig, '/');
                if ($rel_in_orig === '') continue;

                // Build a stable marker that always exists (we iterate inside _originals)
                $marker = $root['wp_root'] . '_originals/' . $rel_in_orig;

                // Apply exclude rules (exclude always wins), same wp_rel style as sweep
                $wp_rel = $root['wp_root'] . $rel_in_orig;
                if (!empty($exclude_pfx) && $matches_prefix($wp_rel, $exclude_pfx)) continue;

                // Cursor resume
                if (!$started) {
                    if ($marker === $cursor_p) { $started = true; continue; }
                    continue;
                }

                // Only restore JPG/PNG from _originals
                if (!preg_match('/\.(jpe?g|png)$/i', $rel_in_orig)) {
                    continue;
                }

                $dest = $abs_root . '/' . $rel_in_orig;
                if (!is_dir(dirname($dest))) wp_mkdir_p(dirname($dest));

                // Restore original back into the root (don’t overwrite an existing file)
                if (!is_file($dest)) {
                    if (@copy($abs, $dest)) {
                        $restored++;
                    } else {
                        self::log_write('REVERT WPCONTENT: FAILED copy ' . $abs . ' -> ' . $dest, 'ERROR');
                    }
                }

                // Move corresponding .webp (if present) into <root>/_webp/<rel>.webp
                $webp_src = self::change_ext_to_webp($dest);
                if (is_file($webp_src)) {
                    $webp_rel = self::change_ext_to_webp($rel_in_orig);
                    $webp_dst = $abs_root . '/_webp/' . ltrim($webp_rel, '/');
                    if (!is_dir(dirname($webp_dst))) wp_mkdir_p(dirname($webp_dst));

                    // If archive already has it, delete the in-place webp so root is clean
                    if (is_file($webp_dst)) {
                        self::fs_delete($webp_src);
                        $moved_webp++;
                    } else {
                        if (self::fs_move($webp_src, $webp_dst) || (@copy($webp_src, $webp_dst) && self::fs_delete($webp_src))) {
                            $moved_webp++;
                        }
                    }
                }

                $processed++;

                // Advance cursor
                update_option(self::OPT_REVERT_WPCONTENT_SWEEP_CURSOR, ['i' => $ri, 'p' => $marker]);

                // Time cap
                if ((microtime(true) - $time_start) >= $time_cap) break 2;

                // Count cap
                if ($processed >= $limit) break 2;
            }

            $cursor_p = '';
        }

        // If we did work, we’re not done yet unless we processed < limit
        if ($processed > 0) {
            $done = ($processed < $limit);
            self::log_write('REVERT WPCONTENT: processed=' . $processed . ' restored=' . $restored . ' moved_webp=' . $moved_webp . ' done=' . ($done ? '1' : '0'), 'INFO');
            return ['done' => $done, 'stats' => ['processed' => $processed, 'restored' => $restored, 'moved_webp' => $moved_webp]];
        }

        // Nothing left to do
        update_option(self::OPT_REVERT_WPCONTENT_SWEEP_CURSOR, '');
        self::log_write('REVERT WPCONTENT: done', 'INFO');
        return ['done' => true, 'stats' => ['processed' => 0, 'restored' => 0, 'moved_webp' => 0, 'note' => 'done']];
    }
    // ---------------- URL replacement (serialized-safe) ----------------

    private static function deep_replace($old, $new, $data) {
        if (is_string($data)) return str_replace($old, $new, $data);
        if (is_array($data)) {
            foreach ($data as $k => $v) $data[$k] = self::deep_replace($old, $new, $v);
            return $data;
        }
        if (is_object($data)) {
            foreach ($data as $k => $v) $data->$k = self::deep_replace($old, $new, $v);
            return $data;
        }
        return $data;
    }

    private static function replace_urls_everywhere($old, $new) {
        global $wpdb;
        if (!$old || !$new || $old === $new) return 0;

        $updates = 0;
        $like = '%' . $wpdb->esc_like($old) . '%';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- admin-only maintenance; prepared query; caching not useful
        $updates += (int) $wpdb->query(
            $wpdb->prepare(
                "UPDATE {$wpdb->posts} SET post_content = REPLACE(post_content, %s, %s) WHERE post_content LIKE %s",
                $old, $new, $like
            )
        );

        // Cursor pagination by meta_id (avoids OFFSET skipping/repeats on large tables)
        $batch = 300;
        $last_id = 0;
        while (true) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.SlowDBQuery.slow_db_query_meta_value -- admin-only maintenance scan
            $rows = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT meta_id, meta_value FROM {$wpdb->postmeta} WHERE meta_value LIKE %s AND meta_id > %d ORDER BY meta_id ASC LIMIT %d",
                    $like, $last_id, $batch
                )
            );
            if (empty($rows)) break;

            foreach ($rows as $r) {
                $val = $r->meta_value;
                $new_val = $val;

                if (is_serialized($val)) {
                    $data = @unserialize($val);
                    if ($data !== false || $val === 'b:0;') {
                        $data = self::deep_replace($old, $new, $data);
                        $new_val = serialize($data);
                    }
                } else {
                    $new_val = str_replace($old, $new, $val);
                }

                if ($new_val !== $val) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.SlowDBQuery.slow_db_query_meta_value -- admin-only maintenance update; caching not useful
                    $wpdb->update(
                        $wpdb->postmeta,
                        ['meta_value' => $new_val], // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_value -- admin-only maintenance update
                        ['meta_id' => (int)$r->meta_id],
                        ['%s'],
                        ['%d']
                    );
                    $updates++;
                }

                $last_id = (int) $r->meta_id;
            }
        }

        return $updates;
    }

    private static function replace_urls_everywhere_variants($old_url, $new_url) {
        $total = 0;

        $total += self::replace_urls_everywhere($old_url, $new_url);

        $op = wp_parse_url($old_url);
        $np = wp_parse_url($new_url);

        if (!empty($op['host']) && !empty($op['path']) && !empty($np['host']) && !empty($np['path'])) {
            $old_proto_rel = '//' . $op['host'] . $op['path'];
            $new_proto_rel = '//' . $np['host'] . $np['path'];
            $total += self::replace_urls_everywhere($old_proto_rel, $new_proto_rel);

            // Path-only replacements (covers mixed scheme / CDN swaps)
            $total += self::replace_urls_everywhere($op['path'], $np['path']);
        }

        if (stripos($old_url, 'http://') === 0) {
            $total += self::replace_urls_everywhere('https://' . substr($old_url, 7), $new_url);
        } elseif (stripos($old_url, 'https://') === 0) {
            $total += self::replace_urls_everywhere('http://' . substr($old_url, 8), $new_url);
        }

        return $total;
    }

    // ---------------- Mapping helpers ----------------

    private static function build_url_map_from_meta($old_meta, $new_meta) {
        $baseurl = self::get_upload_baseurl();
        $map = [];

        if (!empty($old_meta['file']) && !empty($new_meta['file'])) {
            $map[$baseurl . '/' . ltrim($old_meta['file'], '/')] = $baseurl . '/' . ltrim($new_meta['file'], '/');
        }

        $old_sizes = (!empty($old_meta['sizes']) && is_array($old_meta['sizes'])) ? $old_meta['sizes'] : [];
        $new_sizes = (!empty($new_meta['sizes']) && is_array($new_meta['sizes'])) ? $new_meta['sizes'] : [];

        foreach ($old_sizes as $size_key => $o) {
            if (empty($o['file']) || empty($new_sizes[$size_key]['file'])) continue;

            $old_dir = !empty($old_meta['file']) ? dirname($old_meta['file']) : '';
            $new_dir = !empty($new_meta['file']) ? dirname($new_meta['file']) : '';

            $old_rel = ($old_dir && $old_dir !== '.') ? ($old_dir . '/' . $o['file']) : $o['file'];
            $new_rel = ($new_dir && $new_dir !== '.') ? ($new_dir . '/' . $new_sizes[$size_key]['file']) : $new_sizes[$size_key]['file'];

            $map[$baseurl . '/' . ltrim($old_rel, '/')] = $baseurl . '/' . ltrim($new_rel, '/');
        }

        return $map;
    }

    private static function build_url_map_from_orig_map($orig_map, $new_meta) {
        $baseurl = self::get_upload_baseurl();
        $map = [];

        if (!is_array($orig_map) || !is_array($new_meta)) return $map;

        if (!empty($orig_map['full']) && !empty($new_meta['file'])) {
            $map[$baseurl . '/' . ltrim($orig_map['full'], '/')] = $baseurl . '/' . ltrim($new_meta['file'], '/');
        }

        $new_dir = !empty($new_meta['file']) ? dirname($new_meta['file']) : '';
        $new_sizes = (!empty($new_meta['sizes']) && is_array($new_meta['sizes'])) ? $new_meta['sizes'] : [];

        if (!empty($orig_map['sizes']) && is_array($orig_map['sizes'])) {
            foreach ($orig_map['sizes'] as $size_key => $old_rel) {
                if (empty($old_rel) || empty($new_sizes[$size_key]['file'])) continue;
                $new_rel = ($new_dir && $new_dir !== '.') ? ($new_dir . '/' . $new_sizes[$size_key]['file']) : $new_sizes[$size_key]['file'];
                $map[$baseurl . '/' . ltrim($old_rel, '/')] = $baseurl . '/' . ltrim($new_rel, '/');
            }
        }

        return $map;
    }

    private static function move_old_files_from_meta_to_originals($old_meta) {
        $files = [];
        if (!is_array($old_meta)) return 0;

        if (!empty($old_meta['file'])) $files[] = $old_meta['file'];

        if (!empty($old_meta['sizes']) && is_array($old_meta['sizes'])) {
            $dir = dirname($old_meta['file']);
            foreach ($old_meta['sizes'] as $s) {
                if (empty($s['file'])) continue;
                $files[] = ($dir && $dir !== '.') ? ($dir . '/' . $s['file']) : $s['file'];
            }
        }

        $moved = 0;
        foreach ($files as $rel) {
            $rel = ltrim($rel, '/');
            if (!preg_match('/\.(jpe?g|png)$/i', $rel)) continue;
            if (preg_match('/\.bak\.(jpe?g|png)$/i', $rel)) continue;
            [$ok,] = self::move_rel_to_originals($rel);
            if ($ok) $moved++;
        }

        return $moved;
    }

    private static function move_old_files_from_orig_map_to_originals($orig_map) {
        if (!is_array($orig_map)) return 0;

        $rels = [];
        if (!empty($orig_map['full'])) $rels[] = $orig_map['full'];
        if (!empty($orig_map['sizes']) && is_array($orig_map['sizes'])) {
            foreach ($orig_map['sizes'] as $rel) {
                if (!empty($rel)) $rels[] = $rel;
            }
        }

        $moved = 0;
        foreach ($rels as $rel) {
            $rel = ltrim($rel, '/');
            if (!preg_match('/\.(jpe?g|png)$/i', $rel)) continue;
            if (preg_match('/\.bak\.(jpe?g|png)$/i', $rel)) continue;
            [$ok,] = self::move_rel_to_originals($rel);
            if ($ok) $moved++;
        }

        return $moved;
    }

    private static function normalize_root($filename_no_ext) {
        $s = $filename_no_ext;
        $s = preg_replace('/-scaled$/i', '', $s);
        $s = preg_replace('/-\d+x\d+$/', '', $s);
        return $s;
    }

    private static function parse_dims($filename_no_ext) {
        if (preg_match('/-(\d+)x(\d+)$/', $filename_no_ext, $m)) {
            return [(int)$m[1], (int)$m[2]];
        }
        return [0, 0];
    }

    private static function find_best_webp_for_old($old_filename, $webp_files) {
        $old_no_ext = preg_replace('/\.[^.]+$/', '', $old_filename);
        $old_root = self::normalize_root($old_no_ext);
        [$ow, $oh] = self::parse_dims($old_no_ext);

        $cands = [];
        foreach ($webp_files as $wf) {
            $w_no_ext = preg_replace('/\.webp$/i', '', $wf);
            $w_root = self::normalize_root($w_no_ext);
            if ($w_root !== $old_root) continue;
            [$ww, $wh] = self::parse_dims($w_no_ext);
            $cands[] = [$wf, $ww, $wh];
        }

        if (empty($cands)) return '';

        // Prefer exact width match, then closest height.
        if ($ow > 0) {
            $best = null;
            foreach ($cands as $c) {
                [$wf, $ww, $wh] = $c;
                if ($ww !== $ow) continue;
                $score = ($oh > 0) ? abs($wh - $oh) : 0;
                if ($best === null || $score < $best[0]) {
                    $best = [$score, $wf];
                }
            }
            if ($best !== null) return $best[1];
        }

        // Otherwise prefer the non-size "full" webp (no trailing -WxH)
        foreach ($cands as $c) {
            [$wf, $ww, $wh] = $c;
            if ($ww === 0 && $wh === 0) return $wf;
        }

        return $cands[0][0];
    }

    private static function move_original_image_leftovers($new_meta, $replace_everywhere) {
        // Handles WordPress "original_image" leftovers: when WP creates foo-scaled.jpg as the attachment,
        // the original foo.jpg (and its older size files) can still exist. Your screenshot is exactly this.

        if (!is_array($new_meta) || empty($new_meta['file'])) return [0, 0];

        $basedir = self::get_upload_basedir();
        $baseurl = self::get_upload_baseurl();

        $dir_rel = dirname($new_meta['file']);
        if ($dir_rel === '.' || $dir_rel === DIRECTORY_SEPARATOR) $dir_rel = '';

        $abs_dir = rtrim($basedir . '/' . ltrim($dir_rel, '/'), '/');
        if (!is_dir($abs_dir)) return [0, 0];

        // Roots to match (normalized)
        $roots = [];

        // Root from original_image if present
        if (!empty($new_meta['original_image'])) {
            $bn = basename($new_meta['original_image']);
            $bn = preg_replace('/\.(jpe?g|png)$/i', '', $bn);
            $roots[] = self::normalize_root($bn);
        }

        // Root derived from current attachment file (strip -scaled)
        $cur_bn = basename($new_meta['file']);
        $cur_bn = preg_replace('/\.(webp|jpe?g|png)$/i', '', $cur_bn);
        $roots[] = self::normalize_root($cur_bn);

        $roots = array_values(array_unique(array_filter($roots)));
        if (empty($roots)) return [0, 0];

        // Build webp list in directory
        $webps = [];
        $it = new DirectoryIterator($abs_dir);
        foreach ($it as $f) {
            if ($f->isDot() || !$f->isFile()) continue;
            $name = $f->getFilename();
            if (preg_match('/\.webp$/i', $name)) $webps[] = $name;
        }

        $moved = 0;
        $url_updates = 0;

        $it = new DirectoryIterator($abs_dir);
        foreach ($it as $f) {
            if ($f->isDot() || !$f->isFile()) continue;
            $name = $f->getFilename();

            if (!preg_match('/\.(jpe?g|png)$/i', $name)) continue;
            if (preg_match('/\.bak\.(jpe?g|png)$/i', $name)) continue;

            $no_ext = preg_replace('/\.(jpe?g|png)$/i', '', $name);
            $root = self::normalize_root($no_ext);
            if (!in_array($root, $roots, true)) continue;

            // Safety: only move if we can map to some webp sibling (prevents breaking references)
            $best_webp = self::find_best_webp_for_old($name, $webps);
            if ($replace_everywhere && !$best_webp) {
                continue;
            }

            $rel_old = ($dir_rel ? ($dir_rel . '/' . $name) : $name);
            [$ok,] = self::move_rel_to_originals($rel_old);
            if ($ok) {
                $moved++;
                if ($replace_everywhere && $best_webp) {
                    $rel_new = ($dir_rel ? ($dir_rel . '/' . $best_webp) : $best_webp);
                    $url_updates += self::replace_urls_everywhere_variants($baseurl . '/' . ltrim($rel_old, '/'), $baseurl . '/' . ltrim($rel_new, '/'));
                }
            }
        }

        return [$moved, $url_updates];
    }

    // ---------------- Conversion on upload ----------------

    public static function convert_on_generate_metadata($metadata, $attachment_id) {
        if (get_post_meta($attachment_id, self::META_CONVERTED, true)) return $metadata;

        $mime = get_post_mime_type($attachment_id);
        if (!self::is_target_mime($mime)) return $metadata;
        if (!self::image_editor_can_webp()) return $metadata;

        $quality = (int) get_option(self::OPT_QUALITY, 82);
        $reuse_archived = !empty(get_option(self::OPT_REUSE_ARCHIVED, 1));

        $abs_full = get_attached_file($attachment_id);
        if (!$abs_full || !file_exists($abs_full)) return $metadata;

        // Snapshot previous metadata for repair + URL map
        $prev_meta = wp_get_attachment_metadata($attachment_id);
        if (!is_array($prev_meta) || empty($prev_meta)) $prev_meta = $metadata;
        update_post_meta($attachment_id, self::META_PREV_META, wp_json_encode($prev_meta));

        // Convert full
        $abs_full_webp = self::change_ext_to_webp($abs_full);

        $res = null;
        if (!file_exists($abs_full_webp) && $reuse_archived) {
            $rel_full_webp = self::relpath_from_basedir($abs_full_webp);
            if ($rel_full_webp) {
                $arch_full_webp = self::webp_archive_abs_path_for_rel($rel_full_webp);
                if ($arch_full_webp && file_exists($arch_full_webp)) {
                    $did_copy = @copy($arch_full_webp, $abs_full_webp);
                    if ($did_copy) {
                        self::log_write('REUSE ARCHIVED WEBP (bulk full): ' . $rel_full_webp, 'INFO');
                    } else {
                        self::log_write('REUSE ARCHIVED WEBP MISS (bulk full copy failed): ' . $rel_full_webp, 'INFO');
                        $res = self::convert_file_to_webp($abs_full, $abs_full_webp, $quality);
                        if ($res === true) {
                            self::log_write('ENCODE WEBP (bulk full): ' . $rel_full_webp, 'INFO');
                        }
                    }
                } else {
                    self::log_write('REUSE ARCHIVED WEBP MISS (bulk full not found): ' . $rel_full_webp, 'INFO');
                    $res = self::convert_file_to_webp($abs_full, $abs_full_webp, $quality);
                    if ($res === true) {
                        self::log_write('ENCODE WEBP (bulk full): ' . $rel_full_webp, 'INFO');
                    }
                }
            } else {
                self::log_write('REUSE ARCHIVED WEBP MISS (bulk full no relpath): ' . $abs_full_webp, 'INFO');
                $res = self::convert_file_to_webp($abs_full, $abs_full_webp, $quality);
                if ($res === true) {
                    self::log_write('ENCODE WEBP (bulk full): ' . $abs_full_webp, 'INFO');
                }
            }
        } elseif (!file_exists($abs_full_webp)) {
            $res = self::convert_file_to_webp($abs_full, $abs_full_webp, $quality);
            if ($res === true) {
                $rel_full_webp2 = self::relpath_from_basedir($abs_full_webp);
                self::log_write('ENCODE WEBP (bulk full): ' . ($rel_full_webp2 ? $rel_full_webp2 : $abs_full_webp), 'INFO');
            }
        }

        if ($res && is_wp_error($res)) return $metadata;

        $orig_map = [
            'full'  => self::relpath_from_basedir($abs_full),
            'sizes' => [],
        ];

        // Move original full away
        self::move_to_originals($abs_full);

        // Switch attachment to webp
        $rel_webp = self::relpath_from_basedir($abs_full_webp);
        if ($rel_webp) {
            update_attached_file($attachment_id, $abs_full_webp);
            $metadata['file'] = $rel_webp;
        }

        // Convert sizes
        if (!empty($metadata['sizes']) && is_array($metadata['sizes'])) {
            $base_dir = dirname($abs_full_webp);

            foreach ($metadata['sizes'] as $size_key => $size_data) {
                if (empty($size_data['file'])) continue;

                $abs_size = trailingslashit($base_dir) . $size_data['file'];
                if (!file_exists($abs_size)) continue;
                if (!preg_match('~\.(jpe?g|png)$~i', $abs_size)) continue;

                $abs_size_webp = self::change_ext_to_webp($abs_size);

                $res2 = null;
                if (!file_exists($abs_size_webp) && $reuse_archived) {
                    $rel_size_webp = self::relpath_from_basedir($abs_size_webp);
                    if ($rel_size_webp) {
                        $arch_size_webp = self::webp_archive_abs_path_for_rel($rel_size_webp);
                        if ($arch_size_webp && file_exists($arch_size_webp)) {
                            if (!@copy($arch_size_webp, $abs_size_webp)) {
                                $res2 = self::convert_file_to_webp($abs_size, $abs_size_webp, $quality);
                            }
                        } else {
                            $res2 = self::convert_file_to_webp($abs_size, $abs_size_webp, $quality);
                        }
                    } else {
                        $res2 = self::convert_file_to_webp($abs_size, $abs_size_webp, $quality);
                    }
                } elseif (!file_exists($abs_size_webp)) {
                    $res2 = self::convert_file_to_webp($abs_size, $abs_size_webp, $quality);
                }

                if ($res2 && is_wp_error($res2)) continue;

                $orig_map['sizes'][$size_key] = self::relpath_from_basedir($abs_size);
                self::move_to_originals($abs_size);

                $metadata['sizes'][$size_key]['file'] = basename($abs_size_webp);
                $metadata['sizes'][$size_key]['mime-type'] = 'image/webp';
            }
        }

        wp_update_post([
            'ID' => $attachment_id,
            'post_mime_type' => 'image/webp',
        ]);

        update_post_meta($attachment_id, self::META_ORIG_MAP, wp_json_encode($orig_map));
        update_post_meta($attachment_id, self::META_CONVERTED, 1);

        // One-time: fix URLs + handle original_image leftovers for this attachment
        self::post_convert_fixup($attachment_id, $prev_meta, $metadata);

        return $metadata;
    }

    private static function post_convert_fixup($attachment_id, $prev_meta, $new_meta = null) {
        // Keep this light: bulk repair handles the rest.
        if (!$new_meta) $new_meta = wp_get_attachment_metadata($attachment_id);
        if (!is_array($new_meta) || empty($new_meta)) return;

        // No DB replace here; user controls that checkbox in bulk/repair.
        // But we DO move original_image leftovers for future uploads to keep uploads clean.
        self::move_original_image_leftovers($new_meta, false);
    }

    // ---------------- AJAX: bulk convert ----------------

    public static function ajax_bulk_convert() {
        if (function_exists('nocache_headers')) nocache_headers();
        if (function_exists('wp_ob_end_flush_all')) wp_ob_end_flush_all();
        while (ob_get_level()) { @ob_end_clean(); }

        if (!current_user_can('manage_options')) wp_send_json_error('No permission');
        check_ajax_referer('hsbc_webp_bulk');

        if (!self::image_editor_can_webp()) {
            wp_send_json_error('Server cannot generate WebP (missing GD/Imagick WebP support).');
        }

        global $wpdb;

        $limit  = isset($_POST['limit']) ? min(50, max(1, (int)$_POST['limit'])) : self::get_batch_size();
        $replace_everywhere = !empty($_POST['replace_everywhere']);

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- admin-only batch tool; prepared query; caching not useful
        $ids = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT p.ID
                 FROM {$wpdb->posts} p
                 LEFT JOIN {$wpdb->postmeta} m1 ON (m1.post_id = p.ID AND m1.meta_key = %s)
                 LEFT JOIN {$wpdb->postmeta} m2 ON (m2.post_id = p.ID AND m2.meta_key = %s)
                 WHERE p.post_type = 'attachment'
                   AND p.post_status = 'inherit'
                   AND (p.post_mime_type = 'image/jpeg' OR p.post_mime_type = 'image/png')
                   AND m1.post_id IS NULL
                   AND m2.post_id IS NULL
                 ORDER BY p.ID ASC
                 LIMIT %d",
                self::META_CONVERTED, self::META_FAILED, $limit
            )
        );

        $processed = 0;
        $converted = 0;
        $skipped   = 0;
        $failed    = 0;
        $url_updates_total = 0;

        foreach ($ids as $id) {
            $processed++;
            $old_url = wp_get_attachment_url($id);

            $mime = get_post_mime_type($id);
            if (!self::is_target_mime($mime)) { $skipped++; continue; }

            $file = get_attached_file($id);
            if (!$file || !file_exists($file)) {
                update_post_meta($id, self::META_FAILED, 'missing_file');
                $failed++;
                continue;
            }

            $prev_meta = wp_get_attachment_metadata($id);
            if (!is_array($prev_meta) || empty($prev_meta)) $prev_meta = [];
            update_post_meta($id, self::META_PREV_META, wp_json_encode($prev_meta));

            $meta2 = wp_generate_attachment_metadata($id, $file);
            if (!is_wp_error($meta2) && !empty($meta2)) {
                wp_update_attachment_metadata($id, $meta2);
            }

            if (get_post_meta($id, self::META_CONVERTED, true)) {
                $converted++;

                if ($replace_everywhere) {
                    $new_meta = wp_get_attachment_metadata($id);
                    if (is_array($prev_meta) && !empty($prev_meta) && is_array($new_meta) && !empty($new_meta)) {
                        $map = self::build_url_map_from_meta($prev_meta, $new_meta);
                        foreach ($map as $ou => $nu) {
                            $url_updates_total += self::replace_urls_everywhere_variants($ou, $nu);
                        }
                    } else {
                        $new_url = wp_get_attachment_url($id);
                        if ($new_url && $old_url && $new_url !== $old_url) {
                            $url_updates_total += self::replace_urls_everywhere_variants($old_url, $new_url);
                        }
                    }
                }
            } else {
                $msg = is_wp_error($meta2) ? $meta2->get_error_message() : 'convert_failed';
                update_post_meta($id, self::META_FAILED, $msg);
                $failed++;
            }
        }

        wp_send_json_success([
            'processed'   => $processed,
            'converted'   => $converted,
            'skipped'     => $skipped,
            'failed'      => $failed,
            'url_updates' => $url_updates_total,
            'done'        => empty($ids),
        ]);
    }

    // ---------------- AJAX: repair/cleanup ----------------

    public static function ajax_repair_existing() {
        if (function_exists('nocache_headers')) nocache_headers();
        if (function_exists('wp_ob_end_flush_all')) wp_ob_end_flush_all();
        while (ob_get_level()) { @ob_end_clean(); }

        if (!current_user_can('manage_options')) wp_send_json_error('No permission');
        check_ajax_referer('hsbc_webp_repair');

        global $wpdb;

        $limit  = isset($_POST['limit']) ? min(50, max(1, (int)$_POST['limit'])) : self::get_batch_size();
        $replace_everywhere = !empty($_POST['replace_everywhere']);

        $cursor = (int) get_option(self::OPT_REPAIR_CURSOR, 0);

        // Walk converted attachments by ID cursor (so repair can be run repeatedly)
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- admin-only batch tool; prepared query; caching not useful
        $ids = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT p.ID
                 FROM {$wpdb->posts} p
                 INNER JOIN {$wpdb->postmeta} mconv ON (mconv.post_id = p.ID AND mconv.meta_key = %s)
                 WHERE p.post_type = 'attachment'
                   AND p.post_status = 'inherit'
                   AND p.ID > %d
                 ORDER BY p.ID ASC
                 LIMIT %d",
                self::META_CONVERTED, $cursor, $limit
            )
        );

        if (empty($ids)) {
            // End of pass. Reset cursor.
            update_option(self::OPT_REPAIR_CURSOR, 0);
            wp_send_json_success([
                'processed'   => 0,
                'repaired'    => 0,
                'moved_files' => 0,
                'url_updates' => 0,
                'done'        => true,
            ]);
        }

        $processed = 0;
        $repaired = 0;
        $moved_files = 0;
        $url_updates_total = 0;

        foreach ($ids as $id) {
            $processed++;
            update_option(self::OPT_REPAIR_CURSOR, (int)$id);

            $did = false;

            $new_meta = wp_get_attachment_metadata($id);
            if (!is_array($new_meta) || empty($new_meta)) continue;

            $orig_map = null;
            $orig_map_json = get_post_meta($id, self::META_ORIG_MAP, true);
            if ($orig_map_json) {
                $tmp = json_decode($orig_map_json, true);
                if (is_array($tmp)) $orig_map = $tmp;
            }

            $prev_meta = null;
            $prev_meta_json = get_post_meta($id, self::META_PREV_META, true);
            if ($prev_meta_json) {
                $tmp = json_decode($prev_meta_json, true);
                if (is_array($tmp)) $prev_meta = $tmp;
            }

            // 1) URL fix (best available map)
            if ($replace_everywhere) {
                if (is_array($prev_meta) && !empty($prev_meta)) {
                    $map = self::build_url_map_from_meta($prev_meta, $new_meta);
                    foreach ($map as $ou => $nu) {
                        $u = self::replace_urls_everywhere_variants($ou, $nu);
                        if ($u) { $url_updates_total += $u; $did = true; }
                    }
                } elseif (is_array($orig_map)) {
                    $map = self::build_url_map_from_orig_map($orig_map, $new_meta);
                    foreach ($map as $ou => $nu) {
                        $u = self::replace_urls_everywhere_variants($ou, $nu);
                        if ($u) { $url_updates_total += $u; $did = true; }
                    }
                }
            }

            // 2) Cleanup old files
            if (is_array($orig_map)) {
                $m = self::move_old_files_from_orig_map_to_originals($orig_map);
                if ($m) { $moved_files += $m; $did = true; }
            }
            if (is_array($prev_meta) && !empty($prev_meta)) {
                $m = self::move_old_files_from_meta_to_originals($prev_meta);
                if ($m) { $moved_files += $m; $did = true; }
            }

            // 3) IMPORTANT: WordPress "original_image" leftovers (your screenshot)
            [$m2, $u2] = self::move_original_image_leftovers($new_meta, $replace_everywhere);
            if ($m2) { $moved_files += $m2; $did = true; }
            if ($u2) { $url_updates_total += $u2; $did = true; }

            if ($did) $repaired++;
        }

        wp_send_json_success([
            'processed'   => $processed,
            'repaired'    => $repaired,
            'moved_files' => $moved_files,
            'url_updates' => $url_updates_total,
            'done'        => false,
        ]);
    }

    // ---------------- AJAX: sweep uploads folder ----------------

    public static function ajax_sweep_uploads() {
        if (function_exists('nocache_headers')) nocache_headers();
        if (function_exists('wp_ob_end_flush_all')) wp_ob_end_flush_all();
        while (ob_get_level()) { @ob_end_clean(); }

        if (!current_user_can('manage_options')) wp_send_json_error('No permission');
        check_ajax_referer('hsbc_webp_sweep');

        $limit  = isset($_POST['limit']) ? min(50, max(1, (int)$_POST['limit'])) : self::get_batch_size();
        $replace_everywhere = !empty($_POST['replace_everywhere']);

    $res = self::process_sweep_batch($limit, $replace_everywhere);
    $done = !empty($res['done']);
    $stats = isset($res['stats']) && is_array($res['stats']) ? $res['stats'] : [];

    wp_send_json_success([
        'processed'   => (int) ($stats['processed'] ?? 0),
        'converted'   => (int) ($stats['converted'] ?? 0),
        'moved_files' => (int) ($stats['moved_files'] ?? 0),
        'url_updates' => (int) ($stats['url_updates'] ?? 0),
        'done'        => $done,
    ]);
}


    // ---------------- AJAX: revert (one batch, for debugging) ----------------

    public static function ajax_revert_batch() {
        if (function_exists('nocache_headers')) nocache_headers();
        if (function_exists('wp_ob_end_flush_all')) wp_ob_end_flush_all();
        while (ob_get_level()) { @ob_end_clean(); }

        if (!current_user_can('manage_options')) wp_send_json_error('No permission');
        check_ajax_referer('hsbc_webp_revert');

        $limit  = isset($_POST['limit']) ? min(50, max(1, (int)$_POST['limit'])) : self::get_batch_size();
        $replace_everywhere = !empty($_POST['replace_everywhere']);

        $res = self::process_revert_batch($limit, $replace_everywhere);
        wp_send_json_success([
            'processed' => $res['stats']['processed'] ?? 0,
            'reverted' => $res['stats']['reverted'] ?? 0,
            'moved_webp' => $res['stats']['moved_webp'] ?? 0,
            'restored' => $res['stats']['restored'] ?? 0,
            'url_updates' => $res['stats']['url_updates'] ?? 0,
            'done' => !empty($res['done']),
        ]);
    }
}

HSBC_WebP_Only::init();
