<?php
/**
 * Plugin Name: OrderMigo - Export Orders for WooCommerce
 * Description: Export WooCommerce orders to CSV with support for both traditional and HPOS (High-Performance Order Storage) systems.
 * Version: 1.0.0
 * Author: ordermigocom
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: ordermigo-export-orders-for-woocommerce
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * Requires Plugins: woocommerce
 * WC requires at least: 6.0
 * WC tested up to: 8.5
 *
 * @package ExportOrdersWC
 */

namespace ExportOrdersWC;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define plugin constants.
define( 'EXPORT_ORDERS_WC_VERSION', '1.0.0' );
define( 'EXPORT_ORDERS_WC_PLUGIN_FILE', __FILE__ );
define( 'EXPORT_ORDERS_WC_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'EXPORT_ORDERS_WC_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

// Load the autoloader.
require_once EXPORT_ORDERS_WC_PLUGIN_DIR . 'includes/class-autoloader.php';

/**
 * Main plugin class that bootstraps the plugin.
 *
 * @since 1.0.0
 */
final class Plugin {

	/**
	 * Plugin instance.
	 *
	 * @var Plugin
	 */
	private static $instance = null;

	/**
	 * Service container.
	 *
	 * @var ServiceContainer
	 */
	private $container;

	/**
	 * Get plugin instance.
	 *
	 * @return Plugin
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		$this->init();
	}

	/**
	 * Initialize the plugin.
	 *
	 * @return void
	 */
	private function init() {
		// Initialize autoloader.
		Autoloader::register();

		// Initialize service container.
		$this->container = new ServiceContainer();

		// Register hooks.
		add_action( 'before_woocommerce_init', array( $this, 'declare_hpos_compatibility' ) );
		add_action( 'plugins_loaded', array( $this, 'on_plugins_loaded' ) );
		register_activation_hook( EXPORT_ORDERS_WC_PLUGIN_FILE, array( $this, 'on_activation' ) );
		register_deactivation_hook( EXPORT_ORDERS_WC_PLUGIN_FILE, array( $this, 'on_deactivation' ) );
	}

	/**
	 * Declare HPOS (High-Performance Order Storage) compatibility.
	 *
	 * @return void
	 */
	public function declare_hpos_compatibility() {
		if ( class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', EXPORT_ORDERS_WC_PLUGIN_FILE, true );
		}
	}

	/**
	 * Callback for plugins_loaded hook.
	 *
	 * @return void
	 */
	public function on_plugins_loaded() {
		// Check if WooCommerce is active.
		if ( ! $this->is_woocommerce_active() ) {
			add_action( 'admin_notices', array( $this, 'woocommerce_missing_notice' ) );
			return;
		}

		// Bootstrap the plugin.
		$this->container->bootstrap();
	}

	/**
	 * Check if WooCommerce is active.
	 *
	 * @return bool
	 */
	private function is_woocommerce_active() {
		return class_exists( 'WooCommerce' );
	}

	/**
	 * Display admin notice if WooCommerce is not active.
	 *
	 * @return void
	 */
	public function woocommerce_missing_notice() {
		?>
		<div class="notice notice-error">
			<p>
				<?php
				echo wp_kses_post(
					sprintf(
						/* translators: %s: WooCommerce plugin link */
						__( '<strong>OrderMigo - Export Orders for WooCommerce</strong> requires WooCommerce to be installed and active. You can download %s here.', 'ordermigo-export-orders-for-woocommerce' ),
						'<a href="https://wordpress.org/plugins/woocommerce/" target="_blank">WooCommerce</a>'
					)
				);
				?>
			</p>
		</div>
		<?php
	}

	/**
	 * Plugin activation callback.
	 *
	 * @return void
	 */
	public function on_activation() {
		// Check PHP version.
		if ( version_compare( PHP_VERSION, '7.4', '<' ) ) {
			deactivate_plugins( plugin_basename( EXPORT_ORDERS_WC_PLUGIN_FILE ) );
			wp_die(
				esc_html__( 'OrderMigo - Export Orders for WooCommerce requires PHP 7.4 or higher.', 'ordermigo-export-orders-for-woocommerce' ),
				esc_html__( 'Plugin Activation Error', 'ordermigo-export-orders-for-woocommerce' ),
				array( 'back_link' => true )
			);
		}

		// Flush rewrite rules.
		flush_rewrite_rules();
	}

	/**
	 * Plugin deactivation callback.
	 *
	 * @return void
	 */
	public function on_deactivation() {
		// Flush rewrite rules.
		flush_rewrite_rules();
	}

	/**
	 * Get service container.
	 *
	 * @return ServiceContainer
	 */
	public function get_container() {
		return $this->container;
	}
}

/**
 * Get plugin instance.
 *
 * @return Plugin
 */
function plugin() {
	return Plugin::instance();
}

// Initialize the plugin.
plugin();

