<?php
/**
 * Service Container class
 *
 * @package ExportOrdersWC
 */

namespace ExportOrdersWC;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Service Container for Dependency Injection.
 *
 * Follows Dependency Inversion Principle (SOLID).
 *
 * @since 1.0.0
 */
class ServiceContainer {

	/**
	 * Registered services.
	 *
	 * @var array
	 */
	private $services = array();

	/**
	 * Service instances.
	 *
	 * @var array
	 */
	private $instances = array();

	/**
	 * Bootstrap the service container.
	 *
	 * @return void
	 */
	public function bootstrap() {
		$this->register_services();
		$this->init_services();
	}

	/**
	 * Register all services.
	 *
	 * @return void
	 */
	private function register_services() {
		// Register HPOS compatibility checker.
		$this->register(
			'hpos_checker',
			function () {
				return new HposCompatibility();
			}
		);

		// Register order repository (factory pattern based on HPOS).
		$this->register(
			'order_repository',
			function () {
				$hpos_checker = $this->get( 'hpos_checker' );
				return new OrderRepository( $hpos_checker );
			}
		);

		// Register CSV exporter.
		$this->register(
			'csv_exporter',
			function () {
				return new CsvExporter( $this->get( 'order_repository' ) );
			}
		);

		// Register export handler.
		$this->register(
			'export_handler',
			function () {
				return new ExportHandler( $this->get( 'csv_exporter' ) );
			}
		);

		// Register admin UI.
		$this->register(
			'admin_ui',
			function () {
				return new AdminUi( $this->get( 'hpos_checker' ) );
			}
		);
	}

	/**
	 * Initialize services that need to hook into WordPress.
	 *
	 * @return void
	 */
	private function init_services() {
		// Initialize admin UI.
		$this->get( 'admin_ui' )->init();

		// Initialize export handler.
		$this->get( 'export_handler' )->init();
	}

	/**
	 * Register a service.
	 *
	 * @param string   $name    Service name.
	 * @param callable $factory Factory callback.
	 * @return void
	 */
	public function register( $name, $factory ) {
		$this->services[ $name ] = $factory;
	}

	/**
	 * Get a service instance.
	 *
	 * @param string $name Service name.
	 * @return mixed
	 * @throws \Exception If service not found.
	 */
	public function get( $name ) {
		if ( ! isset( $this->instances[ $name ] ) ) {
			if ( ! isset( $this->services[ $name ] ) ) {
				throw new \Exception( sprintf( 'Service "%s" not found.', esc_html( $name ) ) );
			}

			$this->instances[ $name ] = call_user_func( $this->services[ $name ] );
		}

		return $this->instances[ $name ];
	}
}
