<?php
/**
 * Order Repository class
 *
 * @package ExportOrdersWC
 */

namespace ExportOrdersWC;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Order Repository.
 *
 * Single Responsibility Principle: Handles only order data retrieval.
 * Dependency Inversion Principle: Depends on abstraction (HposCompatibility).
 *
 * @since 1.0.0
 */
class OrderRepository {

	/**
	 * HPOS compatibility checker.
	 *
	 * @var HposCompatibility
	 */
	private $hpos_checker;

	/**
	 * Constructor.
	 *
	 * @param HposCompatibility $hpos_checker HPOS compatibility checker.
	 */
	public function __construct( HposCompatibility $hpos_checker ) {
		$this->hpos_checker = $hpos_checker;
	}

	/**
	 * Get orders based on criteria.
	 *
	 * @param array $args Query arguments.
	 * @return \WC_Order[]
	 */
	public function get_orders( $args = array() ) {
		$defaults = array(
			'limit'   => -1,
			'orderby' => 'date',
			'order'   => 'DESC',
			'return'  => 'objects',
		);

		$args = wp_parse_args( $args, $defaults );

		// Use wc_get_orders() which works for both HPOS and non-HPOS.
		return wc_get_orders( $args );
	}

	/**
	 * Get order by ID.
	 *
	 * @param int $order_id Order ID.
	 * @return \WC_Order|false
	 */
	public function get_order( $order_id ) {
		return wc_get_order( $order_id );
	}

	/**
	 * Get orders for export based on current screen filters.
	 *
	 * @return \WC_Order[]
	 */
	public function get_orders_for_export() {
		$args = array(
			'limit'   => -1,
			'orderby' => 'date',
			'order'   => 'DESC',
		);

		// Check for status filter.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! empty( $_GET['status'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$status = sanitize_text_field( wp_unslash( $_GET['status'] ) );
			if ( 'all' !== $status ) {
				$args['status'] = $status;
			}
		}

		// Check for date filters.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! empty( $_GET['m'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$m = sanitize_text_field( wp_unslash( $_GET['m'] ) );
			if ( strlen( $m ) === 6 ) {
				$year  = substr( $m, 0, 4 );
				$month = substr( $m, 4, 2 );

				$args['date_created'] = $year . '-' . $month . '-01...' . $year . '-' . $month . '-31';
			}
		}

		return $this->get_orders( $args );
	}
}
