<?php
/**
 * Export Handler class
 *
 * @package ExportOrdersWC
 */

namespace ExportOrdersWC;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Export Handler.
 *
 * Single Responsibility Principle: Handles export requests and security.
 *
 * @since 1.0.0
 */
class ExportHandler {

	/**
	 * CSV exporter.
	 *
	 * @var CsvExporter
	 */
	private $csv_exporter;

	/**
	 * Export action name.
	 *
	 * @var string
	 */
	const EXPORT_ACTION = 'export_orders_wc';

	/**
	 * Nonce action name.
	 *
	 * @var string
	 */
	const NONCE_ACTION = 'export_orders_wc_nonce';

	/**
	 * Constructor.
	 *
	 * @param CsvExporter $csv_exporter CSV exporter.
	 */
	public function __construct( CsvExporter $csv_exporter ) {
		$this->csv_exporter = $csv_exporter;
	}

	/**
	 * Initialize the export handler.
	 *
	 * @return void
	 */
	public function init() {
		add_action( 'admin_init', array( $this, 'handle_export_request' ) );
	}

	/**
	 * Handle export request.
	 *
	 * @return void
	 */
	public function handle_export_request() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! isset( $_GET['action'] ) ) {
			return;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$action = sanitize_text_field( wp_unslash( $_GET['action'] ) );
		if ( self::EXPORT_ACTION !== $action ) {
			return;
		}

		// Security checks.
		$this->verify_permissions();
		$this->verify_nonce();

		// Perform export.
		$this->csv_exporter->export();
	}

	/**
	 * Verify user permissions.
	 *
	 * @return void
	 */
	private function verify_permissions() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die(
				esc_html__( 'You do not have permission to export orders.', 'ordermigo-export-orders-for-woocommerce' ),
				esc_html__( 'Permission Denied', 'ordermigo-export-orders-for-woocommerce' ),
				array( 'response' => 403 )
			);
		}
	}

	/**
	 * Verify nonce.
	 *
	 * @return void
	 */
	private function verify_nonce() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! isset( $_GET['_wpnonce'] ) ) {
			wp_die(
				esc_html__( 'Invalid security token.', 'ordermigo-export-orders-for-woocommerce' ),
				esc_html__( 'Security Error', 'ordermigo-export-orders-for-woocommerce' ),
				array( 'response' => 403 )
			);
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), self::NONCE_ACTION ) ) {
			wp_die(
				esc_html__( 'Invalid security token.', 'ordermigo-export-orders-for-woocommerce' ),
				esc_html__( 'Security Error', 'ordermigo-export-orders-for-woocommerce' ),
				array( 'response' => 403 )
			);
		}
	}

	/**
	 * Get export URL.
	 *
	 * @return string
	 */
	public static function get_export_url() {
		return wp_nonce_url(
			add_query_arg( 'action', self::EXPORT_ACTION, admin_url() ),
			self::NONCE_ACTION
		);
	}
}
