<?php
/**
 * CSV Exporter class
 *
 * @package ExportOrdersWC
 */

namespace ExportOrdersWC;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * CSV Exporter.
 *
 * Single Responsibility Principle: Handles only CSV export logic.
 * Open/Closed Principle: Extensible for different export formats.
 *
 * @since 1.0.0
 */
class CsvExporter {

	/**
	 * Order repository.
	 *
	 * @var OrderRepository
	 */
	private $order_repository;

	/**
	 * Constructor.
	 *
	 * @param OrderRepository $order_repository Order repository.
	 */
	public function __construct( OrderRepository $order_repository ) {
		$this->order_repository = $order_repository;
	}

	/**
	 * Export orders to CSV.
	 *
	 * @return void
	 */
	public function export() {
		$orders = $this->order_repository->get_orders_for_export();

		if ( empty( $orders ) ) {
			wp_die( esc_html__( 'No orders found to export.', 'ordermigo-export-orders-for-woocommerce' ) );
		}

		// Set headers for CSV download.
		$filename = 'orders-export-' . gmdate( 'Y-m-d-H-i-s' ) . '.csv';
		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename="' . esc_attr( $filename ) . '"' );
		header( 'Pragma: no-cache' );
		header( 'Expires: 0' );

		// Open output stream.
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
		$output = fopen( 'php://output', 'w' );

		// Add BOM for UTF-8.
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fprintf
		fprintf( $output, chr( 0xEF ) . chr( 0xBB ) . chr( 0xBF ) );

		// Write headers.
		$headers = $this->get_csv_headers();
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fputcsv
		fputcsv( $output, $headers );

		// Write order data.
		foreach ( $orders as $order ) {
			$row = $this->get_order_row_data( $order );
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fputcsv
			fputcsv( $output, $row );
		}

		// Stream is automatically closed on exit.
		exit;
	}

	/**
	 * Get CSV headers.
	 *
	 * @return array
	 */
	private function get_csv_headers() {
		$headers = array(
			__( 'Order ID', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Order Number', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Order Status', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Order Date', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Customer Name', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Customer Email', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Billing First Name', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Billing Last Name', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Billing Company', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Billing Address 1', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Billing Address 2', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Billing City', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Billing State', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Billing Postcode', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Billing Country', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Billing Phone', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Shipping First Name', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Shipping Last Name', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Shipping Company', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Shipping Address 1', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Shipping Address 2', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Shipping City', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Shipping State', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Shipping Postcode', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Shipping Country', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Payment Method', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Shipping Method', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Order Total', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Order Subtotal', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Order Tax', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Order Shipping', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Order Discount', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Currency', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Customer Note', 'ordermigo-export-orders-for-woocommerce' ),
			__( 'Items', 'ordermigo-export-orders-for-woocommerce' ),
		);

		/**
		 * Filter CSV headers.
		 *
		 * @param array $headers CSV headers.
		 */
		return apply_filters( 'export_orders_wc_csv_headers', $headers );
	}

	/**
	 * Get order row data.
	 *
	 * @param \WC_Order $order Order object.
	 * @return array
	 */
	private function get_order_row_data( $order ) {
		$items_string = $this->get_items_string( $order );

		$row = array(
			$order->get_id(),
			$order->get_order_number(),
			$order->get_status(),
			$order->get_date_created() ? $order->get_date_created()->date( 'Y-m-d H:i:s' ) : '',
			$order->get_formatted_billing_full_name(),
			$order->get_billing_email(),
			$order->get_billing_first_name(),
			$order->get_billing_last_name(),
			$order->get_billing_company(),
			$order->get_billing_address_1(),
			$order->get_billing_address_2(),
			$order->get_billing_city(),
			$order->get_billing_state(),
			$order->get_billing_postcode(),
			$order->get_billing_country(),
			$order->get_billing_phone(),
			$order->get_shipping_first_name(),
			$order->get_shipping_last_name(),
			$order->get_shipping_company(),
			$order->get_shipping_address_1(),
			$order->get_shipping_address_2(),
			$order->get_shipping_city(),
			$order->get_shipping_state(),
			$order->get_shipping_postcode(),
			$order->get_shipping_country(),
			$order->get_payment_method_title(),
			$order->get_shipping_method(),
			$order->get_total(),
			$order->get_subtotal(),
			$order->get_total_tax(),
			$order->get_shipping_total(),
			$order->get_total_discount(),
			$order->get_currency(),
			$order->get_customer_note(),
			$items_string,
		);

		/**
		 * Filter order row data.
		 *
		 * @param array     $row   Order row data.
		 * @param \WC_Order $order Order object.
		 */
		return apply_filters( 'export_orders_wc_csv_row_data', $row, $order );
	}

	/**
	 * Get items string for order.
	 *
	 * @param \WC_Order $order Order object.
	 * @return string
	 */
	private function get_items_string( $order ) {
		$items = array();

		foreach ( $order->get_items() as $item ) {
			$product_name = $item->get_name();
			$quantity     = $item->get_quantity();
			$total        = $item->get_total();

			$items[] = sprintf( '%s x %d (Total: %s)', $product_name, $quantity, wc_price( $total ) );
		}

		return implode( ' | ', $items );
	}
}
