<?php
/**
 * Admin UI class
 *
 * @package ExportOrdersWC
 */

namespace ExportOrdersWC;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin UI.
 *
 * Single Responsibility Principle: Handles only admin UI elements.
 * Interface Segregation Principle: Clean interface for UI operations.
 *
 * @since 1.0.0
 */
class AdminUi {

	/**
	 * HPOS compatibility checker.
	 *
	 * @var HposCompatibility
	 */
	private $hpos_checker;

	/**
	 * Constructor.
	 *
	 * @param HposCompatibility $hpos_checker HPOS compatibility checker.
	 */
	public function __construct( HposCompatibility $hpos_checker ) {
		$this->hpos_checker = $hpos_checker;
	}

	/**
	 * Initialize the admin UI.
	 *
	 * @return void
	 */
	public function init() {
		// Add export button to legacy orders screen.
		add_action( 'manage_posts_extra_tablenav', array( $this, 'add_export_button_legacy' ), 20 );

		// Add export button to HPOS orders screen.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_hpos_button_script' ) );

		// Add styles.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_styles' ) );
	}

	/**
	 * Add export button for legacy orders screen (non-HPOS).
	 *
	 * @param string $which Top or bottom of the table.
	 * @return void
	 */
	public function add_export_button_legacy( $which ) {
		global $typenow;

		// Only on shop_order post type and top of the table.
		if ( 'shop_order' !== $typenow || 'top' !== $which ) {
			return;
		}

		// Don't show on HPOS screens.
		if ( $this->hpos_checker->is_hpos_enabled() ) {
			return;
		}

		$this->render_export_button();
	}

	/**
	 * Enqueue script to add export button for HPOS orders screen.
	 *
	 * @param string $hook Current admin page hook.
	 * @return void
	 */
	public function enqueue_hpos_button_script( $hook ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
		$screen = get_current_screen();

		// Check if we're on HPOS orders screen.
		if ( ! $screen || 'woocommerce_page_wc-orders' !== $screen->id ) {
			return;
		}

		// Only show on HPOS screens.
		if ( ! $this->hpos_checker->is_hpos_enabled() ) {
			return;
		}

		// Enqueue jQuery first.
		wp_enqueue_script( 'jquery' );

		// Enqueue inline script to inject button.
		$script = '
			jQuery(document).ready(function($) {
				var exportButton = ' . wp_json_encode( $this->get_export_button_html() ) . ";
				var \$filtersBar = $('.wp-list-table.orders .tablenav.top .alignleft.actions');
				if (\$filtersBar.length) {
					\$filtersBar.first().after(exportButton);
				} else {
					$('.wp-list-table.orders').before('<div class=\"tablenav top\">' + exportButton + '</div>');
				}
			});
		";
		wp_add_inline_script( 'jquery', $script );
	}

	/**
	 * Get export button HTML.
	 *
	 * @return string
	 */
	private function get_export_button_html() {
		ob_start();
		$this->render_export_button();
		return ob_get_clean();
	}

	/**
	 * Render the export button.
	 *
	 * @return void
	 */
	private function render_export_button() {
		$export_url = ExportHandler::get_export_url();
		?>
		<div class="alignleft actions ordermigo-export-orders-for-woocommerce-button-container">
			<a href="<?php echo esc_url( $export_url ); ?>" class="button button-primary ordermigo-export-orders-for-woocommerce-button">
				<span class="dashicons dashicons-download" style="vertical-align: middle; margin-top: 3px;"></span>
				<?php esc_html_e( 'Export Orders to CSV', 'ordermigo-export-orders-for-woocommerce' ); ?>
			</a>
		</div>
		<?php
	}

	/**
	 * Enqueue admin styles and scripts.
	 *
	 * @param string $hook Current admin page hook.
	 * @return void
	 */
	public function enqueue_styles( $hook ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
		// Only load on orders screens.
		$screen = get_current_screen();
		if ( ! $screen ) {
			return;
		}

		$order_screen_id = $this->hpos_checker->get_order_screen_id();

		if ( $screen->id !== $order_screen_id ) {
			return;
		}

		// jQuery is enqueued in enqueue_hpos_button_script if needed for HPOS.

		// Inline CSS for the export button.
		$custom_css = '
			.ordermigo-export-orders-for-woocommerce-button-container {
				margin: 0 0 0 6px;
			}
			.ordermigo-export-orders-for-woocommerce-button .dashicons {
				font-size: 16px;
				height: 16px;
				width: 16px;
			}
		';
		wp_add_inline_style( 'wp-admin', $custom_css );
	}
}

