<?php

namespace WDevs\CustomerOrderExport;

/**
 * FreemiusMenu - Clean wrapper around Freemius menu functionality
 * 
 * This class wraps Freemius menu logic with WordPress-style method names
 * and provides the ability to render menu items as HTML without affecting WordPress menus.
 */
class FreemiusMenu {
    
    /**
     * @var \Freemius The Freemius instance
     */
    private $fs;
    
    /**
     * @var array Menu configuration from Freemius
     */
    private $menu_config;
    
    
    /**
     * Constructor
     * 
     * @param \Freemius $fs_instance The Freemius instance to wrap
     * @param array $menu_config Menu configuration array
     */
    public function __construct($fs_instance, $menu_config) {
        $this->fs = $fs_instance;
        $this->menu_config = $menu_config;
    }
    
    
    /**
     * Check if any menu items should be shown at all
     * Replaces the confusing should_add_submenu_or_action_links() method
     * 
     * @return bool Whether to show menu items
     */
    public function can_show_menus() {
        // Simplified - since we're always on our own plugin pages, 
        // we don't need complex activation mode or network admin checks
        
        // Don't show menu items for addons
        if ($this->fs->is_addon()) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Check if a specific menu item should be visible
     * Replaces the vague is_submenu_item_visible() method
     * 
     * @param string $item_type The menu item type (account, pricing, contact, etc.)
     * @return bool Whether the item should be visible
     */
    public function is_item_visible($item_type) {
        // Item-specific visibility logic using EXACT Freemius conditions
        switch ($item_type) {
            case 'account':
                return $this->is_account_item_visible();
                
            case 'pricing':
                return $this->is_pricing_item_visible();
                
            case 'contact':
                return $this->is_contact_item_visible();
                
            case 'support':
                return $this->is_support_item_visible();
                
            case 'affiliation':
                return $this->is_affiliation_item_visible();
                
            case 'addons':
                return $this->is_addons_item_visible();
                
            default:
                return false;
        }
    }
    
    /**
     * Get enabled menu items with rich data
     * 
     * @param array $custom_menu_config Optional custom menu configuration
     * @return array Array of enabled items: ['account' => ['url' => '...', 'class' => '...', 'label' => '...']]
     */
    public function get_enabled_menu_items($custom_menu_config = null) {
        if ($custom_menu_config) {
            $this->menu_config = $custom_menu_config;
        }
        
        if (!$this->can_show_menus()) {
            return array();
        }
        
        $enabled_items = array();
        
        foreach ($this->menu_config as $item_type => $enabled) {
            if ($this->is_item_visible($item_type)) {
                $enabled_items[$item_type] = array(
                    'url'   => $this->get_item_url($item_type),
                    'class' => $this->get_item_class($item_type),
                    'label' => $this->get_item_label($item_type)
                );
            }
        }
        
        return $enabled_items;
    }
    
    /**
     * Generate URL for a specific menu item type
     * 
     * @param string $item_type The menu item type
     * @param array $params Additional parameters for the URL
     * @return string The generated URL
     */
    public function get_item_url($item_type, $params = array()) {
        switch ($item_type) {
            case 'account':
                return $this->fs->get_account_url(false, $params);
                
            case 'pricing':
                return $this->fs->pricing_url();
                
            case 'contact':
                if ($this->fs->is_premium()) {
                    return $this->fs->get_account_url('contact');
                } else {
                    // External contact form for free versions using Contact Form Manager
                    if (class_exists('FS_Contact_Form_Manager')) {
                        return \FS_Contact_Form_Manager::instance()->get_standalone_link($this->fs);
                    }
                    return '#contact';
                }
                
            case 'support':
                return $this->fs->get_account_url('support');
                
            case 'affiliation':
                return $this->fs->get_account_url('affiliation');
                
            case 'addons':
                return $this->fs->get_account_url('addons');
                
            default:
                return '#';
        }
    }
    
    /**
     * Get human readable label for a specific menu item type
     * 
     * @param string $item_type The menu item type
     * @return string The label
     */
    public function get_item_label($item_type) {
        switch ($item_type) {
            case 'account':
                return $this->fs->get_text_inline('Account', 'account');
                
            case 'pricing':
                // Use Freemius dynamic pricing label with arrow
                $label = $this->fs->get_pricing_cta_label();
                
                // Add arrow if not paying (like original Freemius)
                if ($this->fs->is_not_paying()) {
                    $arrow = is_rtl() 
                        ? $this->fs->get_text_x_inline('&#x2190;', 'ASCII arrow left icon', 'symbol_arrow-left')
                        : $this->fs->get_text_x_inline('&#x27a4;', 'ASCII arrow right icon', 'symbol_arrow-right');
                    
                    $label .= '&nbsp;&nbsp;' . $arrow;
                }
                
                return $label;
                
            case 'contact':
                return $this->fs->get_text_inline('Contact Us', 'contact-us');
                
            case 'support':
                return $this->fs->get_text_inline('Support', 'support');
                
            case 'affiliation':
                return $this->fs->get_text_inline('Affiliation', 'affiliation');
                
            case 'addons':
                return $this->fs->get_text_inline('Add-Ons', 'add-ons');
                
            default:
                return ucfirst($item_type);
        }
    }
    
    /**
     * Get CSS class for a specific menu item type
     * 
     * @param string $item_type The menu item type
     * @return string The CSS class
     */
    public function get_item_class($item_type) {
        switch ($item_type) {
            case 'pricing':
                // Determine pricing class based on user state - EXACT original logic
                if ($this->fs->is_not_paying()) {
                    $class = 'upgrade-mode';
                    
                    if ($this->fs->is_in_trial_promotion() && !$this->fs->is_paying_or_trial()) {
                        $class = 'trial-mode';
                    }
                    
                    return $class;
                }
                return '';
                
            case 'contact':
                // External contact for free users, internal for premium
                if (!$this->fs->is_premium()) {
                    return 'fs_external_contact';
                }
                return '';
                
            case 'account':
            case 'support':
            case 'affiliation':
            case 'addons':
            default:
                return '';
        }
    }
    
    /**
     * Private helper methods
     */
    
    
    /**
     * Reproduce Freemius get_bool_option() method
     * 
     * @param array $options
     * @param string $key
     * @param bool $default
     * @return bool
     */
    private function get_bool_option($options, $key, $default = false) {
        return isset($options[$key]) && is_bool($options[$key]) ? $options[$key] : $default;
    }
    
    
    /**
     * Account item visibility logic - EXACT Freemius conditions
     * 
     * @return bool
     */
    private function is_account_item_visible() {
	    if (!$this->menu_item_is_enabled('account')) {
            return false;
        }
        
        // Must be registered to see account
        if (!$this->fs->is_registered()) {
            return false;
        }
        
        // For free wp.org themes, only show if has paid plans
        if ($this->is_free_wordpress_org_theme() && !$this->fs->has_paid_plan()) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Pricing item visibility logic - EXACT Freemius conditions
     * 
     * @return bool
     */
    private function is_pricing_item_visible() {
	    if (!$this->menu_item_is_enabled('pricing')) {
            return false;
        }
        
        // Don't show in demo mode or whitelabeled instances
        if ($this->is_demo_mode() || $this->fs->is_whitelabeled()) {
            return false;
        }
        
        // Must have at least one paid plan
        if (!$this->fs->has_paid_plan()) {
            return false;
        }
        
        // Show if not paying (trial or free plan) OR paying but has multiple plans
        if ($this->fs->is_not_paying() || ($this->fs->is_paying() && !$this->fs->is_single_plan(true))) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Contact item visibility logic - EXACT Freemius conditions
     * 
     * @return bool
     */
    private function is_contact_item_visible() {
	    if (!$this->menu_item_is_enabled('contact')) {
            return false;
        }
        
        // Don't show in demo mode or whitelabeled instances
        if ($this->is_demo_mode() || $this->fs->is_whitelabeled()) {
            return false;
        }
        
        // Show contact for ALL users - premium gets internal page, free gets external form
        return true;
    }
    
    /**
     * Support item visibility logic
     * 
     * @return bool
     */
    private function is_support_item_visible() {
	    if (!$this->menu_item_is_enabled('support')) {
            return false;
        }
        
        // Support typically requires paying status
        if ($this->fs->is_paying_or_trial()) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Affiliation item visibility logic - EXACT Freemius conditions
     * 
     * @return bool
     */
    private function is_affiliation_item_visible() {
        if (!$this->menu_item_is_enabled('affiliation')) {
            return false;
        }
        
        if ($this->fs->has_affiliate_program()) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Add-ons item visibility logic - EXACT Freemius conditions
     * 
     * @return bool
     */
    private function is_addons_item_visible() {
        if (!$this->menu_item_is_enabled('addons')) {
            return false;
        }
        
        if ($this->fs->has_addons()) {
            return true;
        }
        
        return false;
    }
    
    
    /**
     * Helper method to check if this is a free WordPress.org theme
     * 
     * @return bool
     */
    private function is_free_wordpress_org_theme() {
        return $this->fs->is_theme() && 
               $this->fs->is_org_repo_compliant() && 
               !$this->fs->is_premium();
    }

	private function menu_item_is_enabled($menu_item){
		return $this->get_bool_option($this->menu_config, $menu_item);
	}
	
	/**
     * Check if Freemius is in demo mode
     * 
     * @return bool
     */
    private function is_demo_mode() {
        return defined('WP_FS__DEMO_MODE') && WP_FS__DEMO_MODE;
    }
    
    /**
     * Check if Freemius instance is whitelabeled
     * 
     * @return bool
     */
    private function is_whitelabeled() {
        return $this->fs->is_whitelabeled();
    }
}