<?php

namespace WDevs\CustomerOrderExport;

use WDevs\CustomerOrderExport\ColumnCategories\CustomField;
use WDevs\CustomerOrderExport\ColumnCategories\GlobalProductAttribute;
use WDevs\CustomerOrderExport\ColumnCategories\Order;
use WDevs\CustomerOrderExport\ColumnCategories\OrderItem;
use WDevs\CustomerOrderExport\ColumnCategories\OrderItemmeta;
use WDevs\CustomerOrderExport\ColumnCategories\Product;
use WDevs\CustomerOrderExport\ColumnCategories\ProductTaxonomy;
use WDevs\CustomerOrderExport\ColumnCategories\SpecificProductAttribute;
/**
 * Handles columns for the Customer Order Export for WooCommerce plugin.
 *
 * This class provides methods to retrieve various columns that can be used
 * in the order items files.
 *
 * @since      1.0.0
 * @package    Wdevs_Customer_Order_Export
 * @subpackage Wdevs_Customer_Order_Export/includes
 * @author     Wijnberg Developments <contact@wijnberg.dev>
 */
class Column {
    public const CACHE_GROUP = 'wdevs-coe-columns';

    /**
     * Retrieves all allowed columns with their fields and values.
     *
     * @return array An associative array of columns with their fields and descriptions
     * @since 1.0.0
     */
    public static function get_allowed_columns() {
        $categories = [
            OrderItem::class,
            Product::class,
            GlobalProductAttribute::class,
            ProductTaxonomy::class,
            OrderItemmeta::class,
            Order::class
        ];
        $allowed_columns = [];
        $order_item_fields = OrderItem::get_fields();
        $product_fields = Product::get_fields();
        $global_product_attributes = GlobalProductAttribute::get_fields();
        //also add the 'pa_*' version for exclusion
        $keys = array_keys( $global_product_attributes );
        $product_attributes_keys = array_merge( $keys, array_map( 'wc_attribute_taxonomy_name', $keys ) );
        $specific_product_attributes = [];
        $product_taxonomies = ProductTaxonomy::get_fields( $product_attributes_keys );
        $order_itemmeta = OrderItemmeta::get_fields( $product_attributes_keys );
        $order_fields = Order::get_fields();
        $custom_fields = [];
        foreach ( $categories as $category ) {
            $allowed_columns[$category::$category_id] = [
                'name'   => $category::get_category_name(),
                'fields' => [],
            ];
            switch ( $category ) {
                case OrderItem::class:
                    $allowed_columns[$category::$category_id]['fields'] = $order_item_fields;
                    break;
                case Product::class:
                    $allowed_columns[$category::$category_id]['fields'] = $product_fields;
                    break;
                case GlobalProductAttribute::class:
                    $allowed_columns[$category::$category_id]['fields'] = $global_product_attributes;
                    break;
                case ProductTaxonomy::class:
                    $allowed_columns[$category::$category_id]['fields'] = $product_taxonomies;
                    break;
                case OrderItemmeta::class:
                    $allowed_columns[$category::$category_id]['fields'] = $order_itemmeta;
                    break;
                case Order::class:
                    $allowed_columns[$category::$category_id]['fields'] = $order_fields;
                    break;
            }
        }
        return apply_filters( 'wdevs_coe_get_allowed_columns', $allowed_columns );
    }

    /**
     * Flushes the cached specific product attributes keys
     *
     * @since 1.0.0
     */
    public function flush_specific_product_attributes_keys_cache() {
    }

    /**
     * Flushes the cached order item meta keys
     *
     * @since 1.0.0
     */
    public function flush_order_itemmeta_keys_cache() {
        delete_transient( OrderItemmeta::TRANSIENT_KEY_ORDER_ITEMMETA );
        wp_cache_delete( OrderItemmeta::CACHE_KEY_ORDER_ITEMMETA, self::CACHE_GROUP );
    }

    /**
     * Creates a ColumnCategory instance for the specified category
     *
     * @since 1.0.0
     */
    public static function create_column_category_instance( $order_item, $column_id, $column_data ) {
        if ( empty( $column_id ) || empty( $column_data['category_id'] ) ) {
            return null;
        }
        switch ( $column_data['category_id'] ) {
            case 'order_item_fields':
                return new OrderItem($order_item, $column_id, $column_data);
            case 'product_fields':
                return new Product($order_item, $column_id, $column_data);
            case 'global_product_attributes':
                return new GlobalProductAttribute($order_item, $column_id, $column_data);
            case 'product_taxonomies':
                return new ProductTaxonomy($order_item, $column_id, $column_data);
            case 'order_itemmeta':
                return new OrderItemmeta($order_item, $column_id, $column_data);
            case 'order_fields':
                return new Order($order_item, $column_id, $column_data);
            default:
                return null;
        }
    }

}
