<?php

namespace WDevs\CustomerOrderExport\Channels;


interface ChannelInterface {

	/**
	 * Get the channel name
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public static function get_name();

	/**
	 * Get the channel slug
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public static function get_slug();

	/**
	 * Returns the WooCommerce field settings for this channel
	 *
	 * @param array $templates Available template configurations
	 *
	 * @return array Configured settings array for WooCommerce fields
	 * @since 1.0.0
	 */
	public function get_settings( $templates );

	/**
	 * Checks if a file can be generated based on the channel settings
	 *
	 * @return bool True if file generation is allowed, false otherwise
	 * @since 1.0.0
	 */
	public function can_generate();

	/**
	 * Generates the output file for WooCommerce order data
	 *
	 * @param int|null $order_id The ID of the WooCommerce order to export.
	 *                          If null, either processes all orders or generates empty template
	 *                          based on implementation.
	 *
	 * @return string The full filesystem path to the generated output file
	 * @since 1.0.0
	 */
	public function generate( $order_id = null );

	/**
	 * Checks if the channel is enabled
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	public function is_enabled();

	/**
	 * Checks if the channel is connected
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	public function is_connected();

	/**
	 * Render form fields for account page editing
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function render_account_form_fields();

	/**
	 * Process form submission from account page
	 *
	 * @param array $post_data The POST data from the form
	 * @return array Array with 'success' boolean and 'message' string
	 * @since 1.0.0
	 */
	public function process_account_form( $post_data );

	/**
	 * Check if the channel should show a save button
	 *
	 * @return bool True to show save button, false to hide it
	 * @since 1.0.0
	 */
	public function show_save_button();

	/**
	 * Create a new folder in the channel's storage
	 *
	 * @param int $user_id The user ID creating the folder
	 * @param string $parent_id The parent folder ID
	 * @param string $folder_name The name for the new folder
	 * @return array Array with 'success' boolean and 'message' string, optionally 'folder_id'
	 * @since 1.0.0
	 */
	public function create_folder( $user_id, $parent_id, $folder_name );

	/**
	 * Rename an existing folder in the channel's storage
	 *
	 * @param int $user_id The user ID performing the rename
	 * @param string $folder_id The folder ID to rename
	 * @param string $new_name The new name for the folder
	 * @return array Array with 'success' boolean and 'message' string
	 * @since 1.0.0
	 */
	public function rename_folder( $user_id, $folder_id, $new_name );

	/**
	 * Delete a folder from the channel's storage
	 * 
	 * The folder must be empty (no files or subfolders) to be deleted.
	 * Root folder cannot be deleted.
	 *
	 * @param int $user_id The user ID performing the delete
	 * @param string $folder_id The folder ID to delete
	 * @return array Array with 'success' boolean and 'message' string
	 * @since 1.0.0
	 */
	public function delete_folder( $user_id, $folder_id );

}