<?php

namespace WDevs\CustomerOrderExport\Channels;

class ChannelFactory {
    /**
     * Available channel classes
     *
     * @var array
     * @since 1.0.0
     */
    private static $channel_classes = [
        GoogleDriveChannel::class,
        EmailChannel::class,
        OrderChannel::class,
        CustomerAccountChannel::class
    ];

    /**
     * Get channel slug to class mapping (dynamically generated)
     *
     * @return array
     * @since 1.0.0
     */
    private static function get_channel_map() {
        static $map = null;
        if ( $map === null ) {
            $map = [];
            foreach ( self::$channel_classes as $class ) {
                $map[$class::get_slug()] = $class;
            }
        }
        return $map;
    }

    /**
     * Creates a channel processor instance for the specified type
     *
     * @return ChannelProcessor
     * @since 1.0.0
     */
    public static function create( string $type = null ) {
        switch ( $type ) {
            case 'order':
                return new OrderChannel();
            default:
                return new EmailChannel();
        }
    }

    /**
     * Get a channel instance by slug
     *
     * @param string $slug The channel slug
     * @return ChannelInterface|null The channel instance or null if not found
     * @since 1.0.0
     */
    public static function get_channel_by_slug( $slug ) {
        $channel_map = self::get_channel_map();
        if ( !isset( $channel_map[$slug] ) ) {
            return null;
        }
        $class_name = $channel_map[$slug];
        if ( !class_exists( $class_name ) ) {
            return null;
        }
        return new $class_name();
    }

    /**
     * Get all available channels
     *
     * @return array Array of channel instances
     * @since 1.0.0
     */
    public static function get_all_channels() {
        $channels = [];
        foreach ( self::get_channel_map() as $slug => $class_name ) {
            $channel = self::get_channel_by_slug( $slug );
            if ( $channel ) {
                $channels[] = $channel;
            }
        }
        return $channels;
    }

    /**
     * Get all channel slugs
     *
     * @return array Array of channel slugs
     * @since 1.0.0
     */
    public static function get_channel_slugs() {
        return array_keys( self::get_channel_map() );
    }

}
