<?php

namespace WDevs\CustomerOrderExport\Generators;

use WDevs\CustomerOrderExport\Column;
use WDevs\CustomerOrderExport\Helper;
abstract class FileProcessor {
    use Helper;
    public $flattened_selected_columns = [];

    public function __construct() {
    }

    /**
     * Sets the flattened array of selected columns for export
     *
     * @param array $flattened_selected_columns Array of column configurations
     *
     * @since 1.0.0
     */
    public function set_flattened_selected_columns( $flattened_selected_columns ) {
        $this->flattened_selected_columns = $flattened_selected_columns;
    }

    /**
     * Gets available file formats for export
     *
     * @return array Associative array of file formats (key => display name)
     * @since 1.0.0
     */
    public static function get_file_formats() {
        $formats = [
            'csv' => __( 'Comma Separated Values (CSV)', 'orderbridge-for-woocommerce' ),
        ];
        return $formats;
    }

    /**
     * Gets available CSV delimiters
     *
     * @return array Associative array of delimiters (key => display name)
     * @since 1.0.0
     */
    public static function get_csv_delimiters() {
        return array(
            ',' => __( 'Comma (,)', 'orderbridge-for-woocommerce' ),
            ';' => __( 'Semicolon (;)', 'orderbridge-for-woocommerce' ),
        );
    }

    /**
     * Gets the default file name pattern for exports
     *
     * @return string Default filename with placeholder (e.g. "order_{{order_id}}")
     * @since 1.0.0
     */
    public static function get_default_file_name() {
        return sprintf( 
            /* translators: %s: Order ID placeholder (will be replaced like {{order_id}}) */
            __( 'order_%s', 'orderbridge-for-woocommerce' ),
            '{{order_id}}'
         );
    }

    /**
     * Gets headers for the export file
     *
     * @return array Array of column headers
     *
     * @since 1.0.0
     */
    protected function get_file_headers() {
        $headers = [];
        foreach ( $this->flattened_selected_columns as $key => $column ) {
            $headers[] = $column['name'];
        }
        return $headers;
    }

    /**
     * Gets data rows for the export file
     *
     * @param int|null $order_id The order ID to export, or null for all
     *
     * @return array Array of data rows (each row is an array of values)
     *
     * @since 1.0.0
     */
    protected function get_file_rows( $order_id = null ) {
        $rows = [];
        if ( !empty( $order_id ) && is_numeric( $order_id ) ) {
            $order = wc_get_order( $order_id );
            foreach ( $order->get_items( 'line_item' ) as $item_id => $item ) {
                $row = [];
                foreach ( $this->flattened_selected_columns as $key => $column ) {
                    $column_value = $this->get_column_value( $item, $key, $column );
                    if ( !isset( $column_value ) ) {
                        $column_value = '';
                    }
                    $row[] = $column_value;
                }
                $rows[] = $row;
            }
        }
        return $rows;
    }

    /**
     * Gets the value for a specific column
     *
     * @param \WC_Order_Item_Product $order_item The order item object
     * @param string $column_id The column identifier
     * @param array $column_data Column configuration data  Array ( [field] => '' [name] => '' [order] => '' [description] => '' [category] => '' )
     *
     * @return mixed|null The column value or null if not available
     *
     * @since 1.0.0
     */
    protected function get_column_value( $order_item, $column_id, $column_data ) {
        $value = null;
        $processor = Column::create_column_category_instance( $order_item, $column_id, $column_data );
        if ( $processor ) {
            $value = $processor->get_column_value();
        }
        return apply_filters(
            'wdevs_coe_get_column_value',
            $value,
            $order_item,
            $column_id,
            $column_data
        );
    }

    /**
     * Generates the full file path for export
     *
     * @param string $file_name The base file name
     * @param string $file_extension The file extension
     *
     * @return string Full temporary file path
     * @since 1.0.0
     */
    protected function get_file_path( $file_name, $file_extension ) {
        return sys_get_temp_dir() . '/' . $file_name . '.' . $file_extension;
    }

}
