<?php

namespace WDevs\CustomerOrderExport;

use WDevs\CustomerOrderExport\Admin\CustomerOrderExportAdmin;
use WDevs\CustomerOrderExport\Channels\CustomerAccountChannel;
use WDevs\CustomerOrderExport\Channels\EmailChannel;
use WDevs\CustomerOrderExport\Channels\GoogleDriveChannel;
use WDevs\CustomerOrderExport\Channels\OrderChannel;
use WDevs\CustomerOrderExport\Public\CustomerOrderExportPublic;
/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://wijnberg.dev
 * @since      1.0.0
 *
 * @package    Wdevs_Customer_Order_Export
 * @subpackage Wdevs_Customer_Order_Export/includes
 */
/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Wdevs_Customer_Order_Export
 * @subpackage Wdevs_Customer_Order_Export/includes
 * @author     Wijnberg Developments <contact@wijnberg.dev>
 */
class CustomerOrderExport {
    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Loader $loader Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string $plugin_name The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string $version The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        if ( defined( 'WDEVS_CUSTOMER_ORDER_EXPORT_VERSION' ) ) {
            $this->version = WDEVS_CUSTOMER_ORDER_EXPORT_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'wdevs-customer-order-export';
        $this->load_dependencies();
        $this->define_ajax_hooks();
        $this->define_admin_hooks();
        $this->define_woocommerce_hooks();
        $this->define_channel_hooks();
        $this->define_public_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies() {
        $this->loader = new Loader();
    }

    /**
     * Register all of the hooks related to AJAX requests functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_ajax_hooks() {
        if ( is_admin() ) {
            $plugin_ajax = new Ajax($this->get_plugin_name(), $this->get_version());
            //AJAX Requests
            $this->loader->add_action( 'wp_ajax_' . Ajax::AJAX_ACTION_TEMPLATE_DELETE, $plugin_ajax, Ajax::AJAX_ACTION_TEMPLATE_DELETE . '_action' );
            $this->loader->add_action( 'wp_ajax_' . Ajax::AJAX_ACTION_TEMPLATE_COLUMNS, $plugin_ajax, Ajax::AJAX_ACTION_TEMPLATE_COLUMNS . '_action' );
            $this->loader->add_action( 'wp_ajax_' . Ajax::AJAX_ACTION_FOLDER_MANAGEMENT, $plugin_ajax, Ajax::AJAX_ACTION_FOLDER_MANAGEMENT . '_action' );
            //Post Requests
            $this->loader->add_action( 'admin_post_' . Ajax::POST_ACTION_PREVIEW_FILE, $plugin_ajax, Ajax::POST_ACTION_PREVIEW_FILE . '_action' );
        }
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks() {
        if ( is_admin() ) {
            $plugin_admin = new CustomerOrderExportAdmin($this->get_plugin_name(), $this->get_version());
            $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
            $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
            $this->loader->add_filter( 'plugin_action_links_' . plugin_basename( dirname( __DIR__ ) . '/' . $this->plugin_name . '.php' ), $plugin_admin, 'add_action_links' );
        }
    }

    /**
     * Register all of the hooks related to the WooCommerce functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_woocommerce_hooks() {
        if ( is_admin() ) {
            $plugin_woocommerce = new Woocommerce($this->get_plugin_name(), $this->get_version());
            $this->loader->add_filter( 'before_woocommerce_init', $plugin_woocommerce, 'declare_compatibility' );
            $this->loader->add_filter(
                'woocommerce_settings_tabs_array',
                $plugin_woocommerce,
                'add_settings_tab',
                50
            );
            $this->loader->add_action( 'woocommerce_settings_tabs_wdevs_coe', $plugin_woocommerce, 'settings_tab' );
            $this->loader->add_action( 'woocommerce_after_settings_wdevs_coe', $plugin_woocommerce, 'render_footer_info' );
            $this->loader->add_filter(
                'pre_update_option_wdevs_coe_email_file_name',
                $plugin_woocommerce,
                'sanitize_file_name',
                10,
                3
            );
            $this->loader->add_filter(
                'pre_update_option_wdevs_coe_customer_account_file_name',
                $plugin_woocommerce,
                'sanitize_file_name',
                10,
                3
            );
            $this->loader->add_filter(
                'pre_update_option_wdevs_coe_order_edit_file_name',
                $plugin_woocommerce,
                'sanitize_file_name',
                10,
                3
            );
            $plugin_columns = new Column();
            $this->loader->add_action( 'woocommerce_new_order_item', $plugin_columns, 'flush_order_itemmeta_keys_cache' );
            $this->loader->add_action( 'woocommerce_update_order_item', $plugin_columns, 'flush_order_itemmeta_keys_cache' );
        }
    }

    private function define_channel_hooks() {
        $email_channel = new EmailChannel();
        $this->loader->add_action(
            'woocommerce_email_attachments',
            $email_channel,
            'attach_export_to_email',
            10,
            4
        );
        if ( is_admin() ) {
            $order_channel = new OrderChannel();
            $this->loader->add_filter(
                'woocommerce_order_actions',
                $order_channel,
                OrderChannel::ORDER_ACTION_DOWNLOAD . '_add',
                10,
                2
            );
            $this->loader->add_action(
                'woocommerce_order_action_' . OrderChannel::ORDER_ACTION_DOWNLOAD,
                $order_channel,
                OrderChannel::ORDER_ACTION_DOWNLOAD . '_action',
                10,
                1
            );
        } else {
        }
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks() {
        //For the endpoints to work, this also needs to be executed in the admin area if ( ! is_admin() ) {
        $plugin_public = new CustomerOrderExportPublic($this->get_plugin_name(), $this->get_version());
        // Register the customer account endpoint and menu via WooCommerce
        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
        $this->loader->add_filter( 'woocommerce_get_query_vars', $plugin_public, 'register_account_endpoint' );
        $this->loader->add_filter( 'woocommerce_account_menu_items', $plugin_public, 'add_account_menu_item' );
        $this->loader->add_action( 'woocommerce_account_' . CustomerOrderExportPublic::ACCOUNT_ENDPOINT . '_endpoint', $plugin_public, 'render_account_page' );
        $this->loader->add_filter( 'woocommerce_endpoint_' . CustomerOrderExportPublic::ACCOUNT_ENDPOINT . '_title', $plugin_public, 'set_account_page_title' );
        $this->loader->add_action( 'template_redirect', $plugin_public, 'process_channel_form' );
        //}
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @return    string    The name of the plugin.
     * @since     1.0.0
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @return    Loader    Orchestrates the hooks of the plugin.
     * @since     1.0.0
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @return    string    The version number of the plugin.
     * @since     1.0.0
     */
    public function get_version() {
        return $this->version;
    }

}
