<?php


namespace WDevs\CustomerOrderExport\ColumnCategories;


class Product extends ColumnCategoryProcessor implements ColumnCategoryInterface {

	public static $category_id = 'product_fields';

	public function __construct( $order_item, $column_id, $column_data ) {
		parent::__construct( $order_item, $column_id, $column_data );
	}

	/**
	 * @param array $excluded_keys
	 *
	 * @return array $fields The fields that can be selected as column
	 * @since 1.0.0
	 */
	public static function get_fields( $excluded_keys = [] ) {
		return [
			'name'              => [
				'field'       => __( '(Parent) product name', 'orderbridge-for-woocommerce' ),
				'name'        => __( '(Parent) product name', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'The full name/title of the product', 'orderbridge-for-woocommerce' ),
			],
			'status'            => [
				'field'       => __( 'Status', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Status', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'The publication status of the product (e.g., publish, draft)', 'orderbridge-for-woocommerce' ),
			],
			'description'       => [
				'field'       => __( 'Description', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Description', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'The full product description', 'orderbridge-for-woocommerce' ),
			],
			'short_description' => [
				'field'       => __( 'Short description', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Short description', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'The brief product description', 'orderbridge-for-woocommerce' ),
			],
			'sku'               => [
				'field'       => __( 'SKU', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'SKU', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'Stock Keeping Unit - product identifier', 'orderbridge-for-woocommerce' ),
			],
			'global_unique_id'  => [
				'field'       => __( 'Global Trade Item Number', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Global Trade Item Number', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'Unique global identifier for the product', 'orderbridge-for-woocommerce' ),
			],
			'price'             => [
				'field'       => __( 'Price', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Price', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'Current product price (including sales)', 'orderbridge-for-woocommerce' ),
			],
			'regular_price'     => [
				'field'       => __( 'Regular price', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Regular price', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'Standard price without discounts', 'orderbridge-for-woocommerce' ),
			],
			'sale_price'        => [
				'field'       => __( 'Sale price', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Sale price', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'Discounted price during sales', 'orderbridge-for-woocommerce' ),
			],
			'weight'            => [
				'field'       => __( 'Weight', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Weight', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'Product weight with unit', 'orderbridge-for-woocommerce' ),
			],
			'length'            => [
				'field'       => __( 'Length', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Length', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'Product length with unit', 'orderbridge-for-woocommerce' ),
			],
			'width'             => [
				'field'       => __( 'Width', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Width', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'Product width with unit', 'orderbridge-for-woocommerce' ),
			],
			'height'            => [
				'field'       => __( 'Height', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Height', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'Product height with unit', 'orderbridge-for-woocommerce' ),
			],
			'virtual'           => [
				'field'       => __( 'Virtual', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Virtual', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'Whether this is a non-physical product', 'orderbridge-for-woocommerce' ),
			],
			'downloadable'      => [
				'field'       => __( 'Downloadable', 'orderbridge-for-woocommerce' ),
				'name'        => __( 'Downloadable', 'orderbridge-for-woocommerce' ),
				'category'    => self::$category_id,
				'description' => __( 'Whether this product has downloadable files', 'orderbridge-for-woocommerce' ),
			],
		];
	}

	/**
	 * @return string $name returns the name of the category
	 * @since 1.0.0
	 */
	public static function get_category_name() {
		return __( 'Product fields', 'orderbridge-for-woocommerce' );
	}

	/**
	 * Returns the value for the current column
	 *
	 * @return null|string
	 * @since 1.0.0
	 */
	public function get_column_value() {
		if ( ! $this->order_item_is_set() ) {
			return null;
		}

		return $this->get_product_field_value( $this->order_item, $this->column_id );
	}

	/**
	 * Retrieves product field values for order export
	 *
	 * @param \WC_Order_Item_Product $order_item The WooCommerce order item object
	 * @param string $column_id The product field identifier to retrieve
	 *
	 * @return mixed|null The requested product value or null if not available
	 *                    - Returns string for text fields (name, description, etc.)
	 *                    - Returns formatted string for boolean fields (virtual, downloadable)
	 *                    - Returns numeric values for prices and dimensions
	 *                    - Falls back to parent product values for variations when applicable
	 * @since 1.0.0
	 */
	private function get_product_field_value( $order_item, $column_id ) {
		$product = $order_item->get_product();
		if ( ! $product ) {
			return null;
		}

		switch ( $column_id ) {
			case 'name':
				$name = $product->get_name();
				if ( ! empty( $name ) ) {
					return $name;
				}
				if ( $product->is_type( 'variation' ) ) {
					$parent_product = wc_get_product( $product->get_parent_id() );

					return $parent_product ? $parent_product->get_name() : null;
				}

				return null;
			case 'status':
				return $product->get_status();
			case 'description':
				return $product->get_description();
			case 'short_description':
				return $product->get_short_description();
			case 'sku':
				$sku = $product->get_sku();
				if ( ! empty( $sku ) ) {
					return $sku;
				}
				if ( $product->is_type( 'variation' ) ) {
					$parent_product = wc_get_product( $product->get_parent_id() );

					return $parent_product ? $parent_product->get_sku() : null;
				}

				return null;
			case 'global_unique_id':
				if ( method_exists( $product, 'get_global_unique_id' ) ) {
					return $product->get_global_unique_id();
				}

				return null;
			case 'price':
				return $product->get_price();
			case 'regular_price':
				return $product->get_regular_price();
			case 'sale_price':
				return $product->get_sale_price();
			case 'weight':
				return $product->get_weight();
			case 'length':
				return $product->get_length();
			case 'width':
				return $product->get_width();
			case 'height':
				return $product->get_height();
			case 'virtual':
				return $product->get_virtual() ? __( 'Yes', 'orderbridge-for-woocommerce' ) : __( 'No', 'orderbridge-for-woocommerce' );
			case 'downloadable':
				return $product->get_downloadable() ? __( 'Yes', 'orderbridge-for-woocommerce' ) : __( 'No', 'orderbridge-for-woocommerce' );
			default:
				return null;
		}
	}
}